import encoding.hex
import x.crypto.chacha20
import x.crypto.poly1305
import x.crypto.chacha20poly1305

const key_size = chacha20poly1305.key_size
const tag_size = chacha20poly1305.tag_size

// Example and Test Vector for AEAD_CHACHA20_POLY1305
// https://datatracker.ietf.org/doc/html/rfc8439#section-2.8.2
fn test_encrypt_example_test() ! {
	plaintext := "Ladies and Gentlemen of the class of '99: If I could offer you only one tip for the future, sunscreen would be it.".bytes()
	aad := hex.decode('50515253c0c1c2c3c4c5c6c7')!
	key := hex.decode('808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9f')!

	iv := '4041424344454647'
	fixed := '07000000'
	nonce := hex.decode(fixed + iv)!

	exp_onetime_key := hex.decode('7bac2b252db447af09b67a55a4e955840ae1d6731075d9eb2a9375783ed553ff')!

	ncsize := nonce.len
	mut c := chacha20poly1305.new(key, ncsize)!
	mut cs := chacha20.new_cipher(key, nonce)!

	// and then performing Chacha20 block function
	mut polykey := []u8{len: key_size}
	cs.xor_key_stream(mut polykey, polykey)

	assert polykey == exp_onetime_key

	exp_ciphertext := hex.decode('d31a8d34648e60db7b86afbc53ef7ec2a4aded51296e08fea9e2b5a736ee62d63dbea45e8ca9671282fafb69da92728b1a71de0a9e060b2905d6a5b67ecd3b3692ddbd7f2d778b8c9803aee328091b58fab324e4fad675945585808b4831d7bc3ff4def08e4b7a9de576d26586cec64b6116')!
	expected_tag := hex.decode('1ae10b594f09e26a7e902ecbd0600691')!

	cipherout := c.encrypt(plaintext, nonce, aad)!
	// cipherout is concatenation of ciphertext and overhead (tag) bytes
	ciphertext := cipherout[0..cipherout.len - tag_size]
	mac := cipherout[cipherout.len - tag_size..]

	assert ciphertext == exp_ciphertext
	assert mac == expected_tag
}

// A.5.  ChaCha20-Poly1305 AEAD Decryption test
fn test_aead_decrypt_vector_test_51() ! {
	key := hex.decode('1c9240a5eb55d38af333888604f6b5f0473917c1402b80099dca5cbc207075c0')!
	aad := hex.decode('f33388860000000000004e91')!
	nonce := hex.decode('000000000102030405060708')!
	ciphertext := hex.decode('64a0861575861af460f062c79be643bd5e805cfd345cf389f108670ac76c8cb24c6cfc18755d43eea09ee94e382d26b0bdb7b73c321b0100d4f03b7f355894cf332f830e710b97ce98c8a84abd0b948114ad176e008d33bd60f982b1ff37c8559797a06ef4f0ef61c186324e2b3506383606907b6a7c02b0f9f6157b53c867e4b9166c767b804d46a59b5216cde7a4e99040c5a40433225ee282a1b0a06c523eaf4534d7f83fa1155b0047718cbc546a0d072b04b3564eea1b422273f548271a0bb2316053fa76991955ebd63159434ecebb4e466dae5a1073a6727627097a1049e617d91d361094fa68f0ff77987130305beaba2eda04df997b714d6c6f2c29a6ad5cb4022b02709b')!
	// Received Tag
	expected_tag := hex.decode('eead9d67890cbb22392336fea1851f38')!
	// Ciphertext msg is concatenation of ciphertext and tag
	mut ciphertext_msg := []u8{}
	ciphertext_msg << ciphertext
	ciphertext_msg << expected_tag

	ncsize := nonce.len
	mut c := chacha20poly1305.new(key, ncsize)!
	mut cs := chacha20.new_cipher(key, nonce)!

	// and then performing Chacha20 block function
	mut polykey := []u8{len: key_size}
	cs.xor_key_stream(mut polykey, polykey)

	expected_otk := hex.decode('bdf04aa95ce4de8995b14bb6a18fecaf26478f50c054f563dbc0a21e261572aa')!
	assert polykey == expected_otk

	constructed_msg := hex.decode('f33388860000000000004e910000000064a0861575861af460f062c79be643bd5e805cfd345cf389f108670ac76c8cb24c6cfc18755d43eea09ee94e382d26b0bdb7b73c321b0100d4f03b7f355894cf332f830e710b97ce98c8a84abd0b948114ad176e008d33bd60f982b1ff37c8559797a06ef4f0ef61c186324e2b3506383606907b6a7c02b0f9f6157b53c867e4b9166c767b804d46a59b5216cde7a4e99040c5a40433225ee282a1b0a06c523eaf4534d7f83fa1155b0047718cbc546a0d072b04b3564eea1b422273f548271a0bb2316053fa76991955ebd63159434ecebb4e466dae5a1073a6727627097a1049e617d91d361094fa68f0ff77987130305beaba2eda04df997b714d6c6f2c29a6ad5cb4022b02709b000000000000000c000000000000000901000000000000')!

	mut mac := []u8{len: tag_size}
	poly1305.create_tag(mut mac, constructed_msg, polykey)!
	assert mac == expected_tag

	expected_plaintext := hex.decode('496e7465726e65742d4472616674732061726520647261667420646f63756d656e74732076616c696420666f722061206d6178696d756d206f6620736978206d6f6e74687320616e64206d617920626520757064617465642c207265706c616365642c206f72206f62736f6c65746564206279206f7468657220646f63756d656e747320617420616e792074696d652e20497420697320696e617070726f70726961746520746f2075736520496e7465726e65742d447261667473206173207265666572656e6365206d6174657269616c206f7220746f2063697465207468656d206f74686572207468616e206173202fe2809c776f726b20696e2070726f67726573732e2fe2809d')!
	plaintext := c.decrypt(ciphertext_msg, nonce, aad)!
	assert plaintext == expected_plaintext
}

fn test_decrypt_and_verify_tag() ! {
	for i, c in aead_cases {
		key := hex.decode(c.key)!
		aad := hex.decode(c.aad)!
		nonce := hex.decode(c.nonce)!
		msg := hex.decode(c.msg)!
		exp_out := hex.decode(c.out)!
		exp_tag := hex.decode(c.mac)!

		ncsize := nonce.len
		mut co := chacha20poly1305.new(key, ncsize)!
		ciphertext := co.encrypt(msg, nonce, aad)!

		text0 := ciphertext[0..ciphertext.len - co.overhead()]
		mac0 := ciphertext[ciphertext.len - co.overhead()..]
		assert text0 == exp_out
		assert mac0 == exp_tag

		plaintext := co.decrypt(ciphertext, nonce, aad)!

		assert plaintext == msg
	}
}

struct AeadCase {
	key   string
	aad   string
	nonce string
	msg   string
	out   string
	mac   string
}

// from libressl vector test at https://fossies.org/linux/libressl/tests/aeadtests.txt
const aead_cases = [
	AeadCase{
		key:   '808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9f'
		aad:   '50515253c0c1c2c3c4c5c6c7'
		nonce: '404142434445464748494a4b4c4d4e4f5051525354555657'
		msg:   '4c616469657320616e642047656e746c656d656e206f662074686520636c617373206f66202739393a204966204920636f756c64206f6666657220796f75206f6e6c79206f6e652074697020666f7220746865206675747572652c2073756e73637265656e20776f756c642062652069742e'
		out:   'bd6d179d3e83d43b9576579493c0e939572a1700252bfaccbed2902c21396cbb731c7f1b0b4aa6440bf3a82f4eda7e39ae64c6708c54c216cb96b72e1213b4522f8c9ba40db5d945b11b69b982c1bb9e3f3fac2bc369488f76b2383565d3fff921f9664c97637da9768812f615c68b13b52e'
		mac:   'c0875924c1c7987947deafd8780acf49'
	},
	AeadCase{
		key:   '808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9f'
		aad:   'c0c1c2c3c4c5c6c7d0d1d2d3d4d5d6d72e202500000000090000004529000029'
		nonce: 'a0a1a2a31011121314151617'
		msg:   '0000000c000040010000000a00'
		out:   '610394701f8d017f7c12924889'
		mac:   '6b71bfe25236efd7cdc67066906315b2'
	},
	AeadCase{
		// Test vector from RFC7539 2.8.2
		// AEAD: chacha20-poly1305
		key:   '808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9f'
		nonce: '070000004041424344454647'
		msg:   '4c616469657320616e642047656e746c656d656e206f662074686520636c617373206f66202739393a204966204920636f756c64206f6666657220796f75206f6e6c79206f6e652074697020666f7220746865206675747572652c2073756e73637265656e20776f756c642062652069742e'
		aad:   '50515253c0c1c2c3c4c5c6c7'
		out:   'd31a8d34648e60db7b86afbc53ef7ec2a4aded51296e08fea9e2b5a736ee62d63dbea45e8ca9671282fafb69da92728b1a71de0a9e060b2905d6a5b67ecd3b3692ddbd7f2d778b8c9803aee328091b58fab324e4fad675945585808b4831d7bc3ff4def08e4b7a9de576d26586cec64b6116'
		mac:   '1ae10b594f09e26a7e902ecbd0600691'
	},
	AeadCase{
		//# Test vector from RFC7539 Appendix A.5
		// AEAD: chacha20-poly1305
		key:   '1c9240a5eb55d38af333888604f6b5f0473917c1402b80099dca5cbc207075c0'
		nonce: '000000000102030405060708'
		msg:   '496e7465726e65742d4472616674732061726520647261667420646f63756d656e74732076616c696420666f722061206d6178696d756d206f6620736978206d6f6e74687320616e64206d617920626520757064617465642c207265706c616365642c206f72206f62736f6c65746564206279206f7468657220646f63756d656e747320617420616e792074696d652e20497420697320696e617070726f70726961746520746f2075736520496e7465726e65742d447261667473206173207265666572656e6365206d6174657269616c206f7220746f2063697465207468656d206f74686572207468616e206173202fe2809c776f726b20696e2070726f67726573732e2fe2809d'
		aad:   'f33388860000000000004e91'
		out:   '64a0861575861af460f062c79be643bd5e805cfd345cf389f108670ac76c8cb24c6cfc18755d43eea09ee94e382d26b0bdb7b73c321b0100d4f03b7f355894cf332f830e710b97ce98c8a84abd0b948114ad176e008d33bd60f982b1ff37c8559797a06ef4f0ef61c186324e2b3506383606907b6a7c02b0f9f6157b53c867e4b9166c767b804d46a59b5216cde7a4e99040c5a40433225ee282a1b0a06c523eaf4534d7f83fa1155b0047718cbc546a0d072b04b3564eea1b422273f548271a0bb2316053fa76991955ebd63159434ecebb4e466dae5a1073a6727627097a1049e617d91d361094fa68f0ff77987130305beaba2eda04df997b714d6c6f2c29a6ad5cb4022b02709b'
		mac:   'eead9d67890cbb22392336fea1851f38'
	},
	AeadCase{
		//# Test vector from RFC7634 Appendix A
		// AEAD: chacha20-poly1305
		key:   '808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9f'
		nonce: 'a0a1a2a31011121314151617'
		msg:   '45000054a6f200004001e778c6336405c000020508005b7a3a080000553bec100007362708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363701020204'
		aad:   '0102030400000005'
		out:   '24039428b97f417e3c13753a4f05087b67c352e6a7fab1b982d466ef407ae5c614ee8099d52844eb61aa95dfab4c02f72aa71e7c4c4f64c9befe2facc638e8f3cbec163fac469b502773f6fb94e664da9165b82829f641e0'
		mac:   '76aaa8266b7fb0f7b11b369907e1ad43'
	},
]

struct ChapolyTest {
	plaintext string
	aad       string
	key       string
	nonce     string
	out       string
}

// this test data come from golang vector tests of the same module
fn test_cha20poly1305_vector_data() ! {
	for i, t in chapoly_testcases {
		plaintext := hex.decode(t.plaintext)!
		aad := hex.decode(t.aad)!
		key := hex.decode(t.key)!
		nonce := hex.decode(t.nonce)!

		// This output from golang was concatenation of ciphertext+tag
		out := hex.decode(t.out)!

		// split out
		out_ciphertext := out[..plaintext.len]
		out_tag := out[plaintext.len..]

		ncsize := nonce.len
		mut co := chacha20poly1305.new(key, ncsize)!

		ciphertext := co.encrypt(plaintext, nonce, aad)!
		assert ciphertext[0..ciphertext.len - co.overhead()] == out_ciphertext
		assert ciphertext[ciphertext.len - co.overhead()..] == out_tag

		assert ciphertext == out
		message := co.decrypt(ciphertext, nonce, aad)!

		assert message == plaintext
	}
}

const chapoly_testcases = [
	ChapolyTest{'', '', '808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9f', '070000004041424344454647', 'a0784d7a4716f3feb4f64e7f4b39bf04'},
	ChapolyTest{'4c616469657320616e642047656e746c656d656e206f662074686520636c617373206f66202739393a204966204920636f756c64206f6666657220796f75206f6e6c79206f6e652074697020666f7220746865206675747572652c2073756e73637265656e20776f756c642062652069742e', '50515253c0c1c2c3c4c5c6c7', '808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9f', '404142434445464748494a4b4c4d4e4f5051525354555657', 'bd6d179d3e83d43b9576579493c0e939572a1700252bfaccbed2902c21396cbb731c7f1b0b4aa6440bf3a82f4eda7e39ae64c6708c54c216cb96b72e1213b4522f8c9ba40db5d945b11b69b982c1bb9e3f3fac2bc369488f76b2383565d3fff921f9664c97637da9768812f615c68b13b52ec0875924c1c7987947deafd8780acf49'},
	ChapolyTest{'1400000cebccee3bf561b292340fec60', '00000000000000001603030010', 'a5117e70953568bf750862df9e6f92af81677c3a188e847917a4a915bda7792e', '129039b5572e8a7a8131f76a', '2b487a2941bc07f3cc76d1a531662588ee7c2598e59778c24d5b27559a80d163'},
	ChapolyTest{'0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '00000000000000000000000000', 'a5117e70953568bf750862df9e6f92af81677c3a188e847917a4a915bda7792e', '129039b5572e8a7a8131f76a', '3f487a25aa70e9c8391763370569c9e83b7650dd1921c8b78869f241f25d2096c910b180930c5b8747fd90959fe8ca2dcadb4fa50fa1439f916b2301e1cc0810d6725775d3ab86721700f96e22709b0a7a8bef32627dd929b2dd3ba15772b669062bb558bc92e6c241a1d60d9f0035e80c335f854815fe1138ab8af653eab3e122135feeec7dfaba1cc24af82a2b7acccdd824899a7e03cc29c25be8a4f56a66673845b93bae1556f09dafc89a0d22af207718e2a6bb022e9d917597295992ea3b750cc0e7a7c3d33b23c5a8aeab45f5bb542f6c9e6c1747ae5a344aff483ba38577ad534b33b3abc7d284776ea33ed488c2a2475648a4fcda561745ea7787ed60f2368deb27c75adce6ff9b6cc6de1f5e72a741e2d59f64751b3ae482d714e0c90e83c671ff98ed611823afb39e6e5019a6ba548a2a72e829c7b7b4a101ac9deb90a25d3e0c50d22e1fc26c7c02296fa13c6d9c14767f68aaf46450a8d0fd5feb60d9d73c6e68623425b4984a79d619dd6bf896459aa77a681ec9c1a97f645e121f47779b051f8948a817f84d1f55da170d5bbbaf2f64e18b97ed3fd822db2819f523314f1e5ac72e8f69bbe6c87c22daddb0e1ac6790f8534071de2f258064b99789bfb165b065b8fe96f9127cd7dca9f7cb0368420f1e802faa3ca23792f2a5b93773dd405e71c320b211b54f7a26626b03c060e1ab87f32ac588abfa056ce090bd7c69913a700c80f325bfe824fa'},
	ChapolyTest{'0967de57eefe1aaa999b9b746d88a1a248000d8734e0e938c6aa87', 'e4f0a3a4f90a8250f8806aa319053e8d73c62f150e2f239563037e9cc92823ad18c65111d0d462c954cc6c6ed2aafb45702a5a7e597d13bd8091594ab97cf7d1', 'f2db28620582e05f00f31c808475ca3df1c20e340bf14828352499466d79295f', '4349e2131d44dc711148dfe3', 'bd06cc144fdc0d8b735fa4452eabbf78fd4ad2966ea41a84f68da40ca2da439777bc2ba6c4ec2de0d003eb'},
	ChapolyTest{'135a28170fe89066da7bcff3a9ccc1b27dfe942a6f47b23835ef746aaea63dc10066d90f4e697528e5451b8e11dd408fdbd4b94a1c6c82515bf7bc099df9cb9d5fa4acad0d22d5f267f18078cec107a995c1f3b12d7603886dbf910ab85ca7180053c50e759b00dc8c81555a425c03d71df6894a6c8cd2d94b64e303c08a1bc1dee1cf537ccf300850856292e1656aff5bf349c87f1ca1ca8085cd400fe901edcad04146a0714ef0f6b083d715edd670e020385f3cda29bc5ff6fc6edffe5ca9ce9def6e0e3d5f04ede2db02cfb2', '73afd2ab0e0e8537cae42dc6530dc4afb6934ca6', 'a5117e70953568bf750862df9e6f92af81677c3a188e847917a4a915bda7792e', '129039b5572e8a7a8131f76a', '2c125232a59879aee36cacc4aca5085a4688c4f776667a8fbd86862b5cfb1d57c976688fdd652eafa2b88b1b8e358aa2110ff6ef13cdc1ceca9c9f087c35c38d89d6fbd8de89538070f17916ecb19ca3ef4a1c834f0bdaa1df62aaabef2e117106787056c909e61ecd208357dd5c363f11c5d6cf24992cc873cf69f59360a820fcf290bd90b2cab24c47286acb4e1033962b6d41e562a206a94796a8ab1c6b8bade804ff9bdf5ba6062d2c1f8fe0f4dfc05720bd9a612b92c26789f9f6a7ce43f5e8e3aee99a9cd7d6c11eaa611983c36935b0dda57d898a60a0ab7c4b54'},
	// some XChaCha20-Poly1305 vectors
	ChapolyTest{'000000000000000000000000000000', '', '0000000000000000000000000000000000000000000000000000000000000000', '000000000000000000000000000000000000000000000000', '789e9689e5208d7fd9e1f3c5b5341fb2f7033812ac9ebd3745e2c99c7bbfeb'},
	ChapolyTest{'02dc819b71875e49f5e1e5a768141cfd3f14307ae61a34d81decd9a3367c00c7', '', 'b7bbfe61b8041658ddc95d5cbdc01bbe7626d24f3a043b70ddee87541234cff7', 'e293239d4c0a07840c5f83cb515be7fd59c333933027e99c', '7a51f271bd2e547943c7be3316c05519a5d16803712289aa2369950b1504dd8267222e47b13280077ecada7b8795d535'},
	ChapolyTest{'7afc5f3f24155002e17dc176a8f1f3a097ff5a991b02ff4640f70b90db0c15c328b696d6998ea7988edfe3b960e47824e4ae002fbe589be57896a9b7bf5578599c6ba0153c7c', 'd499bb9758debe59a93783c61974b7', '4ea8fab44a07f7ffc0329b2c2f8f994efdb6d505aec32113ae324def5d929ba1', '404d5086271c58bf27b0352a205d21ce4367d7b6a7628961', '26d2b46ad58b6988e2dcf1d09ba8ab6f532dc7e0847cdbc0ed00284225c02bbdb278ee8381ebd127a06926107d1b731cfb1521b267168926492e8f77219ad922257a5be2c5e52e6183ca4dfd0ad3912d7bd1ec968065'},
	ChapolyTest{'', '', '48d8bd02c2e9947eae58327114d35e055407b5519c8019535efcb4fc875b5e2b', 'cc0a587a475caba06f8dbc09afec1462af081fe1908c2cba', 'fc3322d0a9d6fac3eb4a9e09b00b361e'},
	ChapolyTest{'e0862731e5', '', '6579e7ee96151131a1fcd06fe0d52802c0021f214960ecceec14b2b8591f62cd', 'e2230748649bc22e2b71e46a7814ecabe3a7005e949bd491', 'e991efb85d8b1cfa3f92cb72b8d3c882e88f4529d9'},
	ChapolyTest{'00c7dd8f440af1530b44', '', 'ffb733657c849d50ab4ab40c4ae18f8ee2f0acf7c907afefdc04dff3537fdff3', '02c6fd8032a8d89edbedcd1db024c09d29f08b1e74325085', '13dbcdb8c60c3ed28449a57688edfaea89e309ab4faa6d51e532'},
	ChapolyTest{'f7e11b4d372ed7cb0c0e157f2f9488d8efea0f9bbe089a345f51bdc77e30d1392813c5d22ca7e2c7dfc2e2d0da67efb2a559058d4de7a11bd2a2915e', '', '194b1190fa31d483c222ec475d2d6117710dd1ac19a6f1a1e8e894885b7fa631', '6b07ea26bb1f2d92e04207b447f2fd1dd2086b442a7b6852', '25ae14585790d71d39a6e88632228a70b1f6a041839dc89a74701c06bfa7c4de3288b7772cb2919818d95777ab58fe5480d6e49958f5d2481431014a8f88dab8f7e08d2a9aebbe691430011d'},
	ChapolyTest{'', '1e2b11e3', '70cd96817da85ede0efdf03a358103a84561b25453dee73735e5fb0161b0d493', '5ddeba49f7266d11827a43931d1c300dd47a3c33f9f8bf9b', '592fc4c19f3cddec517b2a00f9df9665'},
	ChapolyTest{'81b3cb7eb3', 'efcfd0cf', 'a977412f889281a6d75c24186f1bfaa00dcc5132f0929f20ef15bbf9e63c4c91', '3f26ca997fb9166d9c615babe3e543ca43ab7cab20634ac5', '8e4ade3e254cf52e93eace5c46667f150832725594'},
	ChapolyTest{'f955183b1f762d4536d3f6885ea7f5ac27414caf46c2e24a2fd3bd56b91c53d840fb657224565e0a6f686f8ba320e04a401057399d9a3d995ab17c13', 'c372ddc5', 'a188be3795b2ca2e69b6aa263244f0963c492d694cf6c9b705a1d7045f3f2a26', '51bb484ea094ee140474681e1c838e4442fd148de2cc345a', '48759a5ddfdd829d11de8e0c538ce4a9c475faab6912039b568ad92d737d172fc1eb0c00c3793de6dddbfacfdbbc7f44aeba33684e18005aa982b6fc6c556e63bb90ff7a1dde8153a63eabe0'},
	ChapolyTest{'', 'e013cd0bfafd486d', 'af3d3ba094d38299ecb91c17bfe3d085da5bd42e11acf8acb5bc26a4be9a7583', '7dd63c14173831f109761b1c1abe18f6ba937d825957011b', '8bc685a7d9d501952295cd25d8c92517'},
	ChapolyTest{'284b64597e', '31d013e53aa3ea79', '93c77409d7f805f97fe683b2dd6ee06152a5e918b3eed5b731acccffdcb2cc04', '3d331e90c4597cf0c30d1b7cfbd07bcb6ab927eda056873c', '3538a449d6c18d148a8c6cb76f1bc288657ac7036a'},
	ChapolyTest{'acd08d4938a224b4cb2d723bf75420f3ea27b698fadd815bb7db9548a05651398644354334e69f8e4e5503bf1a6f92b38e860044a7edca6874038ce1', '28a137808d0225b8', 'a031203b963a395b08be55844d81af39d19b23b7cc24b21afa31edc1eea6edd6', 'e8b31c52b6690f10f4ae62ba9d50ba39fb5edcfb78400e35', '35cf39ba31da95ac9b661cdbd5e9c9655d13b8ff065c4ec10c810833a47a87d8057dd1948a7801bfe6904b49fed0aabfb3cd755a1a262d372786908ddcf64cae9f71cb9ed199c3ddacc50116'},
	ChapolyTest{'', 'cda7ee2857e09e9054ef6806', 'd91dffb18132d8dd3d144a2f10ba28bc5df36cb60369f3b19893ec91db3cf904', 'ee56f19c62b0438da6a0d9e01844313902be44f84a6a4ce7', 'ccd48b61a5683c195d4424009eb1d147'},
	ChapolyTest{'350f4c7ac2', '7c104b539c1d2ae022434cd6', 'cbb61e369117f9250f68fa707240c554359262a4d66c757f80e3aeb6920894fb', 'fbb14c9943444eac5413c6f5c8095451eddece02c9461043', 'b5c6a35865ed8e5216ff6c77339ee1ab570de50e51'},
	ChapolyTest{'66234d7a5b71eef134d60eccf7d5096ee879a33983d6f7a575e3a5e3a4022edccffe7865dde20b5b0a37252e31cb9a3650c63e35b057a1bc200a5b5b', 'ccc2406f997bcae737ddd0f5', 'd009eeb5b9b029577b14d200b7687b655eedb7d74add488f092681787999d66d', '99319712626b400f9458dbb7a9abc9f5810f25b47fc90b39', '543a2bbf52fd999027ae7c297353f3ce986f810bc2382583d0a81fda5939e4c87b6e8d262790cd614d6f753d8035b32adf43acc7f6d4c2c44289538928564b6587c2fcb99de1d8e34ffff323'},
]
