// These tests are derived from the Secure Hash Algorithm Validation System
// test vectors contained in:
// https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/shs/shabytetestvectors.zip
//
// For SHA512_256, the test vectors come from:
//     SHA512_256ShortMsg.rsp
import crypto.sha512
import encoding.hex

// This structure deals with short message tests
struct SHA512_256TestCase {
	name    string
	message string
	digest  string
}

// short message test cases
const short_cases = [
	SHA512_256TestCase{
		name:    'test case 0'
		message: ''
		digest:  'c672b8d1ef56ed28ab87c3622c5114069bdd3ad7b8f9737498d0c01ecef0967a'
	},
	SHA512_256TestCase{
		name:    'test case 1'
		message: 'fa'
		digest:  'c4ef36923c64e51e875720e550298a5ab8a3f2f875b1e1a4c9b95babf7344fef'
	},
	SHA512_256TestCase{
		name:    'test case 2'
		message: '74e4'
		digest:  '0c994228b8d3bd5ea5b5259157a9bba7a193118ad22817e6fbed2df1a32a4148'
	},
	SHA512_256TestCase{
		name:    'test case 3'
		message: '6f63b4'
		digest:  'a9e2427cec314b2814aaba87039485fc8d3ade992fa1d9acbb7f6769460a7317'
	},
	SHA512_256TestCase{
		name:    'test case 4'
		message: '2bf98e7f'
		digest:  '97d455e6881adbb8bf26ddc3ef120df4abb330adf8ed2ac7a4b6cce4a1727068'
	},
	SHA512_256TestCase{
		name:    'test case 5'
		message: '77d2ee3fd1'
		digest:  '329d0ae26df3646fb7db0b322cf1f3488d5381f66eb376fee40f0596d6d9dc4b'
	},
	SHA512_256TestCase{
		name:    'test case 6'
		message: '7dae8fc020d9'
		digest:  '447044f03bc30e2caa245d26ce4c72c1454f708cfcd9a215841a88cf5ecd2095'
	},
	SHA512_256TestCase{
		name:    'test case 7'
		message: '5b48b8b01a59d5'
		digest:  '1a86b4d34cd104c13b5b8d419784ce4c6d35594f2a9300e81468f4dc0634832a'
	},
	SHA512_256TestCase{
		name:    'test case 8'
		message: 'b4e2e8501f54be91'
		digest:  'd25265bf9cbc0dd2f108a2f5e8f69db7d15e5b8fe9100fe887dae20b6e054fe8'
	},
	SHA512_256TestCase{
		name:    'test case 9'
		message: '901db0f70f2d10626a'
		digest:  'f84ce7a760c85e67872ffca1722b1b7741032d9139211077f963a32d9f84771b'
	},
	SHA512_256TestCase{
		name:    'test case 10'
		message: 'f24d98591ee3c732a172'
		digest:  'da234c1d6d18ad8f53070e4e5b9cff75385c13f3e44b014fc45c3509d0d06adb'
	},
	SHA512_256TestCase{
		name:    'test case 11'
		message: 'bfb272537f021ade9bb330'
		digest:  '7c49a59a3c3fd8e24de85854f68a3b7ec60c2031742a9a2dabf49199d88c2db5'
	},
	SHA512_256TestCase{
		name:    'test case 12'
		message: '0a5f7424babb5cb85715fb17'
		digest:  '5cd40c51502600f3a0eec7b3a3addbf6672eeecc50b3de2fd86dd1c0a47e8a3f'
	},
	SHA512_256TestCase{
		name:    'test case 13'
		message: '57a10308aa2060aecad05adb2d'
		digest:  'a5208ece1de1342d02ea9ab7cfe537a2c81fa0eaedb427a7b3cb6c291959a6a8'
	},
	SHA512_256TestCase{
		name:    'test case 14'
		message: '5e4674e97a257c231d8e91391d0a'
		digest:  'a88d7cbb6e31e87a6c99734f599d1588cb9439979b8210800eca0aa2067cd556'
	},
	SHA512_256TestCase{
		name:    'test case 15'
		message: 'ba10b0da70e2a017cdc63f996dbc1b'
		digest:  'f8a5aa22937453719eb40b1dcf68a346e202ba225f4d7acc87e5a736abed3423'
	},
	SHA512_256TestCase{
		name:    'test case 16'
		message: 'a3481996c109c9efb6692b82173921b6'
		digest:  'fba0f0813ca5c513d0b547419a15e27a7097d3eb280a298c1c693b55e03d2d2c'
	},
	SHA512_256TestCase{
		name:    'test case 17'
		message: '0877c20cddcc86b833f7bc2dcf7a709a2e'
		digest:  'f4c3a2604e2725e4d179edd1b445ba489979f4ea5937e0c6b20b422c720d8bfa'
	},
	SHA512_256TestCase{
		name:    'test case 18'
		message: '69445bd12f022f6a669bebe0467729f4dc7c'
		digest:  '26bf709b3474554ab8dd12d9524faca43034118bd955fc52cb1f355c6df22391'
	},
	SHA512_256TestCase{
		name:    'test case 19'
		message: 'f751f428a0af0b90a912a3be78daf12a765b43'
		digest:  '5e0f8a8739f556daba4da4e861f39d1edaaece0204963942a4e66a11ec1fcf13'
	},
	SHA512_256TestCase{
		name:    'test case 20'
		message: '19829a8c723c58ca9580cdf0dad2b6d38cdf60a5'
		digest:  '3aa791b396ab81df9107278b8dd1c46ee57579db4cf5f9b83c54b37b0a601fc9'
	},
	SHA512_256TestCase{
		name:    'test case 21'
		message: 'ae43132ac30de180c24dadb88602caea91366b8e51'
		digest:  'b9f0141ba006a3983972dbaa5e2bfe7152e38110e02475f03d84b0a3c743cda9'
	},
	SHA512_256TestCase{
		name:    'test case 22'
		message: '8d2782a7843aa477b8f3bca9f9f2bafb5813db4c8c43'
		digest:  '5424b31989031809b5fc8969b7c48b5dc0c233fc34ffb5b223cd5f3a9712a8d6'
	},
	SHA512_256TestCase{
		name:    'test case 23'
		message: '72321fffd7918d4f5cea105cb0b884a2356af11ebe1267'
		digest:  '17e3fb33e7f85b1ad5634fbbce9b5f9909f3329f4d74acad3dd0941c1531cf54'
	},
	SHA512_256TestCase{
		name:    'test case 24'
		message: 'a7efce3e31433fe68d75145d84bb02960fb968871a3c2f7f'
		digest:  '8ff0be0601e3ae9290715aed30634b13b8eb6f3c8b3294ee9b1fc342c260313e'
	},
	SHA512_256TestCase{
		name:    'test case 25'
		message: 'd4d022cf90e6437aa0bfdb89284244c16dee0b488bc345ab66'
		digest:  'cb3a62feb0484be0ff9b56ea92414c7083f2b5776e883508b3a61e57305ee83c'
	},
	SHA512_256TestCase{
		name:    'test case 26'
		message: '95de55287ad3cff69efec6e97c812456e47be25e433470c3259b'
		digest:  '41c7027d16e37259645d0173c86141f38d808e9e27dc2dfeeaf335ed7c99490c'
	},
	SHA512_256TestCase{
		name:    'test case 27'
		message: '63d8cfd72768c44920d7b015460489ad578c063be19053889cb809'
		digest:  '876e59c8a64faf9d665f7cde5d42fbb331ba818ddcd284491ac51ed50e1613be'
	},
	SHA512_256TestCase{
		name:    'test case 28'
		message: 'bd2c5a9116bf0d75b3498ce2ee4ba2868e06e0c047ec40b181540896'
		digest:  'eb537df04d512c2fc844020f8ff32e043c04614b88cea0a4e05e0ff32ccffa06'
	},
	SHA512_256TestCase{
		name:    'test case 29'
		message: '37a45643a280a9d1a1c83e094e01173acd8aa16825252dee4eb3a3678c'
		digest:  '80bca5ba3e632e8bad1589cd02d75d7d3e6a1aa9ff64c2b982d572f3d00d947d'
	},
	SHA512_256TestCase{
		name:    'test case 30'
		message: 'a7be84014e8d09d8d51657789c07f29f7da49c5b5c0f836fffc6baf8206d'
		digest:  'c575b2ba075a63c4540084493636deabe9b8df61952f071559394d4dab810e08'
	},
	SHA512_256TestCase{
		name:    'test case 31'
		message: '93c45fb91dcba420820bd3fee334b48fdf7b73d7b431ebdba79b2310a7fe19'
		digest:  '98151839b8f3574e0a0b9f19e84afeb1c620fb2dcef248149c00613233337997'
	},
	SHA512_256TestCase{
		name:    'test case 32'
		message: '1b77c8dcfd2fc4b54617054fa6b14d6e9d09ce9185a34a7fd2b27923998aab99'
		digest:  '683b486861e598dabba740ac919522cf3b609c18205b6beca4ccbe6b0f6dc6db'
	},
	SHA512_256TestCase{
		name:    'test case 33'
		message: '8d5cb933401b35dc34b15de0f5474d3c9b6aeb9057531b7f4b6b32599f0eaef7d1'
		digest:  '16c523d2b8830ea90dd6b6b1ca49e4b51673006ca30ff47a4dcd4e7b35ad4f62'
	},
	SHA512_256TestCase{
		name:    'test case 34'
		message: 'd6a66d66bf546b97ebd033cbed3504931e45296334ae28f4178d3de6f03106aa8221'
		digest:  '3f607a5c7aef13fe98dc9f6b6a4b52979e7c68781ef5e12f69ded20c497eda47'
	},
	SHA512_256TestCase{
		name:    'test case 35'
		message: 'ce885ffb340fd7a43e2150699b3f98ef334270036794ec759aa01d29f087ed6b551549'
		digest:  '1071f8b57328b767209c15a43e1d8452763865ac93d58f93c840876c1a5b21e0'
	},
	SHA512_256TestCase{
		name:    'test case 36'
		message: 'f2c28d685486be99afe87bf90c9f309357b44f60b688446a1918aa321180bf4fe5be1559'
		digest:  '35bd192cc10fa24bd7954e0e0ee20c09440efc6416e94ca25a47e765bf32ef1d'
	},
	SHA512_256TestCase{
		name:    'test case 37'
		message: '96e8e30c75fec82bef4de7ec8e053ea661cb72bc4eb9c1c1aaa638df07a826794dd8d00f0d'
		digest:  'b556798f76ef742cc1eb5d10732a925fd8208b4f079a6c0e1fe0d92fd87564a5'
	},
	SHA512_256TestCase{
		name:    'test case 38'
		message: '2915c9b58835f0ec1e011fe5e6db40ae3ef3d890e268934d1bb024a634394b202e503708b543'
		digest:  '3868885b4e3ae2172305acaad25ffe9094c72348ee9d7b0acc20029e26a7d013'
	},
	SHA512_256TestCase{
		name:    'test case 39'
		message: '673ccce2807cb47622b7e09ee8efd812b44faeef436c301505cc63cd2ca52db003e652c4827ba2'
		digest:  'c7752098c3b5fcea98425a180a5c96146b443aa8d76a0c79fdcf3716bdd2b906'
	},
	SHA512_256TestCase{
		name:    'test case 40'
		message: 'c1a6ac69f48008db50958665df9d911b78dfe519d9f34366f6cd691f90f741d1081e592676d093c4'
		digest:  'bd18af5b2b24d769eade525c8fcd535212abf8e5df180d7f3e3d023033fce192'
	},
	SHA512_256TestCase{
		name:    'test case 41'
		message: '3cd123ae7bfd2b31613ecc5ee8b86e8e59a52ff46aeb2bcfa9f4fbe41eba7b2c2c21d0cab98aaa3f8f'
		digest:  '7ce19714e1d3be680dde0e919d90157aabca76439df8f0979d19b3a686df6b28'
	},
	SHA512_256TestCase{
		name:    'test case 42'
		message: 'a52dde5aea1f04399c5d91fb5c4e62da06b73a5d9bdc5a927fe85298e58e166187f154b69ac1057c01a7'
		digest:  '035d362a4265031434c3aa153ab4aef6b00d5176227197430bb62884a085c5fe'
	},
	SHA512_256TestCase{
		name:    'test case 43'
		message: '381906808394c5f9385a06bc9b6e528095e6f6ac07f0fc755e13594768b7686766e897d662576178c2a17c'
		digest:  '22ea5298af2aa311a5b53f8b2838dfc008ae470bf96d2bb6794735ea72108879'
	},
	SHA512_256TestCase{
		name:    'test case 44'
		message: 'a04a5d5d613d23703ab8dd0093d354829c5449823bcf09eb49124183fdc014b5a83b6b93a3c78c1b86edefb9'
		digest:  '8ae688d5e9d0de9de834befe5c0829644def9c5da71ae0da4f79d1890487e25b'
	},
	SHA512_256TestCase{
		name:    'test case 45'
		message: '175e1059931a94e6e37ea53c87e3c71d46be2f410b2131f8f285f810a3c679f9df6fa9275838eeb8c6cdd45737'
		digest:  '5641036ef6221e08e3c795b2ad566aac9a69f85df1a8d35b3ea2b780ced33421'
	},
	SHA512_256TestCase{
		name:    'test case 46'
		message: '3ce38a2e287fda13384cb306237e8c74357848818d340a9488e64a157ddc2975aba9016f6f82418ebbe878f0c388'
		digest:  'bc2041029b16895afb1f59f7280f8a8f09be5dd241c675343f7ad8d75b834b75'
	},
	SHA512_256TestCase{
		name:    'test case 47'
		message: '9628a195ca6e708cfccf9f434dbe6c5105327f0f28ee1cfd4a069d62a39f203366f67d364fa47b677b20439b7d8514'
		digest:  'df6d891d2c204354b827f530c54652940cb483f58d6b433973752c1fd1d2f84d'
	},
	SHA512_256TestCase{
		name:    'test case 48'
		message: 'd85bdb2df5d72f2190e4b3d4fa5b6d848a088bfcf0af52b8c64c584a689ff3fc06e8f1b7f7f0892350489210b6d5f315'
		digest:  'eb2681edf92a1f5559d10177110ad21ffd3fc1296e3cc83ece32e42f56ef7ded'
	},
	SHA512_256TestCase{
		name:    'test case 49'
		message: 'd078a5dc6ec589117119ba3cfb9b11de57fccb563bade8b34a4fd2c7dd6b3f7c0363586cc7a81babb12efee78f1dd5b474'
		digest:  '0fc7046f02d070642d821c746785f209bb332690bb34335e6c6428051793e1a9'
	},
	SHA512_256TestCase{
		name:    'test case 50'
		message: 'b7ae6037d9b452994fb755aa0cf57fbc687298060a471f9b0c3d8e9d5a9ae0ad5b3b4fa7d4e3628c09d327c0259bbd76db2f'
		digest:  'cd1040a0d94382a6446a691c0a07c4f643edc11e3103e343608603b5bd101be3'
	},
	SHA512_256TestCase{
		name:    'test case 51'
		message: '62f016dfc38a4c70b7f95d669e2d475704909e13c997d3da7e96b6cd11d83aa3adef789ddc49f7bcef042b319f033b15f44208'
		digest:  '8aee50572b6b155cf40d6ed6c738bdbcaff2ade09bd4d8bc9c2f9a673373c44f'
	},
	SHA512_256TestCase{
		name:    'test case 52'
		message: 'be890898b03f7cb52a9d4300ac2e8a80715102b1538fe37ce92c3165f2fc3e07f67870bd2a1144a395c593bd71ef1475d463234b'
		digest:  'ad156f2f416cceeb9a6a2520e79bba5295b584c59606fbdfda6ed13ed9d99af0'
	},
	SHA512_256TestCase{
		name:    'test case 53'
		message: 'c12149eed4113efa31bb3c4b9aa933639a0e1ca716062ebe251e553a2217e07affe779cedb015bddd7d6bf44269c9780644e7f0e29'
		digest:  'b4941556b37c22c9d841bd0b1c4e5d8778b1c598d8b1ec55c4cff11cc7402d05'
	},
	SHA512_256TestCase{
		name:    'test case 54'
		message: 'c579bffa9396172aa1d654b47930e65f95c0c74b6abda382a5ab3bd0b4d20bd588ee2a58992873de2b45e27bd01e40e92f5003d70334'
		digest:  '63f898b4fe7a7d8c7ac776ed7811a922052791406429318fd5386a8f7864cfc6'
	},
	SHA512_256TestCase{
		name:    'test case 55'
		message: '71b58f40e28bd9427f0f7821ece9c6e0f34b7f5cc924222692979a8e28ab1e0018445694ee52bfe0b3485011b3d05e3a84bcbdeb434028'
		digest:  '7614582f42d5a907d4ceba0025798ee61d38072485f54ba3d742f29503f1911f'
	},
	SHA512_256TestCase{
		name:    'test case 56'
		message: '592335639cd4f2f9ae6624e895d6a729b5079ee1f45aba911b7fb98c7f8b2cafe24a7e4c94d63267dd686c25a06fb38138dff541a6cccb28'
		digest:  '68d64cb4331b71e72b42c88b8f6513f4bc17d995254f04dc2b7cac022ad5edfa'
	},
	SHA512_256TestCase{
		name:    'test case 57'
		message: 'e26e1dc893fdced833eae15ac04ed4624a07e3c89dc3d1d8205be44bfd63327782ff1bc2c8f7175c920eb22e7790d40d442b46349cff72f1e1'
		digest:  'e1470b2f44e3955d2b5496292679d9086b7c0aa94f0d516e53a142e486c6ded8'
	},
	SHA512_256TestCase{
		name:    'test case 58'
		message: '7def4d439a9b7a6c5700bb91688b1ca6c4ecdff263c6646446ab4d2ab947df80d5dbaac648ee6cd6593f339426fdc2c7303a36271a9ee9b6ae58'
		digest:  'a59591e28eb92dfbf290a06772bb145655979cd0fa0df15bf82aa27665abac03'
	},
	SHA512_256TestCase{
		name:    'test case 59'
		message: '283998b5e2394fb1ce1cee355b677e79f89a09aa28fa52d5d993c307f27ea7b32b60995321333c52aa492d2a658d36ca960dc53a31b59c81355dc9'
		digest:  '82f27f2abbe5fc2184b428eb92551e898a5177e997e337eebdc49320095c605e'
	},
	SHA512_256TestCase{
		name:    'test case 60'
		message: '09f352d1ae807b258a9d49873a4c4aa4c63bfb52bf9180728ae7981bfb93d1f5262df8e2714cd4e4ee91143be8e785cdbc0888cbea44c4926c7e18cf'
		digest:  'c30efcccacda26ded16e92b0f2f2e667dc98f289a754cac628dfcb421ecc7240'
	},
	SHA512_256TestCase{
		name:    'test case 61'
		message: 'e6c2b24789348d646a263ffcdb6fcd511e1aac527b227a7a907aecf6e9500806d09394a566b3ce6cf3a85db47b3588fb926ea6897078d5524890ef1fd6'
		digest:  'bf9ae2941d2bb948d33ae2a822e8738f744b5a9d692997719bd1a118743a0731'
	},
	SHA512_256TestCase{
		name:    'test case 62'
		message: 'b099e1b52348f93a6c22896c72854ab04f2dd3d1f94c5fad21ddd12e5b35f57c80db097150997c4e5cd3f5ac241897b57cf7849334149f29f77b97017433'
		digest:  '01d582c82ce6e6a8c0cfa81792110bc950fa55e31c101cc6f2df6b2aebc18925'
	},
	SHA512_256TestCase{
		name:    'test case 63'
		message: 'ac4065c8dcd58031b7475c3debe032168fa9f7ae5ad02d48134dfe3b48c58bf243af82383ab5eeb2c65d54ad96e8fb1b657b8bd7297cda318e3f02a6a2e2b7'
		digest:  '9fb9d7e6b81e86e441695ca5c762cf3b7735a75dbc4b2b1e3e5667848f6cdd73'
	},
	SHA512_256TestCase{
		name:    'test case 64'
		message: 'd2bc0ce7217ff2e944e1ae47ad5873bf391f1b0cc07f6151eb4c50bb45b2fb6295326f716ce7e687fa0e3d5d25c5a8a8dd13a541a9292e8386e733f4f2a24728'
		digest:  '9c1fea5786702d027bb5b66b3fa92de34621a8626982ec21c0ecf8daa79dea05'
	},
	SHA512_256TestCase{
		name:    'test case 65'
		message: 'a82aa079ad878cf1d699e9612fd3aef603df30c3e8904358fb18a2976d7b91c33f43c0da4976732581c5e00d7a06d8302be4aab4f2d545dbd94d9cd817324fe2da'
		digest:  '46eb49bfc5b2ee3c97b91102525b6666d56bffaab4baa03059b89f24ec4b5622'
	},
	SHA512_256TestCase{
		name:    'test case 66'
		message: '72fee29494c25b119b6c5a600c468ec22e42af5ba93eb79452864ebe469a86f83632c85201800f3288b553f7bec649ddfe704920a27a8f65d13aa755985a238b3cdc'
		digest:  'f7eeddf08d3c1ab524167c908bcd8613d581cb0d03520d7b6b64d181bf13bf1e'
	},
	SHA512_256TestCase{
		name:    'test case 67'
		message: 'feb7a094dffb5c0cb6f723c1b5e7d764d4ab5f896b13e409005940a45defb3ebe67bd85e2a21d8d1b800de1434ffcbd2aac92aba211666dd742d3a575da66a32d375b1'
		digest:  '40cfd88d613bc7d105e178081d2e3c5a09cd629465821909dbc1e9b8cd04a266'
	},
	SHA512_256TestCase{
		name:    'test case 68'
		message: '8ce31ca2ab2e7cff5ff5bcbcee367463cd5eee65b80bdb010116446729d1c31ead8df2a3226024538425538accc35a0dc0b364a50fbca50d9d01480e1c3ea8e9dc4fc2f5'
		digest:  'd61fcf2c3178cbf34042a0fc02869f4670ea400ea384ccf04d8460806fe11dce'
	},
	SHA512_256TestCase{
		name:    'test case 69'
		message: '4963a36af562e045ec25e39a0d9c2a035d4f6d62044fc975986d1eaa9eabb6344ef0275e97007a1f63aa8cef1c44b5da6f54a8d350909a4a3e09a701c8a3711b50c80a760f'
		digest:  '69df54c0fbb35859b5fb247c92b5665ed27962f042a790519db8ce27343b9590'
	},
	SHA512_256TestCase{
		name:    'test case 70'
		message: '0663ac246f643427d7198feb54919f97f9e036864dc32c02fc87af50cd26dbeed4b48a18e9f96e9059605d9fdb10eb57c5583e1c7f501e6338bd4cb8d9361506378526138d50'
		digest:  'db458a327dd21bb9c073bd5736a745e467538d9476b7647b74dcd17031bad72d'
	},
	SHA512_256TestCase{
		name:    'test case 71'
		message: 'e70df9ce76334cc2096c0e715d8c96652492b5b6637d3e801961c59421b9a66e59ff4a68047aa52d6d58036fcefc05dd6f99e64f120003448fb69b8fbc4ad34d8fe5fd1a872319'
		digest:  '229be9deedbd16e948eebf8aa5838d276e0e6e295570c49eadc9f27c0e2e5798'
	},
	SHA512_256TestCase{
		name:    'test case 72'
		message: 'dddad5815834a99d79abf4ed36af74065b2e70e87cb8483b2ee250d877e8de8b8296a2179768367b4bf0f674f478124360e9e2b0c7eb794ffd4e7c02eca935bf4240e394672b54e8'
		digest:  '13b11a38ef74ab16bd65b84e83d34848bf3c56e6cab5c56cc0541e783df1d4b7'
	},
	SHA512_256TestCase{
		name:    'test case 73'
		message: '84617bcdea87420cb0e93feebfde3aabbf7cc7178065c81c99e333684f381563b82fd76420f7c024ab31bc733c485c2996011d60c153ede0b078ebdb20446fdbc3b64af89ecd4f4eaf'
		digest:  '5bb51bcfeedc618a28ccf3bbd96be08972eeb1bc22d9bb64883dabdc6c3c6b54'
	},
	SHA512_256TestCase{
		name:    'test case 74'
		message: '970f5a9308f929c5de78e8119b96caf0ad9c4a06f182c73e75a9711c82cd694fda3524863e2c118533b57b78067bb0870e0763aa31354fbfa9e91dda08fe88dc7a7fbf474b1a0a50a9d1'
		digest:  '930a9c33e393ae8b1c39e072b08f972ad840db5401a4eded1f1fd14f8a0f68c6'
	},
	SHA512_256TestCase{
		name:    'test case 75'
		message: 'f29ec443a7c4bf4fcbc2cc4e992f3b1e0c70ebf60a2019efdcb676c3e2617cd4853701706c8f0d53f3abe96dd99b78e6385dca60bb49ef7c0bad099f8ab4a5f6448c4cb723c754aa6202fc'
		digest:  '36d1cb5d930d73f01999533b035313a54b1e2cc223f07af829424230b8405177'
	},
	SHA512_256TestCase{
		name:    'test case 76'
		message: '4913709a09e606f0034926f21ce0faeefc0483b5c0ade99fef950fd0b09a6bba3a6415e38b491de85d6d22c3562e20497fbf8af6f71299ede9fce33bbb942f5c88dc8800d1a3900f6fa917ea'
		digest:  '0948d1dc7a0a0172d3dae4ae40d128444f2f6309b443ad6a8ab3849def606a7e'
	},
	SHA512_256TestCase{
		name:    'test case 77'
		message: '3365dc87c7f29b50925f2d50b1e437a49ca995caac35fcb3db29dd19cc2e8c130b2ca1e795bd9f1f9be1d3df83f37f7530db383283868d782193e915774fbbcffce4d340b6cd0fe5008d38718a'
		digest:  '579d6c492dcfb032b9272aa0f54d13ece0fb2276619dd7e75f86aef3f58bfaea'
	},
	SHA512_256TestCase{
		name:    'test case 78'
		message: '45a899ee24ce1eb7ffe13aa5acb08cabe94ff003957153b72019c6664072d045e5fa096470e6e4331fc3cd44cc597452d8d468836226e7eabca7f5501ffeab349d45d2105868b0def2ce22db443c'
		digest:  '5c088cd0c18bf699edf47ddbbd1e11c9caa9161b852a6024492e7ab7dd5745ee'
	},
	SHA512_256TestCase{
		name:    'test case 79'
		message: '389d76c18d8d11eb5b7e6e80d79f93be52c784eb1101b7f35948c752c98f6c34ed05abdb0f964a4250b58700f8590a6e0b45170c1d8d08216656fba219ddc855f2bb36360e676e1a00fcaf92213c44'
		digest:  'b32d1b0acc097eddc13dbba91d2c144774f4bf1ba2275770b52858471de26ca1'
	},
	SHA512_256TestCase{
		name:    'test case 80'
		message: '120b80b44a2435d4783766416ede1784e00d3467c7ed1b6304e75b5846febb812d637fa6b043e693bce4f7ca1df07d75a211519e7cbdd7a2e854cbe30e683ebe285bdbbde6306f0d119ba932705a1ef9'
		digest:  '200818a7fa963423d0a90bd85475d67befdfa43bb49be7c026b9693743404dcd'
	},
	SHA512_256TestCase{
		name:    'test case 81'
		message: '713705674e20066d61bfe9ca763a8705cb6ca1e8cf95c6d729507661fcc687156922c8975645e5f36eba8a3069eccb298e96c498767c7c741259107df0180d15b9461d51ffc8e65035c92fcbfff744eead'
		digest:  '786ea368156477616fd76039f9b1caaa95763adcebc1cfdae8198f82a41182a9'
	},
	SHA512_256TestCase{
		name:    'test case 82'
		message: '123ccaec522c2983b6cc2e06b0900c331db49dc5d5809cdf80cb14aeacbe184b1ddab5de643c2509b4204584f0a54a0408d0af6286d271f9648e7f5021566cf914f568b8fff36ee8e714bf1b23228e01b4ec'
		digest:  '91b3e2463558941d0626936a8d275722fe2e600bb24df78d7c46752675d1c4da'
	},
	SHA512_256TestCase{
		name:    'test case 83'
		message: '75d040ddea0428d8231dce4fbc98cc4a2f578a5f47475379d740aed8ab2ceb59bf6cfcf8af7786936fc438f13ceb5c02d207ab68e05e6637b944a2f94c3ce4f65661edb2d604d7b6c2976193f8c9882577f357'
		digest:  '749bae9322af588495dbc0ce426debd9e9cc4c02f50c52ca246b5582840a86cb'
	},
	SHA512_256TestCase{
		name:    'test case 84'
		message: '7c238aafde6aabab177f718e0cfe1386cb1421b4799b15788b862bf03a8072bb30d02303888032baf5be242341822095fa71487c68024aa0c1df31aaa08e82e4b4508440b549df9c9de26844146c3cb82c060cb0'
		digest:  '36ba7bf53f7d253738f590ee25c6120e0e89b1b89c2a0691db6d38b72c20055b'
	},
	SHA512_256TestCase{
		name:    'test case 85'
		message: '470be1ab595f8bfcbe651aafbe58a90cd4d72dba54f3437a50872c16bdc81593d21c9caeb13091874944f9977bca59e77b3031847afac62e82132450a165643549a72b78f77673bc13a88e9e17dd67e8f4560c07b2'
		digest:  '8ef8d6472e09f566f70e17b454489c193d11fd99855f7e69c0eed389515df577'
	},
	SHA512_256TestCase{
		name:    'test case 86'
		message: '6973b2d5fd42e78a2245c76f25681d704ea194e2b578281e58c6db6f6bbda92fef28bd2c86f4e55186f044824cd147c7d379daad8579dbe950d4ce3ffd06d3a2bd44d69d050606f19cc38880f3fb6c69b47be0759ddd'
		digest:  '226fa54ab31f5a78eab4f92b10972fcfc8ccad25e0ffa49365da83f27220f6cd'
	},
	SHA512_256TestCase{
		name:    'test case 87'
		message: 'c4d0d14c93058865d0fbe2e6e14e99f170cecd75c183f95343ef26bf65925fe9aab464dd005ea3a84eaa63bd4c3c43979711e39f0578e14f355d07986bcdc2819bb7d612c4f05170807abd49d8069a725cb70ccf7429d0'
		digest:  '88e183a794f478a850856966fdedd71b14d8aba72819fb26f14d2315c67f2deb'
	},
	SHA512_256TestCase{
		name:    'test case 88'
		message: '73661799c7d2e24ea484299d8c8c1125739dce5cc307c327a3e5eaa879987757ad74b855b1b840fd3ec444bbe0864521b436ef3770434213a58defaa41d405e65870ccf92443f1f04aede48b5e3e7f4caaf1e8ce2b97b858'
		digest:  'ad1affd1b194bc0afc389a2cb23281244a8d80a524262b2560d58f7f6d226f51'
	},
	SHA512_256TestCase{
		name:    'test case 89'
		message: 'd49063453b779b18933d233369881617c987fef4b18eea6fc600c1a24672d5643e0dbbcbfd345bf0c86743f2cf68413e609c300d5bb3122ee68c5f0561529dd49dec3139489475f8acc50afb9a5b7097a0f6d311829f2861e8'
		digest:  '6768946a3a27f46288ddfd4cb7b5fc9e27cd3ba46ec959e6f3d0cb5bff68e226'
	},
	SHA512_256TestCase{
		name:    'test case 90'
		message: '905888121360fe46f86dddd119076195213443f339907f87e5a05b122864a5b7ef2b3c6badabd2a74251ef5f21d63938b213f4f6f516c5336975bb572ce28fc463002b51dc766afaf2411fd6b14e58493b48e47a6aaf1207b72a'
		digest:  '145ff262e203957a88a875ef986ac484204b20d9c4be7d35ad59612764991d49'
	},
	SHA512_256TestCase{
		name:    'test case 91'
		message: 'b752ac4adc97012119d895a800f318db6ae7e3c70344949d09e0f80550cc2fb48a860a41c570b8343581a261343b937487dd1ccce3d9378daf882af00c1daede13d851be5d7c198bc9026a29798f29f91ea422ab7694a753133c7a'
		digest:  '2f399dc553da607b1c2d490e6461664a5a2c007f1800d76a2aaf7742a86b9617'
	},
	SHA512_256TestCase{
		name:    'test case 92'
		message: '1ef7001c52803938e937d93f5c853f661ce483b4262eac63d81a2389030368f841ec046289c2bfc100e012f512d9dfafc271608f3eec5e89b78ca38d6f55b1671413b4e44c9eb636915d12993f092b37ebb35d49f709348063d5dca3'
		digest:  '77e371349c466f6c331fac0284987537c15d0430e3f0cca8f1fe2aa7c794ef09'
	},
	SHA512_256TestCase{
		name:    'test case 93'
		message: '284bff3b6a683ff220f5cbfd3c2fb8a72bdb8506fcdd5b2dc50794a3422f7b14c084524739229a50c1e38e072078f1b8c1b9091ccb3aec888d1ffa651f31abe046318be2996c8243e28f593bc7bd7a19aadff737cd3bce2dba496c45e2'
		digest:  'bfaa54963521e6e1ac72b55ee4d654c041f62b73ad5a5305c58f47bb25db7cc2'
	},
	SHA512_256TestCase{
		name:    'test case 94'
		message: '081d08397e1353073f8df24675588ad215e4dc3615a59d2ad9b9815aeecb9a69fa37e036f36f115e909dbb02fd8a96cad3be182947e944e3a281c3cdf1ad35d4fd62c9417dcb0b3c8beffe8e558e6bab154b78ef43117c2808af1255f7c5'
		digest:  'ea8fbe273cd2bb3ab9cc7f3640fb4e3153a028d8ba9a880b3b8219006c28281f'
	},
	SHA512_256TestCase{
		name:    'test case 95'
		message: '1ed6244a5996d5a4423c0106a4b8e785d750c7313843f3dfd6ac272c6eaa1dbe9abe5dcd457453e26246a3dfddbdc140d2d9bca0feb35a7f73616a428f1a567b2e9af86b1fc8aec6d597b1b55f2aa2219b3b662fa6bd340796c0ecda6c96ab'
		digest:  '8fc2263517c85d7ab8926595b8047b4b838a075a29b24da712303c3e6e597e49'
	},
	SHA512_256TestCase{
		name:    'test case 96'
		message: '63188781f4e9cbd1e89a54a65da053b93722e1106f00f024ad1582421ab919326f8a6e17536d6596e3cf413a9231141733e37aae540f8711cefafe489a87c4f2e6fd942f6809f3bef3076763487de48c2ee88733c5bc870617a668c6f01471ed'
		digest:  '91a8e285029085e224987078066486b6c605cbac27e49e84f4639710ddd05d33'
	},
	SHA512_256TestCase{
		name:    'test case 97'
		message: '371171b626ccea8b2980dcd33bcbb849fe57d201000bc2443d830a9be6befa2a6117d13f9177252b57c8855159ad5c7c02701876ca3c8cd6b78c931923e3f7f4c1be91e9dbc6b119dd4bc05917ee2387bf9d631f35c58b098bc2d2b5459b7a6a5d'
		digest:  'e68655149dc14bc5759b8cacb7c2d2a850456a64a87af41de84f567ffe9c5e60'
	},
	SHA512_256TestCase{
		name:    'test case 98'
		message: '9bf7c7d99059248589b18a3fd3b03f4a72d21d2aacd635b5cc71b86d2043180d440a6a62a2160f5c3afd93659f8fd5f1420c6857f3953adfa5e1731c7231a07cdf86111d60117fe21febf3c7f906e271c9649d0aea86e94bf728e9dad52b45d7f4d3'
		digest:  '385b95541558ad0a9a20cb91116eda03c5a694ca95eab1caf5c0d1501899d3bd'
	},
	SHA512_256TestCase{
		name:    'test case 99'
		message: '1973263a26d4a418c691dd5f5ccb7dd7241ebcae24ccebbe60174c0f4eee279d89cc1a5bc9deb177136437e02d59cb065556b6e132b4fb340bf1301337548da3d5d44321bd370e25b6770e28e969d34495a0b73c059070bcbfefe5e2ff6141fc635346'
		digest:  '39d8740d56e58f027a6af13cce9b6484b23fa6b3693354175c034ac1778a1a75'
	},
	SHA512_256TestCase{
		name:    'test case 100'
		message: 'ec8ee65cef05452defe832e0a3a3ba6116a8d5d6a8638e2e566f63fd3c0aef8758c6c61a9182a1a2f899bb774d597a79c5b7525f0a40e74c845371a3d78e651dad865219b5bb0833eccb00171be5f898a081defc724eef9a606c20a899dcbd4e713550fe'
		digest:  '3e7a652d89df9ed7cacf53d033997c87528b8cc37d59b8eafbedbe4e8e9c5c3b'
	},
	SHA512_256TestCase{
		name:    'test case 101'
		message: 'df410a8b65f17139fc8eb8495df813b1f6f9f1b7471c7e628cc36d2e9c74f60311741a68a9972ef5ec5697060f22d3b719f8b90c4436016fa6ad066010f7cb68370a3c4e110eb78e826ccb07382840f4d470571d4a75524f8a7937096029d278982ef9a567'
		digest:  '294c8788036098f49c23016534d032939d5e9638e3e4e8474533f52b668f2da1'
	},
	SHA512_256TestCase{
		name:    'test case 102'
		message: '27c2ee5f63c114c1f61530685c19c21a4b7d47b3428e6b8265b87bf0884919628a773361e4a536f981e27e5ed1b312d619128ed20db6605ea09537bc1ba8b94c6cee75b0c1b3182c5612e3538bb95ce03aec44e247cdca1c79dafeab88b788e9772e9a922374'
		digest:  'cf027f6017618d970da83865d8da560f454ce2bd86af59624fb87c8e48beb208'
	},
	SHA512_256TestCase{
		name:    'test case 103'
		message: '2f510b72ad1a26fead669ccafe9ad0fe31ce0a12f3d06bee9a6c2750fdebee6d89daabdb910d71aee6e3b2e2a8cab3f95b063ce43daf9427d7f9ad39d3b99b290d244a87edd97182f322c24e937664c94443a25dd4ebe528fe0cdf5a3e050adfe4b6513f68870c'
		digest:  '8969727f522ab392d11c3cfa7d02fa2f9b457a1de2afd360ac9b2b4fece84e32'
	},
	SHA512_256TestCase{
		name:    'test case 104'
		message: '2a81eb0e6f1d13df6544c69321c561a489ff9d77b9ae11d7a604f8a201fc35c25feb945f965753536a0de1df696ee59c130a82b43e70d951b943d0563091124e4f6158a12a8be60eddcd58556f1a9e34fd354695147ff9065e0e496981feaaed440cbd3df460d5f7'
		digest:  '3818a14b4abd900c823275d45afea6ed1df6cc659eb3ef387445f8ad3903623f'
	},
	SHA512_256TestCase{
		name:    'test case 105'
		message: '9aa1fa540acb4d2a59a6a3aa8c2fdbcd0a4a17b6b55508e65a036cb34b68d4f64a50ab05a9d574e1b03153b03fd0cf6db4aad6de0fcf01c655431a5d320ddcfde18bf91e510862848090c2b72b034bb4aa69b6e216858547acad8cfc76d9afde28f9ed87488c9e7d91'
		digest:  'f9424b3dd79d555e2f5f8a185933f77db95864f4b597ee805d5b62fe0396673e'
	},
	SHA512_256TestCase{
		name:    'test case 106'
		message: 'ccf2cc0cd223b80ebbd49db9ede7a22862812dbc4b1aa6c9fad07f8efe01716b9018084e2ea7616f85b7333d945c0c970f8cdd400130b98db67cda61d87ce55bd8f06e4e8f5abff22655d5d93328aca4249bbd76e10ac4a9adb7c6c04d64539d6d2209f53f6459006557'
		digest:  '06d1cd544f954160ade83a102eb935f7f948009591b3eadaf4765868c63450d8'
	},
	SHA512_256TestCase{
		name:    'test case 107'
		message: '5aa1b85ca84f24f487f69e796268826518154bd6a224583ef6c3f15c2bbb856bdd473599d32697badfefb33e06a55260e18b79a1433930751df0873c9e2bdb739e76562c3c83b5bbc67b4eb3da925111cbde4eb64669f71b36852840e45f24e563d722fd2b8f0860044276'
		digest:  'c760de5612ec9d047848727b261c8876b6c33b2c57ee1ac99da0890c009b9540'
	},
	SHA512_256TestCase{
		name:    'test case 108'
		message: 'b613eebbad7a1860928ae7329a38cfe86f1174a579740ecc5cff1216c2470fe25accd5cc212cdd80e096e5d1f582db05579ffc73d50d3d519ecdeac706768d478729112106c627525860e99d05efbdde4c457d9a6744d5da104dec3a43fb463d4612e9e3e7b2be7e05d13bc1'
		digest:  '012499f4e6e2af3b530d264bce97e344405ffe59c04b31432e7f0446a6ef33ae'
	},
	SHA512_256TestCase{
		name:    'test case 109'
		message: '823ef09a0a657d28ffe8173b6cf04bf4795a96b39b19339608d5323c6ed8cacfe42bf04121fc28d5eb439ce2d3a66bd42099437309c9326827bbb9fc29b02324cacd14bb0078650e76052f9d3d922620bf2f29c9f44317c9d82be3151c23eb6a1b6a858e11a1dbda8774faf512'
		digest:  '08042c79507268fdcebf12719610219d1364117e66916b9e0c1a2aedf5b4a8df'
	},
	SHA512_256TestCase{
		name:    'test case 110'
		message: '9607cca45873add19a93dccf3d0f790e856ff30b84c8211ad69b8e628ffa142972ecac5264138423208c524c2b17e9250b780938b41d7cff43005eefeecfbdb53b4b62bf71eeedfe4cd028eaf5cb95c731dd4927d9cffdb18a1463209df4b68f5aea95f3684a11e9882605b28473'
		digest:  '73b4086d690ace6940c01912acb3a57bfc15c4a16c40a90b4329f1fa9f3085d2'
	},
	SHA512_256TestCase{
		name:    'test case 111'
		message: '5731f467c5b923c43af9c5fa849aad21ab8dd7db1ca1a687065571b705ea3ee4febdcd614ad4d98e16b79a4e09818ebb28367918f757ab06e1b481fbda822ef143adbb5b0e704d5d2222a73c0153ef14a817b5c9b7a2313fd115ccce4698e3f0efa9c73d5ab3089a27e3f3adb23759'
		digest:  'f5253b5c69db9c724aebf762ec51c221f8a4d4e2174a4b7f56e4d69aa44adfa3'
	},
	SHA512_256TestCase{
		name:    'test case 112'
		message: '92b23c0bc4d8d07d22e28812710dff06cb9bbecea2c960ac0200f480164fa2e1ee19926c7f0b095cec51d55c040aec990bf9501abd7d355490c366f93a3ae5127347d14dfc3b8d98e0821feefa1cd671b75230ba1da1fa6d0cfbb910c42f491da8a5c455424ea65886db2e735b2d07b9'
		digest:  '55a0597f11ff71c426201715beb585f254bb31c1dbade533f04e499c3391ff79'
	},
	SHA512_256TestCase{
		name:    'test case 113'
		message: 'bcc51ea0a66564a171dabfa279e384f4d9fdcba38028215788ee7d78c3a2769596e6b2070a6fa2d1200d6ccb65e52900c7015154a70c736a2f562ac4e61f4c2c81116453fd0e63b9ea2c92cc0afecb541a16e90ef0c77d97c630c38cd675d4f027501ccea6c90f1f784118ed8fb5d2b97b'
		digest:  'c16ca79c0ab44f39da1c65e8943ad2e90888c3d80b5e3b3b1bff59408b59d6f3'
	},
	SHA512_256TestCase{
		name:    'test case 114'
		message: '2183bf5465b18359a451cf3030bab07fa761d11926444c7dba88c0a01760a4ec8d32db8837f588da76de234a9d3bcd49ee9fe092c63fcbb0149c1dc10f62df792990276a3edaea82124c12d5fa2fd09cd930347e4ebffb0f2cb6a8972b3bd614dde7cc82e328525960e3c358ae5e6481505e'
		digest:  '7e5b4bd808b900735c0fe665da9437dacaf85249df653c6ee72fb296e8a9b8bb'
	},
	SHA512_256TestCase{
		name:    'test case 115'
		message: '959ba94b728a8adf53513521022558183a632e24ad82a3904b47b966c2372a8741524ffa82fe408507423d7aa4344dd43db52482d15b4a03b7af7a3328994965fe1808b1bb36ea90cd3c6aa61122e7de954991bca26d89effad544265ecc312553559b18261bd0272ea7834a51d56f5fc9893d'
		digest:  '85c2f125e6d11f831807820ccf0a53932713dbcef3e23b0e3c0aad8f89139cc4'
	},
	SHA512_256TestCase{
		name:    'test case 116'
		message: '30d62ff4263a731a205069cf7c0a0db0b50d3db13f08c00fb24586862b7e31c5eff1d51f52b6fc83d851eeae99842f9fea67484b39baca933979917340b1467ced9327840dc60b9cbf02a042d7dbf8d881da445527d2df3b94718ec9591606716fe2ae8c1f12a74d1f8c0af3feb77043c190b836'
		digest:  '4543378ed5a054557889b4634d4aa77bd30726b797235010d6a4920bf9cc3b40'
	},
	SHA512_256TestCase{
		name:    'test case 117'
		message: '3761a4861249c4e0115b997c2c87e85e5ea2bcaf44aea31ae65c1975d5cf45a476a3e3ba9d2e82625d05ef7e1fc2f10df205365222b98e330c9957df30f238b388aa135f19813444c797cf216d2fa9ef2db2150905f72290a3c1130e7b890f59167e1abbadcb444ff61820ae3e5378cbb8b03f1c15'
		digest:  '088167f12e82a45698cad25b405da5e24a0f4515b28b915bfe799d0990723598'
	},
	SHA512_256TestCase{
		name:    'test case 118'
		message: '6e26e2b41b80f26b87cba9bf5e11b734611e89110c9210157634c5667a89eaca14a612190d42e3f3db0f1baec34bd1924091fe49a8f63b910d51cc3461dade3adef70d5da55229bda62f3ad5bffdc1ae282a41520e3cfe847e1e05ae85b76c1b96c1db662bd94c33afd20e1c9c4f10efe1915c16868d'
		digest:  'ad549004c75fa7818dfa6e906062a850589f7a8d17a1c411fc955168215e45cc'
	},
	SHA512_256TestCase{
		name:    'test case 119'
		message: 'fe8b3dca283022daf7022a88dfb8ff61847509c41e9cec4f22e652971fb2ea1ce59cb17ced5070e6f5a185d2ebd5e7ef0c43655d4d212e1dfa9796c69d17177921f196163722e8b0f96d953fcdaedc958d6af794e5aaa09dfc0ad34b560dd662f9b37702ce4e95214b91e1384feae63b4d1247cfade248'
		digest:  '37a89a63b4a30e6a21549a245a2633ee5fdb618e6749f60522f5ad03a34d4465'
	},
	SHA512_256TestCase{
		name:    'test case 120'
		message: '04c48b4c37a0dafe8ab2c517eeb472385661001376296ac5ca2aaf67db5c30dfbe0b59fcdd8b83c3fa271903705afa30bc421f6d3d517eeb38462e721f5bcb3d13c01c0ba749f373e7645578994ddb5dc28558bb6c80bcd0a79d391ace5ca4240bcc1da11d42f2b75950c19deff5dd686f12f3228c1dee1d'
		digest:  '29a6b04353ea82f3f2a28503a125233f8afc5b601921637a0a24b83436abf9b5'
	},
	SHA512_256TestCase{
		name:    'test case 121'
		message: 'bc43ae59c1bed74fd4eb61be97b0276ec5adda96c67c3b4680378e026f3b4baff82c16b4fa3af159a7a2b1729277ae4208a2bf9aa42866411eab47986434df7eec2f727248b745f7fb140e3d8b82a9fdd7340ca5ab997db592de0369ebb41a983f53aadcd2bf3af3784a6165aa5a7e67d96ca9000259ea6a83'
		digest:  '6e1244587179eff28881bb80b02d193908a4c4d014b4da5b7e93c09e2b2964f5'
	},
	SHA512_256TestCase{
		name:    'test case 122'
		message: 'ddad789728581044ff760f79099e01a2cfc94ba27a070e22de714e28b85f54eb34db50e6b355432bca981cfa8efc21920fdd9147224d88d5640622f47ba2e32c9f0f57eb5a3df5d58b19495bb1bf538f66eb2a362ab1e41f626fbab4a0ff68cf2531d3cfa578eeea4ae3dec5fb4f5d41dddc5c981a9decaae294'
		digest:  '6e3ff96f4e4a722c912392b3b5fa20ae762ccef244c011268720962184383a66'
	},
	SHA512_256TestCase{
		name:    'test case 123'
		message: '8c7214e14eb6e0ee14de3ca0edcb510aec5079b3d143ff48e5d3268d261b6c99a4f80f432b18d1d82254f80fb3409a3e2d0017cda65de7ebe41896817e7d0c643e3c12cfca75121083121e81ae5934bad2c1c105600563139978f3edad98c32c20e5246d72f42c00ffbfb6e0243fa9e8a33d7e434965fc25ce9c46'
		digest:  'b4b4e5edda4fb4f7c0e734cbcca3c78dbd6f2c9a997ae85a6394108cbc5c567d'
	},
	SHA512_256TestCase{
		name:    'test case 124'
		message: '3b766fbba3339959b194163c36b4eb6505ac17e7b24e753e9916e3e1225da41b44d0031648213939d0fc91b3173dca587a60bdad3838403e76e449388600573368180d5091622084ecac74331f9382010ff6fde5a6aab57fd46fbbabd323c656b036d590075e16e30e8c07e3d41edb07cb561fb7850b9da6bb9e8162'
		digest:  'b450e083d9f3f58284ef80023e3bacb1826e560a22f6e6625d14ad1348df96b1'
	},
	SHA512_256TestCase{
		name:    'test case 125'
		message: '6ed8a9f9f2baf57e659aed127833edee9025a8481cf61d772556c75826a714f01fbc1953e5e184c4bc1cd21d27b3c272ffee09f1c31e470c2fde3adb109931bb564dc43c6b8ec33ac3cd6c49b5dae15789a8d98cd2519575fe5916622b378d0c3c11137b2cc89a27a334235ec7b132e555da7fe2f094eff9b94fc2b61f'
		digest:  '17312a26ef9b41ffebe31ee7ee51162704a2f05a5905ff8b692fd603942e9668'
	},
	SHA512_256TestCase{
		name:    'test case 126'
		message: 'dcb458193c4ba82a45bfdf028000841e4a8c671f8e2b841dd4dc2a3d7b0f5d86421478f00ed417bee7bf330e43058faf87095332cb13f744dd44633b89d055ee186702ca17f400225a8a63c7409b27d9376ccedc83c5490e3ced1cf896d6be4466943ecee1132cfea60ab754ff2b1344639d1e4540ab4225f8be9d9f2988'
		digest:  '9392ce14d344de05e4f87f634b457079343e270fff5b5c7b731e9fef137115dd'
	},
	SHA512_256TestCase{
		name:    'test case 127'
		message: '3e3a52d3261e1194249786d6c0e18d52d92f1c7639f079c26c51aa72d1032e5df13eea1d1006667002ad39de4099c29c3b4719b1f0904557bd2bb0a47374d869ac6b465b5f00c470b18ecb8c0ea53b5d790c4e832006cff534d587a0f77df95117ca4fd43a94935eda422228538d5e5d3a87a436f1db7e63785619ae86a6f9'
		digest:  'b34e72cefefb63d6e309bcfb4f0b1d350f2c5c582de3b93ad137f921a92a7e79'
	},
	SHA512_256TestCase{
		name:    'test case 128'
		message: 'bc8173c878ca60e9a0f823f9a589d4ff84547b389b117fb6bb1b614e7e75a9b1db0b21d9f73b42a73e94eccab3de5ae2845a54e5e24ba6c20fb4d245b964023b863040d6f080e953530d5fd944e8ffa525bf5364f65c88e06e6e22df4b8cee48e67738880a9f3f3406e9e6f001b0ac8f8e0ade7c814c0c5800d0b9e4ddf55622'
		digest:  'f691d01ee9ab675f3872313b77e6a4543c71e3e89aa94c48f91d6ee7fa1ab4fb'
	},
]

fn test_short_messages() {
	for c in short_cases {
		message := hex.decode(c.message)!
		expected_result := hex.decode(c.digest)!

		actual_result := sha512.sum512_256(message)

		assert actual_result == expected_result, 'failed ${c.name}'
	}
}
