module sha3

fn test_round_constants() {
	assert iota_round_constants[0] == 0x0000000000000001
	assert iota_round_constants[1] == 0x0000000000008082
	assert iota_round_constants[2] == 0x800000000000808A
	assert iota_round_constants[3] == 0x8000000080008000
	assert iota_round_constants[4] == 0x000000000000808B
	assert iota_round_constants[5] == 0x0000000080000001
	assert iota_round_constants[6] == 0x8000000080008081
	assert iota_round_constants[7] == 0x8000000000008009
	assert iota_round_constants[8] == 0x000000000000008A
	assert iota_round_constants[9] == 0x0000000000000088
	assert iota_round_constants[10] == 0x0000000080008009
	assert iota_round_constants[11] == 0x000000008000000A
	assert iota_round_constants[12] == 0x000000008000808B
	assert iota_round_constants[13] == 0x800000000000008B
	assert iota_round_constants[14] == 0x8000000000008089
	assert iota_round_constants[15] == 0x8000000000008003
	assert iota_round_constants[16] == 0x8000000000008002
	assert iota_round_constants[17] == 0x8000000000000080
	assert iota_round_constants[18] == 0x000000000000800A
	assert iota_round_constants[19] == 0x800000008000000A
	assert iota_round_constants[20] == 0x8000000080008081
	assert iota_round_constants[21] == 0x8000000000008080
	assert iota_round_constants[22] == 0x0000000080000001
	assert iota_round_constants[23] == 0x8000000080008008
}

fn test_rho_offsets() {
	assert rho_offsets[0][0] == 0
	assert rho_offsets[1][0] == 1
	assert rho_offsets[2][0] == 62
	assert rho_offsets[3][0] == 28
	assert rho_offsets[4][0] == 27
	assert rho_offsets[0][1] == 36
	assert rho_offsets[1][1] == 44
	assert rho_offsets[2][1] == 6
	assert rho_offsets[3][1] == 55
	assert rho_offsets[4][1] == 20
	assert rho_offsets[0][2] == 3
	assert rho_offsets[1][2] == 10
	assert rho_offsets[2][2] == 43
	assert rho_offsets[3][2] == 25
	assert rho_offsets[4][2] == 39
	assert rho_offsets[0][3] == 41
	assert rho_offsets[1][3] == 45
	assert rho_offsets[2][3] == 15
	assert rho_offsets[3][3] == 21
	assert rho_offsets[4][3] == 8
	assert rho_offsets[0][4] == 18
	assert rho_offsets[1][4] == 2
	assert rho_offsets[2][4] == 61
	assert rho_offsets[3][4] == 56
	assert rho_offsets[4][4] == 14
}

fn test_zero_state() {
	s := State{}

	for x in 0 .. 5 {
		for y in 0 .. 5 {
			assert s.a[x][y] == 0, 's.a[${x}][${y}] != 0'
		}
	}
}

fn test_stepping_functions_on_zero() {
	mut s := State{}

	println('verifying initial state of all zeros')
	for x in 0 .. 5 {
		for y in 0 .. 5 {
			assert s.a[x][y] == 0, 's.a[${x}][${y}] != 0'
		}
	}

	s.theta()

	println('verifying theta on state of all zeros')
	for x in 0 .. 5 {
		for y in 0 .. 5 {
			assert s.a[x][y] == 0, 's.a[${x}][${y}] != 0'
		}
	}

	s.rho()

	println('verifying rho on state of all zeros')
	for x in 0 .. 5 {
		for y in 0 .. 5 {
			assert s.a[x][y] == 0, 's.a[${x}][${y}] != 0'
		}
	}

	s.pi()

	println('verifying pi on state of all zeros')
	for x in 0 .. 5 {
		for y in 0 .. 5 {
			assert s.a[x][y] == 0, 's.a[${x}][${y}] != 0'
		}
	}

	s.chi()

	println('verifying chi on state of all zeros')
	for x in 0 .. 5 {
		for y in 0 .. 5 {
			assert s.a[x][y] == 0, 's.a[${x}][${y}] != 0'
		}
	}

	s.iota(0)
	println('verifying iota on state of all zeros')
	assert s.a[0][0] == 0x0000000000000001
	for x in 0 .. 5 {
		for y in 0 .. 5 {
			if x == 0 && y == 0 {
				continue
			}
			assert s.a[x][y] == 0, 's.a[${x}][${y}] != 0'
		}
	}
}

fn test_2_x_24_rounds_on_zero() {
	mut s := State{}

	println('verifying initial state of all zeros')
	for x in 0 .. 5 {
		for y in 0 .. 5 {
			assert s.a[x][y] == 0, 's.a[${x}][${y}] != 0'
		}
	}

	mut round := 0
	for rv in first_24 {
		s.theta()

		println('round ${round} verifying theta')
		for x in 0 .. 5 {
			for y in 0 .. 5 {
				assert s.a[x][y] == rv.theta[x][y], 'x ${x} y ${y} ${s.a[x][y]:016x} != ${rv.theta[x][y]:016x}'
			}
		}

		s.rho()

		println('round ${round} verifying rho')
		for x in 0 .. 5 {
			for y in 0 .. 5 {
				assert s.a[x][y] == rv.rho[x][y], 'x ${x} y ${y} ${s.a[x][y]:016x} != ${rv.rho[x][y]:016x}'
			}
		}

		s.pi()

		println('round ${round} verifying pi')
		for x in 0 .. 5 {
			for y in 0 .. 5 {
				assert s.a[x][y] == rv.pi[x][y], 'x ${x} y ${y} ${s.a[x][y]:016x} != ${rv.pi[x][y]:016x}'
			}
		}

		s.chi()

		println('round ${round} verifying chi')
		for x in 0 .. 5 {
			for y in 0 .. 5 {
				assert s.a[x][y] == rv.chi[x][y], 'x ${x} y ${y} ${s.a[x][y]:016x} != ${rv.chi[x][y]:016x}'
			}
		}

		s.iota(round)

		println('round ${round} verifying iota')
		for x in 0 .. 5 {
			for y in 0 .. 5 {
				assert s.a[x][y] == rv.iota[x][y], 'x ${x} y ${y} ${s.a[x][y]:016x} != ${rv.iota[x][y]:016x}'
			}
		}

		round += 1
	}

	// check the actual bytes in the state
	first_state_bytes := s.to_bytes()
	for i in 0 .. 200 {
		assert first_state_bytes[i] == first_state_as_bytes[i], 'examining state byte ${i} ${first_state_bytes[i]:02x} != ${first_state_as_bytes[i]:02x}'
	}

	println('verifying using previous state')

	round = 0
	for rv in second_24 {
		s.theta()

		println('round ${round} verifying theta')
		for x in 0 .. 5 {
			for y in 0 .. 5 {
				assert s.a[x][y] == rv.theta[x][y], 'x ${x} y ${y} ${s.a[x][y]:016x} != ${rv.theta[x][y]:016x}'
			}
		}

		s.rho()

		println('round ${round} verifying rho')
		for x in 0 .. 5 {
			for y in 0 .. 5 {
				assert s.a[x][y] == rv.rho[x][y], 'x ${x} y ${y} ${s.a[x][y]:016x} != ${rv.rho[x][y]:016x}'
			}
		}

		s.pi()

		println('round ${round} verifying pi')
		for x in 0 .. 5 {
			for y in 0 .. 5 {
				assert s.a[x][y] == rv.pi[x][y], 'x ${x} y ${y} ${s.a[x][y]:016x} != ${rv.pi[x][y]:016x}'
			}
		}

		s.chi()

		println('round ${round} verifying chi')
		for x in 0 .. 5 {
			for y in 0 .. 5 {
				assert s.a[x][y] == rv.chi[x][y], 'x ${x} y ${y} ${s.a[x][y]:016x} != ${rv.chi[x][y]:016x}'
			}
		}

		s.iota(round)

		println('round ${round} verifying iota')
		for x in 0 .. 5 {
			for y in 0 .. 5 {
				assert s.a[x][y] == rv.iota[x][y], 'x ${x} y ${y} ${s.a[x][y]:016x} != ${rv.iota[x][y]:016x}'
			}
		}

		round += 1
	}

	// check the actual bytes in the state
	second_state_bytes := s.to_bytes()
	for i in 0 .. 200 {
		assert second_state_bytes[i] == second_state_as_bytes[i], 'examining state byte ${i} ${second_state_bytes[i]:02x} != ${second_state_as_bytes[i]:02x}'
	}
}

fn test_to_from_bytes() {
	mut s := State{}

	// going from bytes to state as 5 x 5 u64 words to bytes
	// should give you the original byte array

	s.from_bytes(first_state_as_bytes)

	for x in 0 .. 5 {
		for y in 0 .. 5 {
			assert s.a[x][y] == first_state_as_words[x][y], 'x ${x} y ${y} ${s.a[x][y]:016x} != ${first_state_as_words[x][y]:016x}'
		}
	}

	mut result_bytes := s.to_bytes()

	for i in 0 .. 200 {
		assert result_bytes[i] == first_state_as_bytes[i], 'examining state byte ${i} ${result_bytes[i]:02x} != ${first_state_as_bytes[i]:02x}'
	}

	s.from_bytes(second_state_as_bytes)

	for x in 0 .. 5 {
		for y in 0 .. 5 {
			assert s.a[x][y] == second_state_as_words[x][y], 'x ${x} y ${y} ${s.a[x][y]:016x} != ${second_state_as_words[x][y]:016x}'
		}
	}

	result_bytes = s.to_bytes()

	for i in 0 .. 200 {
		assert result_bytes[i] == second_state_as_bytes[i], 'examining state byte ${i} ${result_bytes[i]:02x} != ${second_state_as_bytes[i]:02x}'
	}
}

struct RoundValues {
	theta [][]u64
	rho   [][]u64
	pi    [][]u64
	chi   [][]u64
	iota  [][]u64
}

// The following values are derived from data in file
// tests/TestVectors/KeccakF-1600-IntermediateValues.txt contained
// in https://github.com/XKCP/XKCP.git.

// intermediate state values starting from a zero initial state

const first_24 = [
	// round 0
	RoundValues{
		theta: [
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
		]
		rho:   [
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
		]
		pi:    [
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
		]
		chi:   [
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
		]
		iota:  [
			[u64(0x0000000000000001), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
		]
	},
	// round 1
	RoundValues{
		theta: [
			[u64(0x0000000000000001), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000001), 0x0000000000000001, 0x0000000000000001, 0x0000000000000001,
				0x0000000000000001],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000002), 0x0000000000000002, 0x0000000000000002, 0x0000000000000002,
				0x0000000000000002],
		]
		rho:   [
			[u64(0x0000000000000001), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000002), 0x0000100000000000, 0x0000000000000400, 0x0000200000000000,
				0x0000000000000004],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000010000000), 0x0000000000200000, 0x0000010000000000, 0x0000000000000200,
				0x0000000000008000],
		]
		pi:    [
			[u64(0x0000000000000001), 0x0000000000000000, 0x0000000000000002, 0x0000000010000000,
				0x0000000000000000],
			[u64(0x0000100000000000), 0x0000000000200000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000000000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000400,
				0x0000010000000000],
			[u64(0x0000000000000000), 0x0000200000000000, 0x0000000000000200, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000008000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000004],
		]
		chi:   [
			[u64(0x0000000000000001), 0x0000000000000000, 0x0000000000000002, 0x0000000010000400,
				0x0000010000000000],
			[u64(0x0000100000000000), 0x0000200000200000, 0x0000000000000200, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000008000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000400,
				0x0000010000000004],
			[u64(0x0000000000000001), 0x0000200000000000, 0x0000000000000202, 0x0000000010000000,
				0x0000000000000000],
			[u64(0x0000100000008000), 0x0000000000200000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000004],
		]
		iota:  [
			[u64(0x0000000000008083), 0x0000000000000000, 0x0000000000000002, 0x0000000010000400,
				0x0000010000000000],
			[u64(0x0000100000000000), 0x0000200000200000, 0x0000000000000200, 0x0000000000000000,
				0x0000000000000000],
			[u64(0x0000000000008000), 0x0000000000000000, 0x0000000000000000, 0x0000000000000400,
				0x0000010000000004],
			[u64(0x0000000000000001), 0x0000200000000000, 0x0000000000000202, 0x0000000010000000,
				0x0000000000000000],
			[u64(0x0000100000008000), 0x0000000000200000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000004],
		]
	},
	// round 2
	RoundValues{
		theta: [
			[u64(0x0000700000600487), 0x0000700000608404, 0x0000700000608406, 0x0000700010608004,
				0x0000710000608404],
			[u64(0x0000130010018C89), 0x0000230010218C89, 0x0000030010018E89, 0x0000030010018C89,
				0x0000030010018C89],
			[u64(0x0000700020208606), 0x0000700020200606, 0x0000700020200606, 0x0000700020200206,
				0x0000710020200602],
			[u64(0x000021000041840D), 0x000001000041840C, 0x000021000041860E, 0x000021001041840C,
				0x000021000041840C],
			[u64(0x0000320030018B01), 0x0000220030210B01, 0x0000220030010B01, 0x0000220030010B01,
				0x0000220030010B05],
		]
		rho:   [
			[u64(0x0000700000600487), 0x0608404000070000, 0x0003800003042030, 0xC100080000E00020,
				0xC400018210100001],
			[u64(0x0000260020031912), 0x18C8900002300102, 0x000C0040063A2400, 0x3191200000600200,
				0x00000C0040063224],
			[u64(0x80001C0008082181), 0x001C000808018180, 0x0030300003800101, 0x3800101001030000,
				0x40000E20040400C0],
			[u64(0x00041840D0000210), 0x06000000800020C2, 0x0000830C1C000042, 0x2002083081800004,
				0x0C00002100004184],
			[u64(0x01800C5808000190), 0x20030210B0100002, 0x0085808000110018, 0x00220030010B0100,
				0x08800C0042C14000],
		]
		pi:    [
			[u64(0x0000700000600487), 0x00041840D0000210, 0x0000260020031912, 0x01800C5808000190,
				0x80001C0008082181],
			[u64(0x18C8900002300102), 0x20030210B0100002, 0x001C000808018180, 0x0608404000070000,
				0x06000000800020C2],
			[u64(0x0030300003800101), 0x0003800003042030, 0x0000830C1C000042, 0x000C0040063A2400,
				0x0085808000110018],
			[u64(0x2002083081800004), 0x3191200000600200, 0x00220030010B0100, 0x3800101001030000,
				0xC100080000E00020],
			[u64(0x08800C0042C14000), 0x40000E20040400C0, 0xC400018210100001, 0x0C00002100004184,
				0x00000C0040063224],
		]
		chi:   [
			[u64(0x0030500001E00486), 0x00049840D3042220, 0x0000A50434031950, 0x01840C580E382590,
				0x80859C8008192199],
			[u64(0x38CA983082300106), 0x11932210B0700202, 0x003E0038090A8080, 0x3E08505001060000,
				0xC700080080E020E2],
			[u64(0x08B0340041C14101), 0x40038E20070020F0, 0xC400828E0C100043, 0x040C0061063A6584,
				0x008584804017321C],
			[u64(0x2002783081A00483), 0x31953040D0600010, 0x0022263021081812, 0x39801C4809030010,
				0x4100180008E801A1],
			[u64(0x10488C0040D14100), 0x60030C30241400C2, 0xC41C018A18108081, 0x0A08402100074184,
				0x06000C00C0063266],
		]
		iota:  [
			[u64(0x8030500001E0840C), 0x00049840D3042220, 0x0000A50434031950, 0x01840C580E382590,
				0x80859C8008192199],
			[u64(0x38CA983082300106), 0x11932210B0700202, 0x003E0038090A8080, 0x3E08505001060000,
				0xC700080080E020E2],
			[u64(0x08B0340041C14101), 0x40038E20070020F0, 0xC400828E0C100043, 0x040C0061063A6584,
				0x008584804017321C],
			[u64(0x2002783081A00483), 0x31953040D0600010, 0x0022263021081812, 0x39801C4809030010,
				0x4100180008E801A1],
			[u64(0x10488C0040D14100), 0x60030C30241400C2, 0xC41C018A18108081, 0x0A08402100074184,
				0x06000C00C0063266],
		]
	},
	// round 3
	RoundValues{
		theta: [
			[u64(0x98B0590AC86D7060), 0x1884914A1A89D64C, 0x1880AC0EFD8EED3C, 0x19040552C7B5D1FC,
				0x9805958AC194D5F5],
			[u64(0x298A1D327B0ED626), 0x00D3A712494ED522, 0x117E853AF03457A0, 0x2F48D552F838D720,
				0xD6408D0279DEF7C2],
			[u64(0x0AB50258192BD807), 0x4206B8785FEAB9F6, 0xC605B4D654FA9945, 0x060936395ED0FC82,
				0x0280B2D818FDAB1A],
			[u64(0xD8875F48F4F557EA), 0xC9101738A5355379, 0xF8A70148545D4B7B, 0xC1053B307C565379,
				0xB9853F787DBD52C8],
			[u64(0x7B161D31F07F2ADA), 0x0B5D9D0194BA6B18, 0xAF4290BBA8BEEB5B, 0x6156D110B0A92A5E,
				0x6D5E9D3170A859BC],
		]
		rho:   [
			[u64(0x98B0590AC86D7060), 0xA89D64C1884914A1, 0xC4056077EC7769E0, 0x6BA3F832080AA58F,
				0x562B065357D66016],
			[u64(0x53143A64F61DAC4C), 0xED52200D3A712494, 0xFA14EBC0D15E8045, 0x1AE405E91AAA5F07,
				0x59023409E77BDF0B],
			[u64(0xC2AD4096064AF601), 0x81AE1E17FAAE7D90, 0xD4CA2E302DA6B2A7, 0x9B1CAF687E410304,
				0x4050165B031FB563],
			[u64(0x8F4F557EAD8875F4), 0xBCE4880B9C529AA9, 0x90A8BA96F7F14E02, 0x660F8ACA6F3820A7,
				0xC8B9853F787DBD52],
			[u64(0x8F83F956D3D8B0E9), 0xD0194BA6B180B5D9, 0x5F75ADD7A1485DD4, 0x56D110B0A92A5E61,
				0xA74C5C2A166F1B57],
		]
		pi:    [
			[u64(0x98B0590AC86D7060), 0x8F4F557EAD8875F4, 0x53143A64F61DAC4C, 0x8F83F956D3D8B0E9,
				0xC2AD4096064AF601],
			[u64(0xED52200D3A712494), 0xD0194BA6B180B5D9, 0x81AE1E17FAAE7D90, 0xA89D64C1884914A1,
				0xBCE4880B9C529AA9],
			[u64(0xD4CA2E302DA6B2A7), 0xC4056077EC7769E0, 0x90A8BA96F7F14E02, 0xFA14EBC0D15E8045,
				0x5F75ADD7A1485DD4],
			[u64(0x660F8ACA6F3820A7), 0x1AE405E91AAA5F07, 0x56D110B0A92A5E61, 0x9B1CAF687E410304,
				0x6BA3F832080AA58F],
			[u64(0xA74C5C2A166F1B57), 0x4050165B031FB563, 0x562B065357D66016, 0xC8B9853F787DBD52,
				0x59023409E77BDF0B],
		]
		chi:   [
			[u64(0x8838573ACDEBE243), 0x8B4B752FE1FF3DD4, 0x43149AE4F34CAE4E, 0xDD83725682CE30AD,
				0x81BC65422742B355],
			[u64(0xCF57A0C778692494), 0xCAF94E2EA308A3DE, 0xC7FF1E37F2A46DF1, 0xA99560E9A64817A1,
				0x9C66D82B94503AA2],
			[u64(0x558A7A103DE1A9F7), 0x84157265ED62C980, 0x9082BCD5A1256E14, 0xBAB5EBD7D1623C17,
				0x4F75A9DE463907D4],
			[u64(0x7EBF8BCAA7384087), 0x95EB44CDB62A1F93, 0x57C528940923D229, 0x9C1ED728FDC103AD,
				0xE90EB8A4080A858F],
			[u64(0xC20E7C2F247F1FC3), 0x10401CDB131F356A, 0xD68102405F743186, 0xE8A581BE707CB952,
				0x6542BC007F6BD7A3],
		]
		iota:  [
			[u64(0x0838573A4DEB6243), 0x8B4B752FE1FF3DD4, 0x43149AE4F34CAE4E, 0xDD83725682CE30AD,
				0x81BC65422742B355],
			[u64(0xCF57A0C778692494), 0xCAF94E2EA308A3DE, 0xC7FF1E37F2A46DF1, 0xA99560E9A64817A1,
				0x9C66D82B94503AA2],
			[u64(0x558A7A103DE1A9F7), 0x84157265ED62C980, 0x9082BCD5A1256E14, 0xBAB5EBD7D1623C17,
				0x4F75A9DE463907D4],
			[u64(0x7EBF8BCAA7384087), 0x95EB44CDB62A1F93, 0x57C528940923D229, 0x9C1ED728FDC103AD,
				0xE90EB8A4080A858F],
			[u64(0xC20E7C2F247F1FC3), 0x10401CDB131F356A, 0xD68102405F743186, 0xE8A581BE707CB952,
				0x6542BC007F6BD7A3],
		]
	},
	// round 4
	RoundValues{
		theta: [
			[u64(0x6E5498081D5398EC), 0xED27BA1DB147C77B, 0x257855D6A3F454E1, 0xBBEFBD64D276CA02,
				0xE7D0AA7077FA49FA],
			[u64(0x3AB4E2714F473DF4), 0x3F1A0C989426BABE, 0x321C5C81C58A7491, 0x5C76225F91660EC1,
				0x69859A9DA37E23C2],
			[u64(0x312B2233FDC87870), 0xE0B42A462D4B1807, 0xF423E4F6610CBF93, 0xDE14B3F4114BED90,
				0x2BD4F1FD8610D653],
			[u64(0xD832C3778FC39E9A), 0x33660C709ED1C18E, 0xF148602921D80C34, 0x3A939F95D53ADDB0,
				0x4F83F01920F15B92],
			[u64(0x333EABFB3C2DF09F), 0xE170CB0F0B4DDA36, 0x27B1D5944726DEDA, 0x1995566A682E560E,
				0x94726BD4673938FF],
		]
		rho:   [
			[u64(0x6E5498081D5398EC), 0x147C77BED27BA1DB, 0x2BC2AEB51FA2A709, 0xED940577DF7AC9A4,
				0xA9C1DFE927EB9F42],
			[u64(0x7569C4E29E8E7BE8), 0x6BABE3F1A0C98942, 0x7172071629D244C8, 0xC1D82B8EC44BF22C,
				0xA6166A768DF88F09],
			[u64(0x0C4AC88CFF721E1C), 0x2D0A918B52C601F8, 0x65FC9FA11F27B308, 0x59FA08A5F6C86F0A,
				0x657A9E3FB0C21ACA],
			[u64(0x78FC39E9AD832C37), 0xC719B306384F68E0, 0x5243B01869E290C0, 0xF2BAA75BB6075273,
				0x924F83F01920F15B],
			[u64(0xD9E16F84F999F55F), 0xB0F0B4DDA36E170C, 0x936F6D13D8EACA23, 0x95566A682E560E19,
				0x9AF519CE4E3FE51C],
		]
		pi:    [
			[u64(0x6E5498081D5398EC), 0x78FC39E9AD832C37, 0x7569C4E29E8E7BE8, 0xD9E16F84F999F55F,
				0x0C4AC88CFF721E1C],
			[u64(0x6BABE3F1A0C98942), 0xB0F0B4DDA36E170C, 0x2D0A918B52C601F8, 0x147C77BED27BA1DB,
				0xC719B306384F68E0],
			[u64(0x65FC9FA11F27B308), 0x2BC2AEB51FA2A709, 0x5243B01869E290C0, 0x7172071629D244C8,
				0x936F6D13D8EACA23],
			[u64(0xF2BAA75BB6075273), 0xC1D82B8EC44BF22C, 0x95566A682E560E19, 0x59FA08A5F6C86F0A,
				0xED940577DF7AC9A4],
			[u64(0x9AF519CE4E3FE51C), 0x657A9E3FB0C21ACA, 0xA9C1DFE927EB9F42, 0x924F83F01920F15B,
				0xA6166A768DF88F09],
		]
		chi:   [
			[u64(0x6A0084080275AAE4), 0x73FE33C9B1038C36, 0x2728E4F2B7AEEBE8, 0xB8E36F84D019B15F,
				0x1C2C849D3FD29C1F],
			[u64(0xF9A9C3AB00C9C931), 0x70E8B5D763274728, 0xA81EDBEB54D20FE1, 0x1CF47F1F04738AD9,
				0xAB89B3623F5F6964],
			[u64(0x6DB98725571F1604), 0x0FE03A842F22AFCB, 0x7AC22599684B0182, 0xF377844620F2D499,
				0x916D0713D86ACC2A],
			[u64(0x96BA275BA7474A93), 0xD95C0A4EC94AD619, 0xC17E6A6AB6526EB1, 0x105A64A116516B0E,
				0xE5DC85FFAD78D9B0],
			[u64(0x9B5E7A3FEEB7E41E), 0xE57A1A2BB2AE09C2, 0xA1C3CEE067AB9F52, 0x965393CA1B42F1DB,
				0x650759748DF5EFE9],
		]
		iota:  [
			[u64(0x6A00840802752A6F), 0x73FE33C9B1038C36, 0x2728E4F2B7AEEBE8, 0xB8E36F84D019B15F,
				0x1C2C849D3FD29C1F],
			[u64(0xF9A9C3AB00C9C931), 0x70E8B5D763274728, 0xA81EDBEB54D20FE1, 0x1CF47F1F04738AD9,
				0xAB89B3623F5F6964],
			[u64(0x6DB98725571F1604), 0x0FE03A842F22AFCB, 0x7AC22599684B0182, 0xF377844620F2D499,
				0x916D0713D86ACC2A],
			[u64(0x96BA275BA7474A93), 0xD95C0A4EC94AD619, 0xC17E6A6AB6526EB1, 0x105A64A116516B0E,
				0xE5DC85FFAD78D9B0],
			[u64(0x9B5E7A3FEEB7E41E), 0xE57A1A2BB2AE09C2, 0xA1C3CEE067AB9F52, 0x965393CA1B42F1DB,
				0x650759748DF5EFE9],
		]
	},
	// round 5
	RoundValues{
		theta: [
			[u64(0x6AF72396B7508258), 0x7309945704262401, 0x27DF436C028B43DF, 0xB814C81A653C1968,
				0x1CDB23038AF73428],
			[u64(0x96B24D5A3A07E83C), 0x1FF33B2659E96625, 0xC705551A6E1C2EEC, 0x73EFF1EE3EBDABD4,
				0xC4923D9305914869],
			[u64(0x0DA6AA8D9DE3F54B), 0x6FFF172CE5DE4C84, 0x1ADD0831A2B7E2CD, 0x9368A9EEEA0E37D6,
				0xF1722ABB12962F65],
			[u64(0xB55DF4A315A33315), 0xFABBD9B67BAEAF9F, 0xE299B99204B61737, 0x33BDB759A4B51288,
				0xC63B56071F9CA036],
			[u64(0xD473AC4B5BE76578), 0xAA57CC5F07FE88A4, 0xEEEE1894D2FB1E34, 0xD97E45BEAE1270BD,
				0x2A2A8F0038A56E8F],
		]
		rho:   [
			[u64(0x6AF72396B7508258), 0x4262401730994570, 0x3EFA1B60145A1EF9, 0x7832D170299034CA,
				0x8C0E2BDCD0A0736C],
			[u64(0x2D649AB4740FD079), 0x966251FF33B2659E, 0x155469B870BBB31C, 0xB57A8E7DFE3DC7D7,
				0x1248F64C164521A7],
			[u64(0xC369AAA36778FD52), 0xFFC5CB397793211B, 0xBF1668D6E8418D15, 0x54F775071BEB49B4,
				0xBE2E45576252C5EC],
			[u64(0x315A33315B55DF4A), 0xCFFD5DECDB3DD757, 0x24096C2E6FC53373, 0xEB3496A2510677B6,
				0x36C63B56071F9CA0],
			[u64(0x5ADF3B2BC6A39D62), 0xC5F07FE88A4AA57C, 0x7D8F1A77770C4A69, 0x7E45BEAE1270BDD9,
				0xA3C00E295BA3CA8A],
		]
		pi:    [
			[u64(0x6AF72396B7508258), 0x315A33315B55DF4A, 0x2D649AB4740FD079, 0x5ADF3B2BC6A39D62,
				0xC369AAA36778FD52],
			[u64(0x966251FF33B2659E), 0xC5F07FE88A4AA57C, 0xFFC5CB397793211B, 0x4262401730994570,
				0xCFFD5DECDB3DD757],
			[u64(0xBF1668D6E8418D15), 0x3EFA1B60145A1EF9, 0x24096C2E6FC53373, 0x155469B870BBB31C,
				0x7D8F1A77770C4A69],
			[u64(0xEB3496A2510677B6), 0xB57A8E7DFE3DC7D7, 0x7E45BEAE1270BDD9, 0x54F775071BEB49B4,
				0x7832D170299034CA],
			[u64(0xA3C00E295BA3CA8A), 0xBE2E45576252C5EC, 0x8C0E2BDCD0A0736C, 0x36C63B56071F9CA0,
				0x1248F64C164521A7],
		]
		chi:   [
			[u64(0x43E30B967F110A59), 0x0B5033314F45C5CB, 0x2D6CBEB27C4BC219, 0x4FCB128386812F6E,
				0xF36BA8B04378F57A],
			[u64(0xD642C7DF22B4173C), 0x44F0FBF5606F647A, 0xA58159B967A3AD93, 0x02C154103BD90DD0,
				0xCFCD9CECD3ADE3D5],
			[u64(0xBFD660DFE2E0051D), 0x34FE5A6214181ED1, 0xA4036D7EAF457157, 0x375463E874AF271C,
				0x7FC73C7B61494B4C],
			[u64(0xA303B734F55677E6), 0xB42ABC5DE738DDD5, 0x5F252E8E367F3DC8, 0x1CEE752EDB4B48F6,
				0xB913D9D348A8E89A],
			[u64(0x37C05E405B01AF0C), 0x7A8E099FE258E5D8, 0x5E8F6AD5D330526E, 0x36E67B423707DCB0,
				0x1EDCA3008E4023A2],
		]
		iota:  [
			[u64(0x43E30B96FF110A58), 0x0B5033314F45C5CB, 0x2D6CBEB27C4BC219, 0x4FCB128386812F6E,
				0xF36BA8B04378F57A],
			[u64(0xD642C7DF22B4173C), 0x44F0FBF5606F647A, 0xA58159B967A3AD93, 0x02C154103BD90DD0,
				0xCFCD9CECD3ADE3D5],
			[u64(0xBFD660DFE2E0051D), 0x34FE5A6214181ED1, 0xA4036D7EAF457157, 0x375463E874AF271C,
				0x7FC73C7B61494B4C],
			[u64(0xA303B734F55677E6), 0xB42ABC5DE738DDD5, 0x5F252E8E367F3DC8, 0x1CEE752EDB4B48F6,
				0xB913D9D348A8E89A],
			[u64(0x37C05E405B01AF0C), 0x7A8E099FE258E5D8, 0x5E8F6AD5D330526E, 0x36E67B423707DCB0,
				0x1EDCA3008E4023A2],
		]
	},
	// round 6
	RoundValues{
		theta: [
			[u64(0x8C67B401B6278C51), 0xC4D48CA6067343C2, 0xE2E80125357D4410, 0x804FAD14CFB7A967,
				0x3CEF17270A4E7373],
			[u64(0xC04DEB59B3E4CD34), 0x52FFD773F13FBE72, 0xB38E753FF6F3779B, 0x14CE7896AA89D7D8,
				0xD9C2B06A42FD39DD],
			[u64(0x9E0ADF8540085AE2), 0x1522E538B6F0412E, 0x85DFD2240DAD2EA8, 0x1688DCB2D64778E3,
				0x5E1B8321C3A114B3],
			[u64(0xB34C75F51F50BE7D), 0xA4657E9C0D3E144E, 0x4F6AEC4FDC79F453, 0x0CA1B7EF314D816D,
				0xA95C1B12A2AE2101],
			[u64(0x68CFAF16FF3E37A6), 0x2581F8C946677D72, 0x01809B83770FCAC4, 0x69E98A149338441A,
				0x41D352562A7FBB08],
		]
		rho:   [
			[u64(0x8C67B401B6278C51), 0x67343C2C4D48CA60, 0x17400929ABEA2087, 0x6F52CF009F5A299F,
				0x5C9C2939CDCCF3BC],
			[u64(0x809BD6B367C99A69), 0xFBE7252FFD773F13, 0x39D4FFDBCDDE6ECE, 0x3AFB0299CF12D551,
				0x670AC1A90BF4E777],
			[u64(0xA782B7E1500216B8), 0x48B94E2DBC104B85, 0x6975442EFE91206D, 0x6E596B23BC718B44,
				0x6BC3706438742296],
			[u64(0x51F50BE7DB34C75F), 0x275232BF4E069F0A, 0x9FB8F3E8A69ED5D8, 0xFDE629B02DA19436,
				0x01A95C1B12A2AE21],
			[u64(0xB7F9F1BD33467D78), 0x8C946677D722581F, 0x87E56200C04DC1BB, 0xE98A149338441A69,
				0xD4958A9FEEC21074],
		]
		pi:    [
			[u64(0x8C67B401B6278C51), 0x51F50BE7DB34C75F, 0x809BD6B367C99A69, 0xB7F9F1BD33467D78,
				0xA782B7E1500216B8],
			[u64(0xFBE7252FFD773F13), 0x8C946677D722581F, 0x48B94E2DBC104B85, 0x67343C2C4D48CA60,
				0x275232BF4E069F0A],
			[u64(0x6975442EFE91206D), 0x17400929ABEA2087, 0x9FB8F3E8A69ED5D8, 0x39D4FFDBCDDE6ECE,
				0x87E56200C04DC1BB],
			[u64(0xFDE629B02DA19436), 0x3AFB0299CF12D551, 0xE98A149338441A69, 0x6E596B23BC718B44,
				0x6F52CF009F5A299F],
			[u64(0xD4958A9FEEC21074), 0x6BC3706438742296, 0x5C9C2939CDCCF3BC, 0x01A95C1B12A2AE21,
				0x670AC1A90BF4E777],
		]
		chi:   [
			[u64(0x8C77F401B4A78C3D), 0x42B502EFF3FCE7DF, 0x179B677365470E31, 0xAF39326EB3D059F6,
				0x2727F7E1D04B5609],
			[u64(0x6F650CBFFC57AB01), 0xA42F64E793328D4F, 0x28BB4A3EA45041A4, 0x213D3C0C7D694B60,
				0x4F40BFBF5114B70E],
			[u64(0x6964C6213CD3202D), 0x5640794D9B8E0201, 0x8BACDAC06316344C, 0x3874EBC3CF5C4AEF,
				0x87ED62A9C0E907DB],
			[u64(0xF5841DB03D841837), 0x2ACF091A0C121018, 0x6989C2111A451228, 0xD809CA879D35DA1C,
				0xEFD2F940CF583917],
			[u64(0xA7158BB1A7922376), 0xE7C314743C763A96, 0x14BC213555DCB238, 0x41AD501B5EAA2C21,
				0x675AC1B705F06E75],
		]
		iota:  [
			[u64(0x0C77F40134A70CBC), 0x42B502EFF3FCE7DF, 0x179B677365470E31, 0xAF39326EB3D059F6,
				0x2727F7E1D04B5609],
			[u64(0x6F650CBFFC57AB01), 0xA42F64E793328D4F, 0x28BB4A3EA45041A4, 0x213D3C0C7D694B60,
				0x4F40BFBF5114B70E],
			[u64(0x6964C6213CD3202D), 0x5640794D9B8E0201, 0x8BACDAC06316344C, 0x3874EBC3CF5C4AEF,
				0x87ED62A9C0E907DB],
			[u64(0xF5841DB03D841837), 0x2ACF091A0C121018, 0x6989C2111A451228, 0xD809CA879D35DA1C,
				0xEFD2F940CF583917],
			[u64(0xA7158BB1A7922376), 0xE7C314743C763A96, 0x14BC213555DCB238, 0x41AD501B5EAA2C21,
				0x675AC1B705F06E75],
		]
	},
	// round 7
	RoundValues{
		theta: [
			[u64(0x65F398F66F54D239), 0x2B316E18A80F395A, 0x7E1F0B843EB4D0B4, 0xC6BD5E99E8238773,
				0x4EA39B168BB8888C],
			[u64(0xA8018120AA2CF704), 0x634BE978C549D14A, 0xEFDFC7A1F22B1DA1, 0xE659B1932B121765,
				0x88243220076FEB0B],
			[u64(0xE6DBAD0C28E649B0), 0xD9FF12608FBB6B9C, 0x0413B1ED77235DD1, 0xB7CB80EEDB692372,
				0x08520984D4DC6E46],
			[u64(0x1BAFAFCFDCBF907B), 0xC4E4BB65ED299854, 0x87A2706EFB7E9A64, 0x362278F87C0E5250,
				0x01F94B3F2E63B15B],
			[u64(0x8482C6E85D230F21), 0xC454592DC6C716C1, 0x372B6C6CAF6D9E6F, 0x623A1D42A41B0076,
				0x44CD8CEEFF414222],
		]
		rho:   [
			[u64(0x65F398F66F54D239), 0x80F395A2B316E18A, 0xF0F85C21F5A685A3, 0x470EE78D7ABD33D0,
				0x6C5A2EE222313A8E],
			[u64(0x500302415459EE09), 0x9D14A634BE978C54, 0x7F1E87C8AC7687BF, 0x42ECBCCB36326562,
				0x2090C8801DBFAC2E],
			[u64(0x39B6EB430A39926C), 0x7FC49823EEDAE736, 0x1AEE88209D8F6BB9, 0xC0776DB491B95BE5,
				0xC10A41309A9B8DC8],
			[u64(0xFDCBF907B1BAFAFC), 0x2A62725DB2F694CC, 0xDDF6FD34C90F44E0, 0x1F0F81CA4A06C44F,
				0x5B01F94B3F2E63B1],
			[u64(0x42E918790C241637), 0x92DC6C716C1C4545, 0xB6CF379B95B63657, 0x3A1D42A41B007662,
				0x633BBFD050889133],
		]
		pi:    [
			[u64(0x65F398F66F54D239), 0xFDCBF907B1BAFAFC, 0x500302415459EE09, 0x42E918790C241637,
				0x39B6EB430A39926C],
			[u64(0x9D14A634BE978C54), 0x92DC6C716C1C4545, 0x7FC49823EEDAE736, 0x80F395A2B316E18A,
				0x2A62725DB2F694CC],
			[u64(0x1AEE88209D8F6BB9), 0xF0F85C21F5A685A3, 0xDDF6FD34C90F44E0, 0x7F1E87C8AC7687BF,
				0xB6CF379B95B63657],
			[u64(0x1F0F81CA4A06C44F), 0x42ECBCCB36326562, 0x3A1D42A41B007662, 0xC0776DB491B95BE5,
				0x470EE78D7ABD33D0],
			[u64(0x633BBFD050889133), 0xC10A41309A9B8DC8, 0x6C5A2EE222313A8E, 0x5B01F94B3F2E63B1,
				0x2090C8801DBFAC2E],
		]
		chi:   [
			[u64(0x671990F66E5CB190), 0x9DEBE90720187A5E, 0xD0316755555CEEC9, 0x3DE51A3100441002,
				0xAD3BEEC10F39B07F],
			[u64(0x9815A7FEFC970812), 0x90D8CCBB6E0C2505, 0x5DCD9AA3FCDAD534, 0x0092FD96A29FB9CA,
				0x6B62B259D8FF954C],
			[u64(0x7ADEB6308D077A89), 0x71FA1D117D2F0D2B, 0x99B4D176E93E4C6C, 0x641E17838270A7AF,
				0x965F3F9B90B4BA79],
			[u64(0x1BCF81EC65528647), 0x7E2D04CC17121756, 0x2A1C42A54F48B263, 0xC09F6D8491B94FE3,
				0x5E28C4CE78BD2190],
			[u64(0xFB3F99D0C00B9D77), 0xC31E4540D69F88C9, 0x439EB6C088B33BB8, 0xDB137CC98C3C8239,
				0x22D0D89CAD79A8AE],
		]
		iota:  [
			[u64(0xE71990F66E5C3199), 0x9DEBE90720187A5E, 0xD0316755555CEEC9, 0x3DE51A3100441002,
				0xAD3BEEC10F39B07F],
			[u64(0x9815A7FEFC970812), 0x90D8CCBB6E0C2505, 0x5DCD9AA3FCDAD534, 0x0092FD96A29FB9CA,
				0x6B62B259D8FF954C],
			[u64(0x7ADEB6308D077A89), 0x71FA1D117D2F0D2B, 0x99B4D176E93E4C6C, 0x641E17838270A7AF,
				0x965F3F9B90B4BA79],
			[u64(0x1BCF81EC65528647), 0x7E2D04CC17121756, 0x2A1C42A54F48B263, 0xC09F6D8491B94FE3,
				0x5E28C4CE78BD2190],
			[u64(0xFB3F99D0C00B9D77), 0xC31E4540D69F88C9, 0x439EB6C088B33BB8, 0xDB137CC98C3C8239,
				0x22D0D89CAD79A8AE],
		]
	},
	// round 8
	RoundValues{
		theta: [
			[u64(0x188422A1F97D9C42), 0x62765B50B739D785, 0x2FACD502C27D4312, 0xC278A8669765BDD9,
				0x52A65C9698181DA4],
			[u64(0x63AAE934FF564151), 0x6B6782716DCD6C46, 0xA672D469FF1B9C77, 0xFB2DB35CA15EF089,
				0x90DDFC93DB3EDC0F],
			[u64(0xE6BCD586313E342F), 0xED987EA7C116438D, 0x05D6B2C0550702CA, 0xF87C74353E49E909,
				0x0A3D5C2D2C8DF4DF],
			[u64(0x7FE74FA81044A97C), 0x1A05CA886204386D, 0x4E348CE13A5E9D58, 0xA4B7A3C0E4AF60D8,
				0x3A000A8A0DAB0EAB],
			[u64(0x5E4D23B73CCDDA90), 0x666CFF272A59CF2E, 0xE6EC0CA774757C5F, 0x7E61C6AE70FAC5DE,
				0x87A262FB51BFEF49],
		]
		rho:   [
			[u64(0x188422A1F97D9C42), 0x739D78562765B50B, 0x7D66A81613EA1891, 0xCB7BB384F150CD2E,
				0x725A606076914A99],
			[u64(0xC755D269FEAC82A2), 0xD6C466B6782716DC, 0xCB51A7FC6E71DE99, 0xDE113F65B66B942B,
				0x4377F24F6CFB703E],
			[u64(0xF9AF35618C4F8D0B), 0x661FA9F04590E37B, 0x3816502EB59602A8, 0x3A1A9F24F484FC3E,
				0xE147AB85A591BE9B],
			[u64(0x81044A97C7FE74FA), 0x368D02E54431021C, 0xC274BD3AB09C6919, 0x781C95EC1B1496F4,
				0xAB3A000A8A0DAB0E],
			[u64(0xB9E66ED482F2691D), 0xF272A59CF2E666CF, 0x3ABE2FF3760653BA, 0x61C6AE70FAC5DE7E,
				0x98BED46FFBD261E8],
		]
		pi:    [
			[u64(0x188422A1F97D9C42), 0x81044A97C7FE74FA, 0xC755D269FEAC82A2, 0xB9E66ED482F2691D,
				0xF9AF35618C4F8D0B],
			[u64(0xD6C466B6782716DC), 0xF272A59CF2E666CF, 0x661FA9F04590E37B, 0x739D78562765B50B,
				0x368D02E54431021C],
			[u64(0x3816502EB59602A8), 0x7D66A81613EA1891, 0xC274BD3AB09C6919, 0xCB51A7FC6E71DE99,
				0x3ABE2FF3760653BA],
			[u64(0x781C95EC1B1496F4), 0xDE113F65B66B942B, 0x61C6AE70FAC5DE7E, 0x3A1A9F24F484FC3E,
				0xCB7BB384F150CD2E],
			[u64(0x98BED46FFBD261E8), 0xE147AB85A591BE9B, 0x725A606076914A99, 0xAB3A000A8A0DAB0E,
				0x4377F24F6CFB703E],
		]
		chi:   [
			[u64(0x309632A97CED9C62), 0x8C004295C6F66CEA, 0x4735C6634EA08AA2, 0x31A6E97CCAE2238D,
				0xF19D1873BE49DCA9],
			[u64(0x96CCE37672278288), 0x7063B2FD56E7E2E5, 0x479DABB00FD1751D, 0x43976056B7E1952D,
				0xF7CC92E1C5618E18],
			[u64(0xB8B4102D555463A0), 0x5C202896127A3201, 0xD06CFD3AB48C6998, 0x4A71A7F66478DD99,
				0x3ABA6FB87AAD63AA],
			[u64(0x781CB76C1B390AF6), 0xDE117F77F405D44B, 0xE4C33C7972E95E5C, 0x2ADEF1F0F476BC2F,
				0x73F3B6A47154402F],
			[u64(0x5EFE9079FBD06374), 0x93350E8D9591BC9E, 0x525049F077812BC0, 0xE9231008AF083F0C,
				0x4577F0CB2CCB722A],
		]
		iota:  [
			[u64(0x309632A97CED9CE8), 0x8C004295C6F66CEA, 0x4735C6634EA08AA2, 0x31A6E97CCAE2238D,
				0xF19D1873BE49DCA9],
			[u64(0x96CCE37672278288), 0x7063B2FD56E7E2E5, 0x479DABB00FD1751D, 0x43976056B7E1952D,
				0xF7CC92E1C5618E18],
			[u64(0xB8B4102D555463A0), 0x5C202896127A3201, 0xD06CFD3AB48C6998, 0x4A71A7F66478DD99,
				0x3ABA6FB87AAD63AA],
			[u64(0x781CB76C1B390AF6), 0xDE117F77F405D44B, 0xE4C33C7972E95E5C, 0x2ADEF1F0F476BC2F,
				0x73F3B6A47154402F],
			[u64(0x5EFE9079FBD06374), 0x93350E8D9591BC9E, 0x525049F077812BC0, 0xE9231008AF083F0C,
				0x4577F0CB2CCB722A],
		]
	},
	// round 9
	RoundValues{
		theta: [
			[u64(0x298B147655CC396E), 0x951D644AEFD7C96C, 0x5E28E0BC67812F24, 0x28BBCFA3E3C3860B,
				0xE8803EAC9768792F],
			[u64(0x2532BFB928D80B18), 0xC39DEE320C186B75, 0xF463F77F552EFC8D, 0xF0693C99ED1E1CBD,
				0x4432CE2E9F9E0788],
			[u64(0x9A1A7ECD3D2B9427), 0x7E8E46767A05C586, 0xF2C293DADCF39E1F, 0x68DFC9160C072A1E,
				0x1814015812D2942D],
			[u64(0x5BB1D52CC249FEE4), 0xFDBC1D372D752059, 0xC76E5E39AB99AA4E, 0x097393B02D06483D,
				0x505ED4E4A824B43D],
			[u64(0x322DADEEE306149D), 0xFFE6331A8D47CB77, 0x3E8374676F575C29, 0x85F02D9FB7DE48E5,
				0x29A4CD5C341D05C3],
		]
		rho:   [
			[u64(0x298B147655CC396E), 0xFD7C96C951D644AE, 0xF14705E33C097922, 0x870C1651779F47C7,
				0xFAB25DA1E4BFA200],
			[u64(0x4A657F7251B01630), 0x86B75C39DEE320C1, 0x8FDDFD54BBF237D1, 0xC397BE0D27933DA3,
				0x10CB38BA7E781E21],
			[u64(0xE6869FB34F4AE509), 0xA3919D9E8171619F, 0x9CF0FF96149ED6E7, 0xE48B0603950F346F,
				0xA302802B025A5285],
			[u64(0xCC249FEE45BB1D52), 0x2CFEDE0E9B96BA90, 0x735733549D8EDCBC, 0x7605A0C907A12E72,
				0x3D505ED4E4A824B4],
			[u64(0x771830A4E9916D6F), 0x31A8D47CB77FFE63, 0xABAE149F41BA33B7, 0xF02D9FB7DE48E585,
				0x33570D074170CA69],
		]
		pi:    [
			[u64(0x298B147655CC396E), 0xCC249FEE45BB1D52, 0x4A657F7251B01630, 0x771830A4E9916D6F,
				0xE6869FB34F4AE509],
			[u64(0x86B75C39DEE320C1), 0x31A8D47CB77FFE63, 0xA3919D9E8171619F, 0xFD7C96C951D644AE,
				0x2CFEDE0E9B96BA90],
			[u64(0x9CF0FF96149ED6E7), 0xF14705E33C097922, 0x735733549D8EDCBC, 0x8FDDFD54BBF237D1,
				0xABAE149F41BA33B7],
			[u64(0x7605A0C907A12E72), 0xC397BE0D27933DA3, 0xF02D9FB7DE48E585, 0xE48B0603950F346F,
				0x870C1651779F47C7],
			[u64(0x33570D074170CA69), 0xA302802B025A5285, 0xFAB25DA1E4BFA200, 0x3D505ED4E4A824B4,
				0x10CB38BA7E781E21],
		]
		chi:   [
			[u64(0x31CBB7F055D0EF48), 0x0C639E6D4DBB1C52, 0x1A235D324D3E8A10, 0x759959B043B15E3E,
				0x65869F220F62E42E],
			[u64(0xE4B25C70DDC208D1), 0x33386E70B4EDFAE2, 0x23B9113DC331409E, 0x9D7E94CA55DB4480,
				0x28FEDC4EAD93FED0],
			[u64(0x9DA2F29054CE16EE), 0xD14705C13C413B26, 0x79C57354BD39DEBC, 0x968DA580DB523741,
				0xBB6D3C3549DA2B97],
			[u64(0x7E8DB0B9132D1F74), 0x8FB3A1C9623230F1, 0xF068BDE5CF48F1B5, 0xA68326239C1E7D24,
				0x61089150769DA6CF],
			[u64(0xB563450ECB53CAE8), 0x928AC03BB01EB0A4, 0x5B22DD2D64FEC38F, 0xB534D89DF4EE2434,
				0x18B378B6EEEC04B1],
		]
		iota:  [
			[u64(0x31CBB7F055D0EFC0), 0x0C639E6D4DBB1C52, 0x1A235D324D3E8A10, 0x759959B043B15E3E,
				0x65869F220F62E42E],
			[u64(0xE4B25C70DDC208D1), 0x33386E70B4EDFAE2, 0x23B9113DC331409E, 0x9D7E94CA55DB4480,
				0x28FEDC4EAD93FED0],
			[u64(0x9DA2F29054CE16EE), 0xD14705C13C413B26, 0x79C57354BD39DEBC, 0x968DA580DB523741,
				0xBB6D3C3549DA2B97],
			[u64(0x7E8DB0B9132D1F74), 0x8FB3A1C9623230F1, 0xF068BDE5CF48F1B5, 0xA68326239C1E7D24,
				0x61089150769DA6CF],
			[u64(0xB563450ECB53CAE8), 0x928AC03BB01EB0A4, 0x5B22DD2D64FEC38F, 0xB534D89DF4EE2434,
				0x18B378B6EEEC04B1],
		]
	},
	// round 10
	RoundValues{
		theta: [
			[u64(0x63E198B1F4CD677C), 0x5E49B12CECA694EE, 0x48097273EC2302AC, 0x27B376F1E2ACD682,
				0x37ACB063AE7F6C92],
			[u64(0xE2A6D52D4A391407), 0x352CE72D2316E634, 0x25AD986054CA5C48, 0x9B6A1D97C2205856,
				0x2EEA55133A68E206],
			[u64(0x51ABAEE5AF3015A4), 0x1D4E59B4C7BF386C, 0xB5CC2F2146C7DDF6, 0x5A84F9F520AC340B,
				0x77646040B22428DD],
			[u64(0xC4D45D6F5F70C25B), 0x35EA4C1F2E6FEDDE, 0x4A31503383152C9A, 0x1CDACBF5D043A00B,
				0xDB517C863AC07BE0],
			[u64(0x1C973A92AC8A4817), 0x3B7EBFA7D7C7325B, 0xF2D6A2B103274170, 0x1CC0A7019337A6CB,
				0xB147072A8935864E],
		]
		rho:   [
			[u64(0x63E198B1F4CD677C), 0xCA694EE5E49B12CE, 0x404B939F61181562, 0x59AD044F66EDE3C5,
				0xC18EB9FDB248DEB2],
			[u64(0xC54DAA5A9472280F), 0x6E634352CE72D231, 0xB661815329712096, 0x0B0AD36D43B2F844,
				0xBBA9544CE9A38818],
			[u64(0x146AEBB96BCC0569), 0x53966D31EFCE1B07, 0x3EEFB5AE61790A36, 0x7CFA90561A05AD42,
				0xAEEC8C081644851B],
			[u64(0xF5F70C25BC4D45D6), 0xEF1AF5260F9737F6, 0x67062A59349462A0, 0x7EBA087401639B59,
				0xE0DB517C863AC07B],
			[u64(0x95645240B8E4B9D4), 0xFA7D7C7325B3B7EB, 0x93A0B8796B515881, 0xC0A7019337A6CB1C,
				0xC1CAA24D6193AC51],
		]
		pi:    [
			[u64(0x63E198B1F4CD677C), 0xF5F70C25BC4D45D6, 0xC54DAA5A9472280F, 0x95645240B8E4B9D4,
				0x146AEBB96BCC0569],
			[u64(0x6E634352CE72D231), 0xFA7D7C7325B3B7EB, 0x53966D31EFCE1B07, 0xCA694EE5E49B12CE,
				0xEF1AF5260F9737F6],
			[u64(0x3EEFB5AE61790A36), 0x404B939F61181562, 0x67062A59349462A0, 0xB661815329712096,
				0x93A0B8796B515881],
			[u64(0x7EBA087401639B59), 0x0B0AD36D43B2F844, 0xC0A7019337A6CB1C, 0x7CFA90561A05AD42,
				0x59AD044F66EDE3C5],
			[u64(0xC1CAA24D6193AC51), 0xAEEC8C081644851B, 0xC18EB9FDB248DEB2, 0xE0DB517C863AC07B,
				0xBBA9544CE9A38818],
		]
		chi:   [
			[u64(0x736D2C1DD5C46F7A), 0xF5F58FA9FC4545D6, 0xE14DA812846248AF, 0xA164D352B18499C4,
				0x04CAE3E00B8C4D68],
			[u64(0x2E734B02CE704378), 0xF17D3C1327115FEF, 0xD3376CB3ECEC921B, 0x82F35EE1F69F9F8E,
				0xA717F1200B3B94B2],
			[u64(0xBFAF17A701E92E36), 0xE4AF9F9F755C1079, 0x660E9235B4DC7602, 0x3660C07BAD4B60AF,
				0x31A0E879E2535099],
			[u64(0x5C9B10C4952FD875), 0x5A19D348EBBBB880, 0xC4E603913394EB11, 0x69DE925622C194C6,
				0x5DEFAFFE64A1E6A4],
			[u64(0xCDC8E10F6BA13C50), 0xA4E4FC5A17F63732, 0xD31CFCDCD9C4CDB2, 0xAAD25DD9C221C271,
				0x50B9404AEDB0BA8E],
		]
		iota:  [
			[u64(0x736D2C1D55C4EF73), 0xF5F58FA9FC4545D6, 0xE14DA812846248AF, 0xA164D352B18499C4,
				0x04CAE3E00B8C4D68],
			[u64(0x2E734B02CE704378), 0xF17D3C1327115FEF, 0xD3376CB3ECEC921B, 0x82F35EE1F69F9F8E,
				0xA717F1200B3B94B2],
			[u64(0xBFAF17A701E92E36), 0xE4AF9F9F755C1079, 0x660E9235B4DC7602, 0x3660C07BAD4B60AF,
				0x31A0E879E2535099],
			[u64(0x5C9B10C4952FD875), 0x5A19D348EBBBB880, 0xC4E603913394EB11, 0x69DE925622C194C6,
				0x5DEFAFFE64A1E6A4],
			[u64(0xCDC8E10F6BA13C50), 0xA4E4FC5A17F63732, 0xD31CFCDCD9C4CDB2, 0xAAD25DD9C221C271,
				0x50B9404AEDB0BA8E],
		]
	},
	// round 11
	RoundValues{
		theta: [
			[u64(0x608DB8C02F955A3C), 0xE6151B748614F099, 0xF2AD3CCFFE33FDE0, 0xB284478FCBD52C8B,
				0x172A773D71DDF827],
			[u64(0x9994140947798528), 0x469A6318AE1899BF, 0x64D033B865E5544B, 0x351401EA7F9659DE,
				0x10F0AE2B823252E2],
			[u64(0x7AD958AEEF01588B), 0x21D9D0969BB466C4, 0xA378DD3C5A3400BF, 0xF3168F7243A31612,
				0xF4D6A7700CBB2624],
			[u64(0xE6E2DAFE0E5BDC50), 0xE060197270CFBCA5, 0x7E9FC9ABA8E0EF34, 0xD3A7586CB9B590E3,
				0xE79665C4FFD5E281],
			[u64(0xBF6B6A934F17A89B), 0xD64777C63340A3F9, 0xA1BF7740FD725979, 0xD871D645E69756BA,
				0x221ACBD6C9062E45],
		]
		rho:   [
			[u64(0x608DB8C02F955A3C), 0x614F099E6151B748, 0x9569E67FF19FEF07, 0xAA591765088F1F97,
				0xDCF5C777E09C5CA9],
			[u64(0x332828128EF30A51), 0x899BF469A6318AE1, 0x40CEE19795512D93, 0xCB3BC6A2803D4FF2,
				0x43C2B8AE08C94B88],
			[u64(0xDEB6562BBBC05622), 0x767425A6ED19B108, 0xA005FD1BC6E9E2D1, 0x47B921D18B09798B,
				0x9E9AD4EE019764C4],
			[u64(0xE0E5BDC50E6E2DAF), 0x52F0300CB93867DE, 0x5751C1DE68FD3F93, 0x0D9736B21C7A74EB,
				0x81E79665C4FFD5E2],
			[u64(0x9A78BD44DDFB5B54), 0x7C63340A3F9D6477, 0xB92CBCD0DFBBA07E, 0x71D645E69756BAD8,
				0xB2F5B2418B914886],
		]
		pi:    [
			[u64(0x608DB8C02F955A3C), 0xE0E5BDC50E6E2DAF, 0x332828128EF30A51, 0x9A78BD44DDFB5B54,
				0xDEB6562BBBC05622],
			[u64(0x899BF469A6318AE1), 0x7C63340A3F9D6477, 0x767425A6ED19B108, 0x614F099E6151B748,
				0x52F0300CB93867DE],
			[u64(0xA005FD1BC6E9E2D1), 0x9569E67FF19FEF07, 0x5751C1DE68FD3F93, 0x40CEE19795512D93,
				0xB92CBCD0DFBBA07E],
			[u64(0x0D9736B21C7A74EB), 0xCB3BC6A2803D4FF2, 0x71D645E69756BAD8, 0x47B921D18B09798B,
				0xAA591765088F1F97],
			[u64(0xB2F5B2418B914886), 0x9E9AD4EE019764C4, 0xDCF5C777E09C5CA9, 0x81E79665C4FFD5E2,
				0x43C2B8AE08C94B88],
		]
		chi:   [
			[u64(0x4089B1D26F5D3A2C), 0x61ED7FB0CE6CA6AF, 0x3229E84A8E1704C2, 0x9AF85D4549FB53C7,
				0x77BADAFBFD43D602],
			[u64(0x8409F6C9BE239ECB), 0x3671348A3FBD6487, 0x56F221867A1B3140, 0x667E09DE6B59E740,
				0x50A13329B93C785F],
			[u64(0x12657D5A4568EAD5), 0x81E9F633F01DCF03, 0xDB7043CF08757BB2, 0xC08877B3D1A7A9F3,
				0xF8AE145ADFFBE076],
			[u64(0x4D9F3E32387E66D3), 0xAB5EEFA38E5546D9, 0x52DE6DE69935B888, 0x5DA108D19209739F,
				0x366D5164BB8F0BB5],
			[u64(0x3BE7F6680BB1C847), 0x8298D4E430062494, 0x98A1C2D38194EDA1, 0xE0E096FFE4FF71EA,
				0x438298AA08F16A54],
		]
		iota:  [
			[u64(0x4089B1D2EF5D3A26), 0x61ED7FB0CE6CA6AF, 0x3229E84A8E1704C2, 0x9AF85D4549FB53C7,
				0x77BADAFBFD43D602],
			[u64(0x8409F6C9BE239ECB), 0x3671348A3FBD6487, 0x56F221867A1B3140, 0x667E09DE6B59E740,
				0x50A13329B93C785F],
			[u64(0x12657D5A4568EAD5), 0x81E9F633F01DCF03, 0xDB7043CF08757BB2, 0xC08877B3D1A7A9F3,
				0xF8AE145ADFFBE076],
			[u64(0x4D9F3E32387E66D3), 0xAB5EEFA38E5546D9, 0x52DE6DE69935B888, 0x5DA108D19209739F,
				0x366D5164BB8F0BB5],
			[u64(0x3BE7F6680BB1C847), 0x8298D4E430062494, 0x98A1C2D38194EDA1, 0xE0E096FFE4FF71EA,
				0x438298AA08F16A54],
		]
	},
	// round 12
	RoundValues{
		theta: [
			[u64(0x669EEDBCEAB088CD), 0x47FA23DECB811444, 0x143EB4248BFAB629, 0xBCEF012B4C16E12C,
				0x51AD8695F8AE64E9],
			[u64(0x9BB301C0C305AC87), 0x29CBC383429B56CB, 0x4948D68F073D030C, 0x79C4FED7167FD50C,
				0x4F1BC420C41A4A13],
			[u64(0x7F976FEC61B97F97), 0xEC1BE485D4CC5A41, 0xB68251792CA4EEF0, 0xAD7A6505F5763CB1,
				0x955C06ECFB2A7534],
			[u64(0x383C4969277844AB), 0xDEFD98F8915364A1, 0x277D1ABD86339AF0, 0x28027F8A8D0F51E7,
				0x43CE263FA48929CD],
			[u64(0x182B50863A1513F2), 0xA154720A01A2FF21, 0xBB6D643DB0303614, 0xC32C3011D55BAA5F,
				0x604E3E443955B1E1],
		]
		rho:   [
			[u64(0x669EEDBCEAB088CD), 0xB81144447FA23DEC, 0xA1F5A1245FD5B148, 0x2DC25979DE025698,
				0x1A57E2B993A546B6],
			[u64(0x37660381860B590F), 0xB56CB29CBC383429, 0x235A3C1CF40C3125, 0xFAA18F389FDAE2CF,
				0x3C6F10831069284D],
			[u64(0xDFE5DBFB186E5FE5), 0x06F921753316907B, 0x277785B4128BC965, 0x3282FABB1E58D6BD,
				0x92AB80DD9F654EA6],
			[u64(0x9277844AB383C496), 0x50EF7ECC7C48A9B2, 0x7B0C6735E04EFA35, 0xF151A1EA3CE5004F,
				0xCD43CE263FA48929],
			[u64(0x31D0A89F90C15A84), 0x20A01A2FF21A1547, 0x181B0A5DB6B21ED8, 0x2C3011D55BAA5FC3,
				0x8F910E556C785813],
		]
		pi:    [
			[u64(0x669EEDBCEAB088CD), 0x9277844AB383C496, 0x37660381860B590F, 0x31D0A89F90C15A84,
				0xDFE5DBFB186E5FE5],
			[u64(0xB56CB29CBC383429), 0x20A01A2FF21A1547, 0x06F921753316907B, 0xB81144447FA23DEC,
				0x50EF7ECC7C48A9B2],
			[u64(0x277785B4128BC965), 0xA1F5A1245FD5B148, 0x7B0C6735E04EFA35, 0x235A3C1CF40C3125,
				0x181B0A5DB6B21ED8],
			[u64(0xF151A1EA3CE5004F), 0xFAA18F389FDAE2CF, 0x2C3011D55BAA5FC3, 0x3282FABB1E58D6BD,
				0x2DC25979DE025698],
			[u64(0x8F910E556C785813), 0x92AB80DD9F654EA6, 0x1A57E2B993A546B6, 0xCD43CE263FA48929,
				0x3C6F10831069284D],
		]
		chi:   [
			[u64(0x648DE89CE8334189), 0x1322254ABE46649E, 0x4E6245814643330B, 0x329A908710CD5A85,
				0xD7F5DBEA9ADC49AD],
			[u64(0x656C92D6905C3423), 0x7AA01437721057C0, 0x02C931B528B695B9, 0xA89186E775F2FB74,
				0x752F2FEC3448E9B2],
			[u64(0x29F78BA152939175), 0xA1FFA1E15FF0BD68, 0x694B851D604BFA01, 0xEE1B3818D5A83825,
				0x08360ADFB6DB369D],
			[u64(0x915F4042BE658083), 0xFAF58B3ABF5862DF, 0x091010D55FA046CA, 0x0212DA229E198439,
				0xEE429201D6040138],
			[u64(0x1EF11C5578706C33), 0xB22B9AF8DF7D5FE7, 0x1ACEC2CDA2B1C6C6, 0x45428A665086AC41,
				0x3C6534877469885F],
		]
		iota:  [
			[u64(0x648DE89C6833C102), 0x1322254ABE46649E, 0x4E6245814643330B, 0x329A908710CD5A85,
				0xD7F5DBEA9ADC49AD],
			[u64(0x656C92D6905C3423), 0x7AA01437721057C0, 0x02C931B528B695B9, 0xA89186E775F2FB74,
				0x752F2FEC3448E9B2],
			[u64(0x29F78BA152939175), 0xA1FFA1E15FF0BD68, 0x694B851D604BFA01, 0xEE1B3818D5A83825,
				0x08360ADFB6DB369D],
			[u64(0x915F4042BE658083), 0xFAF58B3ABF5862DF, 0x091010D55FA046CA, 0x0212DA229E198439,
				0xEE429201D6040138],
			[u64(0x1EF11C5578706C33), 0xB22B9AF8DF7D5FE7, 0x1ACEC2CDA2B1C6C6, 0x45428A665086AC41,
				0x3C6534877469885F],
		]
	},
	// round 13
	RoundValues{
		theta: [
			[u64(0x2AC82EA25FE1D937), 0x5D67E37489947CAB, 0x002783BF71912B3E, 0x7CDF56B9271F42B0,
				0x99B01DD4AD0E5198],
			[u64(0xB7136AD896CC00D4), 0xA8DFEC3974806337, 0xD0B6C9BB2E26A14E, 0x7AEE7EE97362CF83,
				0xA750D7E232D8DD45],
			[u64(0xF499B2E2F4D336C6), 0x7C9198A2F9B01ADB, 0xB425BC5EC60B5DB2, 0x3375015B73E89F96,
				0xD558339C109B912E],
			[u64(0x085628DAF299FA3E), 0x63FCE3A2F3A41862, 0x9019784D135C3C77, 0x9B1BB2BAD2E5FE84,
				0x774BFA999AF87B85],
			[u64(0x295E09AF5ABF46DB), 0x85848F02FDB2750F, 0x2D61D737807EEC2E, 0x72ED9F9C724986A9,
				0x0BCA217D56A6A2B7],
		]
		rho:   [
			[u64(0x2AC82EA25FE1D937), 0x9947CAB5D67E3748, 0x013C1DFB8C8959F0, 0x3E8560F9BEAD724E,
				0x7752B439466266C0],
			[u64(0x6E26D5B12D9801A9), 0x06337A8DFEC39748, 0xDB26ECB89A853B42, 0x59F06F5DCFDD2E6C,
				0x9D435F88CB637516],
			[u64(0xBD266CB8BD34CDB1), 0x246628BE6C06B6DF, 0x5AED95A12DE2F630, 0x80ADB9F44FCB19BA,
				0xDAAB067382137225],
			[u64(0xAF299FA3E085628D), 0x3131FE71D179D20C, 0x9A26B878EF2032F0, 0x575A5CBFD0936376,
				0x85774BFA999AF87B],
			[u64(0x7AD5FA36D94AF04D), 0xF02FDB2750F85848, 0x3F761716B0EB9BC0, 0xED9F9C724986A972,
				0x885F55A9A8ADC2F2],
		]
		pi:    [
			[u64(0x2AC82EA25FE1D937), 0xAF299FA3E085628D, 0x6E26D5B12D9801A9, 0x7AD5FA36D94AF04D,
				0xBD266CB8BD34CDB1],
			[u64(0x06337A8DFEC39748), 0xF02FDB2750F85848, 0x246628BE6C06B6DF, 0x9947CAB5D67E3748,
				0x3131FE71D179D20C],
			[u64(0x5AED95A12DE2F630), 0x013C1DFB8C8959F0, 0x9A26B878EF2032F0, 0xDB26ECB89A853B42,
				0x3F761716B0EB9BC0],
			[u64(0x575A5CBFD0936376), 0x59F06F5DCFDD2E6C, 0xED9F9C724986A972, 0x80ADB9F44FCB19BA,
				0x3E8560F9BEAD724E],
			[u64(0x885F55A9A8ADC2F2), 0xDAAB067382137225, 0x7752B439466266C0, 0x85774BFA999AF87B,
				0x9D435F88CB637516],
		]
		chi:   [
			[u64(0x7204AB825EC1B907), 0xAE399B7B6C84633D, 0xF42645F1AEB80189, 0x38F5DE3ED1CBF84F,
				0xB3606DBE9DB6C471],
			[u64(0x032132932ED2960E), 0xA8EFB92313AC7E44, 0x41FF2CBC6C803FDD, 0x99CEDBF1933437F0,
				0x31B09E98DF7DB202],
			[u64(0xD2E894A105CE76B0), 0x83371DD98C8B09F1, 0x88669871E9407470, 0xDE74AEB20A95DB03,
				0xBE340816F1A99ED0],
			[u64(0x75DA76BD87D37A73), 0x7CF0F6DDAF592EE4, 0xE5BBDDF2601EA85B, 0xFA2D09F00F8B19BE,
				0x1EA140C98AB9FAEF],
			[u64(0x8C6C05A408AFC4BA), 0x8AAD4677926B6A65, 0x77129C370664D096, 0x04754B7B9FAEFF7B,
				0x9D52CDC98B2A671A],
		]
		iota:  [
			[u64(0xF204AB825EC1B98C), 0xAE399B7B6C84633D, 0xF42645F1AEB80189, 0x38F5DE3ED1CBF84F,
				0xB3606DBE9DB6C471],
			[u64(0x032132932ED2960E), 0xA8EFB92313AC7E44, 0x41FF2CBC6C803FDD, 0x99CEDBF1933437F0,
				0x31B09E98DF7DB202],
			[u64(0xD2E894A105CE76B0), 0x83371DD98C8B09F1, 0x88669871E9407470, 0xDE74AEB20A95DB03,
				0xBE340816F1A99ED0],
			[u64(0x75DA76BD87D37A73), 0x7CF0F6DDAF592EE4, 0xE5BBDDF2601EA85B, 0xFA2D09F00F8B19BE,
				0x1EA140C98AB9FAEF],
			[u64(0x8C6C05A408AFC4BA), 0x8AAD4677926B6A65, 0x77129C370664D096, 0x04754B7B9FAEFF7B,
				0x9D52CDC98B2A671A],
		]
	},
	// round 14
	RoundValues{
		theta: [
			[u64(0x9E6F361EED8BFB6E), 0xC25206E7DFCE21DF, 0x984DD86D1DF2436B, 0x549E43A26281BAAD,
				0xDF0BF0222EFC8693],
			[u64(0x535C9B40C820ECCD), 0xF89210F0F55E0487, 0x1182856F8A72451E, 0xC9B3722275C64D33,
				0x61CD374B398FC8C1],
			[u64(0x809D5F9383351BEF), 0xD142D6EB0A7064AE, 0xDA1353436FBB192F, 0x8C0165808C6EB65C,
				0xEC41C3247752F38F],
			[u64(0x1DCB73BD0CA3F8C0), 0x14E1F3DD2429AC57, 0x8DAAD8F2EB6E2AE8, 0x923C0CF084FB9B0D,
				0x76B045C901C9785C],
			[u64(0xC36C9C1E6408152B), 0xC5ADDFCDFECCBBF4, 0x3812058D6AC30107, 0x4B75D2C1F3092EEA,
				0xD2525473E78DB68B],
		]
		rho:   [
			[u64(0x9E6F361EED8BFB6E), 0xFCE21DFC25206E7D, 0xC26EC368EF921B5C, 0x03755AA93C8744C5,
				0xC088BBF21A4F7C2F],
			[u64(0xA6B936819041D99A), 0xE0487F89210F0F55, 0x0A15BE29C9147846, 0xC9A679366E444EB8,
				0x8734DD2CE63F2305],
			[u64(0xE02757E4E0CD46FB), 0x50B5BAC29C192BB4, 0xD8C97ED09A9A1B7D, 0xB2C046375B2E4600,
				0xFD8838648EEA5E71],
			[u64(0xD0CA3F8C01DCB73B), 0x2B8A70F9EE9214D6, 0xE5D6DC55D11B55B1, 0x9E109F7361B24781,
				0x5C76B045C901C978],
			[u64(0xF32040A95E1B64E0), 0xFCDFECCBBF4C5ADD, 0x6180839C0902C6B5, 0x75D2C1F3092EEA4B,
				0x951CF9E36DA2F494],
		]
		pi:    [
			[u64(0x9E6F361EED8BFB6E), 0xD0CA3F8C01DCB73B, 0xA6B936819041D99A, 0xF32040A95E1B64E0,
				0xE02757E4E0CD46FB],
			[u64(0xE0487F89210F0F55), 0xFCDFECCBBF4C5ADD, 0x50B5BAC29C192BB4, 0xFCE21DFC25206E7D,
				0x2B8A70F9EE9214D6],
			[u64(0xD8C97ED09A9A1B7D), 0xC26EC368EF921B5C, 0xE5D6DC55D11B55B1, 0x0A15BE29C9147846,
				0x6180839C0902C6B5],
			[u64(0x9E109F7361B24781), 0xC9A679366E444EB8, 0x75D2C1F3092EEA4B, 0xB2C046375B2E4600,
				0x03755AA93C8744C5],
			[u64(0x951CF9E36DA2F494), 0xFD8838648EEA5E71, 0xC088BBF21A4F7C2F, 0x5C76B045C901C978,
				0x8734DD2CE63F2305],
		]
		chi:   [
			[u64(0x86EE364E771BEB46), 0xD2EA3CAC414EB63B, 0x03FB7294D1438D9B, 0xF135E2A8960F74E2,
				0xA027D4E0E1CD84DA],
			[u64(0xE658FEAA402F4BD5), 0xF55FD4DDBF081E7D, 0x40B5BB60943D81FE, 0x4C225DEA370A687D,
				0x29FF28D8DA171496],
			[u64(0xD9C51E50969AAB69), 0xF666C3286F380B1D, 0x65DEE655C35A4195, 0x46230E694915F13E,
				0xE5800698CB3AE5B5],
			[u64(0x9473996FE1BB4CEB), 0xC9E47EBE6F50EFB2, 0x53E3C5F2892E6BDB, 0x11C0069F4D346280,
				0x637658693C47003F],
			[u64(0xF51CB0626DA6F085), 0xD19DF82730EA16B5, 0x908C33B016575E0B, 0x50B4AD11E821C365,
				0x8CBCFD35E82D3301],
		]
		iota:  [
			[u64(0x06EE364E771B6BCF), 0xD2EA3CAC414EB63B, 0x03FB7294D1438D9B, 0xF135E2A8960F74E2,
				0xA027D4E0E1CD84DA],
			[u64(0xE658FEAA402F4BD5), 0xF55FD4DDBF081E7D, 0x40B5BB60943D81FE, 0x4C225DEA370A687D,
				0x29FF28D8DA171496],
			[u64(0xD9C51E50969AAB69), 0xF666C3286F380B1D, 0x65DEE655C35A4195, 0x46230E694915F13E,
				0xE5800698CB3AE5B5],
			[u64(0x9473996FE1BB4CEB), 0xC9E47EBE6F50EFB2, 0x53E3C5F2892E6BDB, 0x11C0069F4D346280,
				0x637658693C47003F],
			[u64(0xF51CB0626DA6F085), 0xD19DF82730EA16B5, 0x908C33B016575E0B, 0x50B4AD11E821C365,
				0x8CBCFD35E82D3301],
		]
	},
	// round 15
	RoundValues{
		theta: [
			[u64(0x0234D5D4300372EA), 0xD630DF360656AF1E, 0x0721910E965B94BE, 0xF5EF0132D1176DC7,
				0xA4FD377AA6D59DFF],
			[u64(0xB309D72DA1540157), 0xA00EFD5A5E7354FF, 0x15E492E77546CB7C, 0x1973746DD67122FF,
				0x7CAE015F3B6C5E14],
			[u64(0x162E03DFFDF057AE), 0x398DDEA70452F7DA, 0xAA35FBDAA830BD52, 0x89C813E6227F0DF9,
				0x2A6B1B17A0501972],
			[u64(0xADA7FD11CF42293F), 0xF0301AC041A98A66, 0x6A37A18CA7D70E0F, 0x281462E163CD0754,
				0x5AA23C1712BE65EB],
			[u64(0x840450CA3AB91A17), 0xA085188F67F5FC27, 0xE194D3184148B499, 0x21AC4DB9BF3E29F7,
				0xFDA41D9DBF32D993],
		]
		rho:   [
			[u64(0x0234D5D4300372EA), 0x656AF1ED630DF360, 0x390C8874B2DCA5F0, 0x2EDB8FEBDE0265A2,
				0xDDEA9B5677FE93F4],
			[u64(0x6613AE5B42A802AF), 0x354FFA00EFD5A5E7, 0x924B9DD51B2DF057, 0x245FE32E6E8DBACE,
				0xF2B8057CEDB17851],
			[u64(0x858B80F7FF7C15EB), 0x6377A9C114BDF68E, 0x85EA9551AFDED541, 0x09F3113F86FCC4E4,
				0x454D6362F40A032E],
			[u64(0x1CF42293FADA7FD1), 0x3378180D6020D4C5, 0x194FAE1C1ED46F43, 0x5C2C79A0EA85028C,
				0xEB5AA23C1712BE65],
			[u64(0x51D5C8D0BC202286), 0x88F67F5FC27A0851, 0xA45A4CF0CA698C20, 0xAC4DB9BF3E29F721,
				0x07676FCCB664FF69],
		]
		pi:    [
			[u64(0x0234D5D4300372EA), 0x1CF42293FADA7FD1, 0x6613AE5B42A802AF, 0x51D5C8D0BC202286,
				0x858B80F7FF7C15EB],
			[u64(0x354FFA00EFD5A5E7), 0x88F67F5FC27A0851, 0x6377A9C114BDF68E, 0x656AF1ED630DF360,
				0x3378180D6020D4C5],
			[u64(0x85EA9551AFDED541), 0x390C8874B2DCA5F0, 0x194FAE1C1ED46F43, 0x924B9DD51B2DF057,
				0xA45A4CF0CA698C20],
			[u64(0x5C2C79A0EA85028C), 0x245FE32E6E8DBACE, 0xAC4DB9BF3E29F721, 0x09F3113F86FCC4E4,
				0x2EDB8FEBDE0265A2],
			[u64(0x07676FCCB664FF69), 0x454D6362F40A032E, 0xDDEA9B5677FE93F4, 0xEB5AA23C1712BE65,
				0xF2B8057CEDB17851],
		]
		chi:   [
			[u64(0x8294D085300922EA), 0x2DFCA2B3CA5EDA71, 0x7E1BA84748E80BEE, 0xC3D4C4C0A4002291,
				0x0189C40775351DCB],
			[u64(0x6D4B92A0AFD4A76B), 0x8CA51C558E7B125F, 0xC777B862349466AE, 0x6CDAF1C7E7DDF7C0,
				0x39F99B067422B547],
			[u64(0x86A9931DBBBE2820), 0x780C883422DEA4D0, 0x48EDAC5C5F026F97, 0x70433FD50A2FCA56,
				0x747A4CE4EBD89471],
			[u64(0x5C3CE9B0EA86020E), 0x3CEFE3BF645DC61F, 0x8E5C9DB63E29F72A, 0x197659FF2EDCC466,
				0x2BD80F68CC4E6008],
			[u64(0x322C45CC79B07A6C), 0xC54F3E2EF42A032E, 0xDC8E9AD663EB67F4, 0xCF709311541F6F05,
				0xC0C81D74EDB1B855],
		]
		iota:  [
			[u64(0x0294D0853009A2E9), 0x2DFCA2B3CA5EDA71, 0x7E1BA84748E80BEE, 0xC3D4C4C0A4002291,
				0x0189C40775351DCB],
			[u64(0x6D4B92A0AFD4A76B), 0x8CA51C558E7B125F, 0xC777B862349466AE, 0x6CDAF1C7E7DDF7C0,
				0x39F99B067422B547],
			[u64(0x86A9931DBBBE2820), 0x780C883422DEA4D0, 0x48EDAC5C5F026F97, 0x70433FD50A2FCA56,
				0x747A4CE4EBD89471],
			[u64(0x5C3CE9B0EA86020E), 0x3CEFE3BF645DC61F, 0x8E5C9DB63E29F72A, 0x197659FF2EDCC466,
				0x2BD80F68CC4E6008],
			[u64(0x322C45CC79B07A6C), 0xC54F3E2EF42A032E, 0xDC8E9AD663EB67F4, 0xCF709311541F6F05,
				0xC0C81D74EDB1B855],
		]
	},
	// round 16
	RoundValues{
		theta: [
			[u64(0xC1B507796A5F4935), 0xEEDD754F900831AD, 0xBD3A7FBB12BEE032, 0x00F5133CFE56C94D,
				0xC2A813FB2F63F617],
			[u64(0x9A86C09E837591C6), 0x7B684E6BA2DA24F2, 0x30BAEA5C18355003, 0x9B17A3F9CB7CC16D,
				0xCE34C938588383EA],
			[u64(0x4D504D1799BB9796), 0xB3F5563E00DB1B66, 0x831472567D07D021, 0xBBBAE1DF282A75E0,
				0xBF8392EEC9DD2BC7],
			[u64(0xA6E7F35662AC2C82), 0xC634F959EC77E893, 0x74878750B603D9A6, 0xE3AD4319A6F6EAEA,
				0xD103158E44644E84],
			[u64(0xC850318EECC47560), 0x3F334A6C615E0C22, 0x26F2EE94F69F68F8, 0x350CE753C16B6009,
				0x3AB4693678C5B759],
		]
		rho:   [
			[u64(0xC1B507796A5F4935), 0x00831ADEEDD754F9, 0xE9D3FDD895F70195, 0xAD929A01EA2679FC,
				0x4FECBD8FD85F0AA0],
			[u64(0x350D813D06EB238D), 0xA24F27B684E6BA2D, 0xEBA97060D5400CC2, 0x982DB362F47F396F,
				0x38D324E1620E0FAB],
			[u64(0x93541345E66EE5E5), 0xFD558F8036C6D9AC, 0x3E810C18A392B3E8, 0x70EF94153AF05DDD,
				0xF7F0725DD93BA578],
			[u64(0x662AC2C82A6E7F35), 0x49E31A7CACF63BF4, 0xA16C07B34CE90F0E, 0x6334DEDD5D5C75A8,
				0x84D103158E44644E],
			[u64(0x776623AB0642818C), 0xA6C615E0C223F334, 0x4FB47C1379774A7B, 0x0CE753C16B600935,
				0x1A4D9E316DD64EAD],
		]
		pi:    [
			[u64(0xC1B507796A5F4935), 0x662AC2C82A6E7F35, 0x350D813D06EB238D, 0x776623AB0642818C,
				0x93541345E66EE5E5],
			[u64(0xA24F27B684E6BA2D), 0xA6C615E0C223F334, 0xFD558F8036C6D9AC, 0x00831ADEEDD754F9,
				0x49E31A7CACF63BF4],
			[u64(0x3E810C18A392B3E8), 0xE9D3FDD895F70195, 0xA16C07B34CE90F0E, 0xEBA97060D5400CC2,
				0x4FB47C1379774A7B],
			[u64(0x6334DEDD5D5C75A8), 0x982DB362F47F396F, 0x0CE753C16B600935, 0x70EF94153AF05DDD,
				0xAD929A01EA2679FC],
			[u64(0x1A4D9E316DD64EAD), 0xF7F0725DD93BA578, 0x4FECBD8FD85F0AA0, 0x84D103158E44644E,
				0x38D324E1620E0FAB],
		]
		chi:   [
			[u64(0xDD350F71494F48F5), 0x2F3B2AD03FBA7FB4, 0x3525810E4EC2258F, 0x9C4E438B1642898E,
				0x95407746B76FA5EE],
			[u64(0xE37BF573D8AAFE2D), 0xB6EA17C2A22BCB5E, 0xF1D6DFC015C6D99D, 0x10C59ECBC76705E4,
				0xE9E1987C2EF60A70],
			[u64(0x26C80C388310B9ED), 0x8E03BDC59CF78585, 0xE264ABBDDCF60D8E, 0x6FB9736051442CC0,
				0x5FF558F3797F4C78],
			[u64(0xA284DF955F5574B8), 0x982733E2D63B636A, 0x3CE653F16DC02838, 0x03C9B4BF3AF2DC5D,
				0x2E9689056E4699B8],
			[u64(0x3807BEB7E976FCA5), 0x7734677D193A2578, 0x87BCB30FE85BD280, 0x84501B4167D1303F,
				0x70702CD96A9E15BB],
		]
		iota:  [
			[u64(0x5D350F71494FC8F7), 0x2F3B2AD03FBA7FB4, 0x3525810E4EC2258F, 0x9C4E438B1642898E,
				0x95407746B76FA5EE],
			[u64(0xE37BF573D8AAFE2D), 0xB6EA17C2A22BCB5E, 0xF1D6DFC015C6D99D, 0x10C59ECBC76705E4,
				0xE9E1987C2EF60A70],
			[u64(0x26C80C388310B9ED), 0x8E03BDC59CF78585, 0xE264ABBDDCF60D8E, 0x6FB9736051442CC0,
				0x5FF558F3797F4C78],
			[u64(0xA284DF955F5574B8), 0x982733E2D63B636A, 0x3CE653F16DC02838, 0x03C9B4BF3AF2DC5D,
				0x2E9689056E4699B8],
			[u64(0x3807BEB7E976FCA5), 0x7734677D193A2578, 0x87BCB30FE85BD280, 0x84501B4167D1303F,
				0x70702CD96A9E15BB],
		]
	},
	// round 17
	RoundValues{
		theta: [
			[u64(0xDB5C25A151BA20DA), 0xA9520000274F9799, 0xB34CABDE5637CDA2, 0x1A27695B0EB761A3,
				0x13295D96AF9A4DC3],
			[u64(0x589806B697E4E23D), 0x0D09E407ED65D74E, 0x4A352C055A88C58D, 0xAB266D0E882919F4,
				0x52026BB961B81660],
			[u64(0x2D9E338765F2AE89), 0x8555827A7A1592E1, 0xE93294023A141AEA, 0x64EF4CDFB7A63BA4,
				0x54A3674C9F9D5B1C],
			[u64(0xA13954FC9ECF7854), 0x9B9AB88B17A16F86, 0x3F5BD898AC5A24D4, 0x00743FD6FB68D0B1,
				0x2D2B026CAFDC9554],
			[u64(0x8F561C4E6B59FBF2), 0xC065C5849B15222F, 0x30ED11F66A74D5D7, 0x3301B9B8E5FE3768,
				0xC7218E20E8B112EC],
		]
		rho:   [
			[u64(0xDB5C25A151BA20DA), 0x74F9799A95200002, 0x9A655EF2B1BE6D15, 0x6EC346344ED2B61D,
				0x765ABE69370C4CA5],
			[u64(0xB1300D6D2FC9C47A), 0x5D74E0D09E407ED6, 0xD4B0156A23163528, 0x233E9564CDA1D105,
				0x4809AEE586E05981],
			[u64(0x4B678CE1D97CABA2), 0x55609E9E8564B861, 0xA0D7574994A011D0, 0xA66FDBD31DD23277,
				0x8A946CE993F3AB63],
			[u64(0xC9ECF7854A13954F), 0xC34DCD5C458BD0B7, 0x3158B449A87EB7B1, 0xFADF6D1A16200E87,
				0x542D2B026CAFDC95],
			[u64(0x735ACFDF947AB0E2), 0x5849B15222FC065C, 0x3A6AEB987688FB35, 0x01B9B8E5FE376833,
				0x63883A2C44BB31C8],
		]
		pi:    [
			[u64(0xDB5C25A151BA20DA), 0xC9ECF7854A13954F, 0xB1300D6D2FC9C47A, 0x735ACFDF947AB0E2,
				0x4B678CE1D97CABA2],
			[u64(0x5D74E0D09E407ED6), 0x5849B15222FC065C, 0x55609E9E8564B861, 0x74F9799A95200002,
				0xC34DCD5C458BD0B7],
			[u64(0xA0D7574994A011D0), 0x9A655EF2B1BE6D15, 0x3158B449A87EB7B1, 0xD4B0156A23163528,
				0x3A6AEB987688FB35],
			[u64(0xFADF6D1A16200E87), 0x233E9564CDA1D105, 0x01B9B8E5FE376833, 0xA66FDBD31DD23277,
				0x6EC346344ED2B61D],
			[u64(0x63883A2C44BB31C8), 0x8A946CE993F3AB63, 0x765ABE69370C4CA5, 0x542D2B026CAFDC95,
				0x4809AEE586E05981],
		]
		chi:   [
			[u64(0x7BDF32A8511A21DA), 0x4BC8B925DB11FC4E, 0x91282D2C07D3C3EA, 0xF35ACBBFB66C85CA,
				0x7345AE61EB7C80A2],
			[u64(0x077CC8C29C4070D1), 0x795330566EFD965C, 0x55C1963AD365F063, 0x56B6B30B89E00255,
				0x87CCC9784DD9D4BF],
			[u64(0xA1D7456DD43B2098), 0x12E5367BA3EC4777, 0x471AB241A976B335, 0x84B0356A433BF9A8,
				0x3A624359F6A8B2B5],
			[u64(0x628B689B07200E95), 0x6256066085A1C509, 0x8099B9E1F6F6E869, 0x853D1F0E8D821215,
				0x6DA5463417CE143F],
			[u64(0x67A8FA7CCAFB6FCC), 0x9A956CBBB31FA973, 0x321A2CFBB72874A4, 0x508C1B026DAFDC95,
				0xC801EFF982630994],
		]
		iota:  [
			[u64(0xFBDF32A8511A215A), 0x4BC8B925DB11FC4E, 0x91282D2C07D3C3EA, 0xF35ACBBFB66C85CA,
				0x7345AE61EB7C80A2],
			[u64(0x077CC8C29C4070D1), 0x795330566EFD965C, 0x55C1963AD365F063, 0x56B6B30B89E00255,
				0x87CCC9784DD9D4BF],
			[u64(0xA1D7456DD43B2098), 0x12E5367BA3EC4777, 0x471AB241A976B335, 0x84B0356A433BF9A8,
				0x3A624359F6A8B2B5],
			[u64(0x628B689B07200E95), 0x6256066085A1C509, 0x8099B9E1F6F6E869, 0x853D1F0E8D821215,
				0x6DA5463417CE143F],
			[u64(0x67A8FA7CCAFB6FCC), 0x9A956CBBB31FA973, 0x321A2CFBB72874A4, 0x508C1B026DAFDC95,
				0xC801EFF982630994],
		]
	},
	// round 18
	RoundValues{
		theta: [
			[u64(0x595D55D4BBD9C649), 0xE94ADE5931D21B5D, 0x33AA4A50ED1024F9, 0x51D8ACC35CAF62D9,
				0xD1C7C91D01BF67B1],
			[u64(0x33A965759AED54C9), 0x4D869DE16850B244, 0x61143B8DD5C8D47B, 0x62631EBC8F4D264D,
				0xB31964CF4B74F0A7],
			[u64(0x8AFA4DF1EDACAB22), 0x39C83EE79A7BCCCD, 0x6C37BADD90E1388F, 0xAF9D3DF67AAC7212,
				0x114F4BC5CF3F390F],
			[u64(0x872542712E125F66), 0x87F82C8AAC9394FA, 0x6537930BDFC4B99A, 0x609335E4A4B043E6,
				0x880B6CDE3EFC45CC],
			[u64(0x4D35F2A385507D3E), 0xB0086464FCB4BB81, 0x18872424F8836656, 0x7A1113DD2204CE67,
				0xE29CE726CDC81B66],
		]
		rho:   [
			[u64(0x595D55D4BBD9C649), 0x1D21B5DE94ADE593, 0x9D525287688127C9, 0x5EC5B2A3B15986B9,
				0x247406FD9EC7471F],
			[u64(0x6752CAEB35DAA992), 0x0B2444D869DE1685, 0x50EE37572351ED84, 0xA4C9AC4C63D791E9,
				0xCC65933D2DD3C29E],
			[u64(0xA2BE937C7B6B2AC8), 0x720FB9E69EF3334E, 0x09C47B61BDD6EC87, 0x9EFB3D56390957CE,
				0xE229E978B9E7E721],
			[u64(0x12E125F668725427), 0x7D43FC16455649CA, 0x17BF897334CA6F26, 0xBC9496087CCC1266,
				0xCC880B6CDE3EFC45],
			[u64(0x1C2A83E9F269AF95), 0x464FCB4BB81B0086, 0x41B32B0C4392127C, 0x1113DD2204CE677A,
				0x39C9B37206D9B8A7],
		]
		pi:    [
			[u64(0x595D55D4BBD9C649), 0x12E125F668725427, 0x6752CAEB35DAA992, 0x1C2A83E9F269AF95,
				0xA2BE937C7B6B2AC8],
			[u64(0x0B2444D869DE1685), 0x464FCB4BB81B0086, 0x720FB9E69EF3334E, 0x1D21B5DE94ADE593,
				0x7D43FC16455649CA],
			[u64(0x09C47B61BDD6EC87), 0x9D525287688127C9, 0x17BF897334CA6F26, 0x50EE37572351ED84,
				0x41B32B0C4392127C],
			[u64(0xBC9496087CCC1266), 0xA4C9AC4C63D791E9, 0x1113DD2204CE677A, 0x9EFB3D56390957CE,
				0x5EC5B2A3B15986B9],
			[u64(0x39C9B37206D9B8A7), 0xE229E978B9E7E721, 0x247406FD9EC7471F, 0xCC880B6CDE3EFC45,
				0xCC65933D2DD3C29E],
		]
		chi:   [
			[u64(0x599D6EF52FD92E4B), 0x8BF1357228F2736E, 0x62E2CAFA15D2E5B2, 0x5CE481E8D139A791,
				0xA20E907479EB38FC],
			[u64(0xBF34C0D029D604E5), 0x66C66703BB4D90A6, 0x720FEDE69EF73316, 0x9330BDDE8CA5F7D9,
				0x63076CB5F51FCD4B],
			[u64(0x088D5A13BFC74406), 0xDF7213B7F0A141C9, 0x33DB8BAEAECB6F23, 0x10EE357FE5674585,
				0xC1932A104F10527A],
			[u64(0xFC80D28CC5CC542E), 0xB409A8CA23C781EF, 0x5211152025D6CFFA, 0x8ED9BDD71948545E,
				0x7C5FB2E3E371AEF9],
			[u64(0x3BE9B37A46DFA823), 0xA627237129EEE7A1, 0x347937F914E65553, 0xCD893F7ADABABC47,
				0x9124FF3F29C7839C],
		]
		iota:  [
			[u64(0x599D6EF52FD9AE41), 0x8BF1357228F2736E, 0x62E2CAFA15D2E5B2, 0x5CE481E8D139A791,
				0xA20E907479EB38FC],
			[u64(0xBF34C0D029D604E5), 0x66C66703BB4D90A6, 0x720FEDE69EF73316, 0x9330BDDE8CA5F7D9,
				0x63076CB5F51FCD4B],
			[u64(0x088D5A13BFC74406), 0xDF7213B7F0A141C9, 0x33DB8BAEAECB6F23, 0x10EE357FE5674585,
				0xC1932A104F10527A],
			[u64(0xFC80D28CC5CC542E), 0xB409A8CA23C781EF, 0x5211152025D6CFFA, 0x8ED9BDD71948545E,
				0x7C5FB2E3E371AEF9],
			[u64(0x3BE9B37A46DFA823), 0xA627237129EEE7A1, 0x347937F914E65553, 0xCD893F7ADABABC47,
				0x9124FF3F29C7839C],
		]
	},
	// round 19
	RoundValues{
		theta: [
			[u64(0x1B123FFE4CDEB0C5), 0xC97E64794BF56DEA, 0x206D9BF176D5FB36, 0x1E6BD0E3B23EB915,
				0xE081C17F1AEC2678],
			[u64(0x9BE3FAFB04495933), 0x42115D2896D2CD70, 0x56D8D7CDB3686EC0, 0xB7E787F5A13AAA0F,
				0x47D0569ED880909D],
			[u64(0x837B01E9B9D818F8), 0x5484484DF6BE1D37, 0xB82DD054A8D433DD, 0x9B186E85E378197B,
				0x4A6571EA490F0E84],
			[u64(0x23EDC0869F426328), 0x6B64BAC07949B6E9, 0x8D7C072A7F58F8FC, 0x51B4AFDD43C66358,
				0xA332A0E9B9FF99FF],
			[u64(0x4F3ED2EB0B6C075F), 0xD2F042E0645D48DD, 0x40AE56685955FA2F, 0xB95E5EEB9709133B,
				0xE5F39EAE64742CE0],
		]
		rho:   [
			[u64(0x1B123FFE4CDEB0C5), 0xBF56DEAC97E64794, 0x036CDF8BB6AFD9B1, 0x7D722A3CD7A1C764,
				0x05FC6BB099E38207],
			[u64(0x37C7F5F60892B267), 0x2CD7042115D2896D, 0x635F36CDA1BB015B, 0x5541F6FCF0FEB427,
				0x1F415A7B62024275],
			[u64(0x20DEC07A6E76063E), 0x2112137DAF874DD5, 0xA19EEDC16E82A546, 0x3742F1BC0CBDCD8C,
				0x894CAE3D4921E1D0],
			[u64(0x69F42632823EDC08), 0x74B5B25D603CA4DB, 0x54FEB1F1F91AF80E, 0xFBA878CC6B0A3695,
				0xFFA332A0E9B9FF99],
			[u64(0x585B603AFA79F697), 0x2E0645D48DDD2F04, 0xAAFD17A0572B342C, 0x5E5EEB9709133BB9,
				0xE7AB991D0B38397C],
		]
		pi:    [
			[u64(0x1B123FFE4CDEB0C5), 0x69F42632823EDC08, 0x37C7F5F60892B267, 0x585B603AFA79F697,
				0x20DEC07A6E76063E],
			[u64(0x2CD7042115D2896D), 0x2E0645D48DDD2F04, 0x2112137DAF874DD5, 0xBF56DEAC97E64794,
				0x74B5B25D603CA4DB],
			[u64(0xA19EEDC16E82A546), 0x036CDF8BB6AFD9B1, 0x54FEB1F1F91AF80E, 0x635F36CDA1BB015B,
				0xAAFD17A0572B342C],
			[u64(0xFBA878CC6B0A3695), 0x5541F6FCF0FEB427, 0x5E5EEB9709133BB9, 0x3742F1BC0CBDCD8C,
				0x7D722A3CD7A1C764],
			[u64(0xE7AB991D0B38397C), 0x894CAE3D4921E1D0, 0x05FC6BB099E38207, 0xFFA332A0E9B9FF99,
				0x1F415A7B62024275],
		]
		chi:   [
			[u64(0x9A1AD63E26DE94C7), 0x689CBC39B01C0CB9, 0x632B5576588A026D, 0x1852407BDA60F6DC,
				0xAA96C5DA7975161A],
			[u64(0x76F7142D14DA9BFC), 0x7A0765A0CD8D0B02, 0x2B12597BAF864E64, 0xAB561F9C9BE28B10,
				0x21B79A41E0BC679B],
			[u64(0xA59D6CD06EB2AC2E), 0x8B60D78ABFAE9861, 0x555EB1D169FA7808, 0xABFE34CD40BB334A,
				0xA8FC47E37729343D],
			[u64(0xE3B85E2E2FCCB614), 0x35F1F6FE72E0A82F, 0x6C5D7FD109030BD9, 0x371AB1A61EFDCD8A,
				0x5DECAA3CDBD5C36E],
			[u64(0xC36E991C1A383054), 0x8F4EEFF944E0C2D4, 0x05EC69B93EE6CF97, 0x58A7AC24EC3FFE99,
				0x4B60687E620AE2B4],
		]
		iota:  [
			[u64(0x1A1AD63EA6DE94CD), 0x689CBC39B01C0CB9, 0x632B5576588A026D, 0x1852407BDA60F6DC,
				0xAA96C5DA7975161A],
			[u64(0x76F7142D14DA9BFC), 0x7A0765A0CD8D0B02, 0x2B12597BAF864E64, 0xAB561F9C9BE28B10,
				0x21B79A41E0BC679B],
			[u64(0xA59D6CD06EB2AC2E), 0x8B60D78ABFAE9861, 0x555EB1D169FA7808, 0xABFE34CD40BB334A,
				0xA8FC47E37729343D],
			[u64(0xE3B85E2E2FCCB614), 0x35F1F6FE72E0A82F, 0x6C5D7FD109030BD9, 0x371AB1A61EFDCD8A,
				0x5DECAA3CDBD5C36E],
			[u64(0xC36E991C1A383054), 0x8F4EEFF944E0C2D4, 0x05EC69B93EE6CF97, 0x58A7AC24EC3FFE99,
				0x4B60687E620AE2B4],
		]
	},
	// round 20
	RoundValues{
		theta: [
			[u64(0x1A16576E53CBD1D4), 0x68903D69450949A0, 0x6327D426AD9F4774, 0x185EC12B2F75B3C5,
				0xAA9A448A8C605303],
			[u64(0x24DC5DB6E76F7743), 0x282C2C3B3E38E7BD, 0x793910E05C33A2DB, 0xF97D5607685767AF,
				0x739CD3DA13098B24],
			[u64(0xA95B58CC4133A832), 0x87A6E396902F9C7D, 0x599885CD467B7C14, 0xA73800D16F3A3756,
				0xA43A73FF58A83021],
			[u64(0x2F0E91867CAEBF50), 0xF94739562182A16B, 0xA0EBB0795A61029D, 0xFBAC7E0E4D9FC4CE,
				0x915A659488B7CA2A],
			[u64(0x555F20265185DEED), 0x197F56C30F5D2C6D, 0x93DDD083755B212E, 0xCE96151EA7821020,
				0xDD51D14429B70C0D],
		]
		rho:   [
			[u64(0x1A16576E53CBD1D4), 0x50949A068903D694, 0x193EA1356CFA3BA3, 0xEB678A30BD82565E,
				0x122A31814C0EAA69],
			[u64(0x49B8BB6DCEDEEE86), 0x8E7BD282C2C3B3E3, 0xE4438170CE8B6DE4, 0xECF5FF2FAAC0ED0A,
				0xCE734F684C262C91],
			[u64(0xAA56D633104CEA0C), 0xE9B8E5A40BE71F61, 0xDBE0A2CCC42E6A33, 0x0068B79D1BAB539C,
				0x34874E7FEB150604],
			[u64(0x67CAEBF502F0E918), 0xB5FCA39CAB10C150, 0xF2B4C2053B41D760, 0xC1C9B3F899DF758F,
				0x2A915A659488B7CA],
			[u64(0x328C2EF76AAAF901), 0x6C30F5D2C6D197F5, 0xAD909749EEE841BA, 0x96151EA7821020CE,
				0x74510A6DC3037754],
		]
		pi:    [
			[u64(0x1A16576E53CBD1D4), 0x67CAEBF502F0E918, 0x49B8BB6DCEDEEE86, 0x328C2EF76AAAF901,
				0xAA56D633104CEA0C],
			[u64(0x8E7BD282C2C3B3E3), 0x6C30F5D2C6D197F5, 0xE9B8E5A40BE71F61, 0x50949A068903D694,
				0xB5FCA39CAB10C150],
			[u64(0xDBE0A2CCC42E6A33), 0x193EA1356CFA3BA3, 0xF2B4C2053B41D760, 0xE4438170CE8B6DE4,
				0xAD909749EEE841BA],
			[u64(0xC1C9B3F899DF758F), 0xECF5FF2FAAC0ED0A, 0x96151EA7821020CE, 0x0068B79D1BAB539C,
				0xEB678A30BD82565E],
			[u64(0x74510A6DC3037754), 0x34874E7FEB150604, 0x122A31814C0EAA69, 0x2A915A659488B7CA,
				0xCE734F684C262C91],
		]
		chi:   [
			[u64(0x4B96772257E799C4), 0x76C4EBD02ADAC11A, 0x5BBCB96CFEDE2E86, 0x96CF2F872C22D061,
				0xA256C27254A4EAA6],
			[u64(0x8E72C3B2DB12A66F), 0x88F1ABD844D153FD, 0xEDB9F9068BF73FEF, 0x50BCAC8B9823C48C,
				0xF79BABACBA12D714],
			[u64(0xEFF0AAC9862E6863), 0x093CA1652DEF39A7, 0xF29EE305774F5D41, 0xCED2C9104A8BC9A6,
				0xA980D201AECC693B],
			[u64(0xCBCFE6FA8917F50F), 0xAFBD5EAFAA200412, 0xDF8594CB00C06448, 0x1064930F71891B9D,
				0xCB631A23ADCA9452],
			[u64(0xF0388AED43035577), 0x3CB75A7D2F1410E1, 0xB22A75014D2FBB08, 0x6A81CA651589B15E,
				0xDBDB6EE4E7362DC1],
		]
		iota:  [
			[u64(0xCB967722D7E71945), 0x76C4EBD02ADAC11A, 0x5BBCB96CFEDE2E86, 0x96CF2F872C22D061,
				0xA256C27254A4EAA6],
			[u64(0x8E72C3B2DB12A66F), 0x88F1ABD844D153FD, 0xEDB9F9068BF73FEF, 0x50BCAC8B9823C48C,
				0xF79BABACBA12D714],
			[u64(0xEFF0AAC9862E6863), 0x093CA1652DEF39A7, 0xF29EE305774F5D41, 0xCED2C9104A8BC9A6,
				0xA980D201AECC693B],
			[u64(0xCBCFE6FA8917F50F), 0xAFBD5EAFAA200412, 0xDF8594CB00C06448, 0x1064930F71891B9D,
				0xCB631A23ADCA9452],
			[u64(0xF0388AED43035577), 0x3CB75A7D2F1410E1, 0xB22A75014D2FBB08, 0x6A81CA651589B15E,
				0xDBDB6EE4E7362DC1],
		]
	},
	// round 21
	RoundValues{
		theta: [
			[u64(0x9C525AA4686BC88E), 0x2100C656955610D1, 0x0C7894EA4152FF4D, 0xC10B020193AE01AA,
				0xF592EFF4EB283B6D],
			[u64(0xBA04ECA9D1E43241), 0xBC8784C34E27C7D3, 0xD9CFD61D8101ABC1, 0x64CA839092D550A2,
				0xC3ED84B7B0E4433A],
			[u64(0x620C77E74F4384B2), 0x84C07C4BE482D576, 0x7F623E2BBE22B190, 0x432E143E83E62577,
				0x247C0F2F67A185EA],
			[u64(0x2731176316D09D14), 0x4343AF3635E76C09, 0x337B65529F070C53, 0xFC9A6296EE4E7386,
				0x279DEBBA320DFC49],
			[u64(0x3427BF894A7CD7D0), 0xF8A86F19266B9246, 0x76354065445039AF, 0xAE9EFF011CF633F9,
				0x1FC45B80EE49AF66],
		]
		rho:   [
			[u64(0x9C525AA4686BC88E), 0x55610D12100C6569, 0x63C4A7520A97FA68, 0x5C03558216040327,
				0xBFD3ACA0EDB7D64B],
			[u64(0x7409D953A3C86483), 0x7C7D3BC8784C34E2, 0x3F58760406AF0767, 0xAA144C995072125A,
				0x0FB612DEC3910CEB],
			[u64(0x98831DF9D3D0E12C), 0x301F12F920B55DA1, 0x158C83FB11F15DF1, 0x0A1F41F312BBA197,
				0x448F81E5ECF430BD],
			[u64(0x316D09D142731176), 0x04A1A1D79B1AF3B6, 0xA53E0E18A666F6CA, 0x52DDC9CE70DF934C,
				0x49279DEBBA320DFC],
			[u64(0x4A53E6BE81A13DFC), 0xF19266B9246F8A86, 0x281CD7BB1AA032A2, 0x9EFF011CF633F9AE,
				0x16E03B926BD987F1],
		]
		pi:    [
			[u64(0x9C525AA4686BC88E), 0x316D09D142731176, 0x7409D953A3C86483, 0x4A53E6BE81A13DFC,
				0x98831DF9D3D0E12C],
			[u64(0x7C7D3BC8784C34E2), 0xF19266B9246F8A86, 0x301F12F920B55DA1, 0x55610D12100C6569,
				0x04A1A1D79B1AF3B6],
			[u64(0x158C83FB11F15DF1), 0x63C4A7520A97FA68, 0xA53E0E18A666F6CA, 0x3F58760406AF0767,
				0x281CD7BB1AA032A2],
			[u64(0x52DDC9CE70DF934C), 0xAA144C995072125A, 0x9EFF011CF633F9AE, 0x0A1F41F312BBA197,
				0x5C03558216040327],
			[u64(0x16E03B926BD987F1), 0x448F81E5ECF430BD, 0xBFD3ACA0EDB7D64B, 0x49279DEBBA320DFC,
				0x0FB612DEC3910CEB],
		]
		chi:   [
			[u64(0x9DD2DA9769DA819F), 0x3329889348E3611E, 0xF129D553258AC6C9, 0x604B94BA87023FFA,
				0xB09F4BD1D370E12C],
			[u64(0x3E2C73CC1842B6EE), 0x79822E30740F8A94, 0x2ADE13FD70A45485, 0x55660CE1001CC5F9,
				0x50A2A1D79F1EF2B3],
			[u64(0x11ACB1EB1AF15940), 0x274F2636A613DACD, 0x843EA2B8AFE2F08B, 0x7E78EA0CAEAF0B0F,
				0x2BA8D5E7DB313E6A],
			[u64(0xDACF89EA70FDDB42), 0x9B74448952711318, 0xDEF7504FF47BD92E, 0x084F23E7133A9197,
				0xCC0258A30644E223],
			[u64(0x76CD1ADA7BDDB391), 0x841DE7CDC8F8BA3D, 0xBFC5AE08ED82CF6B, 0x5C0794EBAA3E4DFD,
				0x0B96B2D8CB9B1E79],
		]
		iota:  [
			[u64(0x1DD2DA9769DA011F), 0x3329889348E3611E, 0xF129D553258AC6C9, 0x604B94BA87023FFA,
				0xB09F4BD1D370E12C],
			[u64(0x3E2C73CC1842B6EE), 0x79822E30740F8A94, 0x2ADE13FD70A45485, 0x55660CE1001CC5F9,
				0x50A2A1D79F1EF2B3],
			[u64(0x11ACB1EB1AF15940), 0x274F2636A613DACD, 0x843EA2B8AFE2F08B, 0x7E78EA0CAEAF0B0F,
				0x2BA8D5E7DB313E6A],
			[u64(0xDACF89EA70FDDB42), 0x9B74448952711318, 0xDEF7504FF47BD92E, 0x084F23E7133A9197,
				0xCC0258A30644E223],
			[u64(0x76CD1ADA7BDDB391), 0x841DE7CDC8F8BA3D, 0xBFC5AE08ED82CF6B, 0x5C0794EBAA3E4DFD,
				0x0B96B2D8CB9B1E79],
		]
	},
	// round 22
	RoundValues{
		theta: [
			[u64(0xD63F69D4510E2B36), 0xF8C43BD070374B37, 0x3AC466101D5EECE0, 0xABA627F9BFD615D3,
				0x7B72F892EBA4CB05],
			[u64(0xFF303EEC85BF4237), 0xB89E6310E9F27E4D, 0xEBC25EDDED59A05C, 0x947A41C19DE13120,
				0x91BEECF702E3066A],
			[u64(0xCF1B9E0D1E08C375), 0xF9F809D0A2EA40F8, 0x5A898D5EAB1B6ABE, 0xA0CFC5EAAA56913A,
				0xF51FFA01DFC8A45F],
			[u64(0x08CA693C6866B7A7), 0x4971A45F4AEA7FFD, 0x0CF2B099ECE0B5CB, 0xDA4AC3310BA1FD72,
				0x1E07B8751EDF8EC6],
			[u64(0x33C04CCA19D6216D), 0xC110B1DDAAF328C1, 0xFAC8F8188F895D97, 0x190AC2FBC835DF01,
				0x4E9BE4C8A9908C85],
		]
		rho:   [
			[u64(0xD63F69D4510E2B36), 0x0374B37F8C43BD07, 0xD6233080EAF76701, 0xAC2BA7574C4FF37F,
				0xE24BAE932C15EDCB],
			[u64(0xFE607DD90B7E846F), 0x27E4DB89E6310E9F, 0x097B77B5668173AF, 0x2624128F483833BC,
				0x46FBB3DC0B8C19AA],
			[u64(0x73C6E783478230DD), 0x7E027428BA903E3E, 0xDB55F2D44C6AF558, 0xE2F5552B489D5067,
				0xFEA3FF403BF9148B],
			[u64(0xC6866B7A708CA693), 0xFEA4B8D22FA5753F, 0x33D9C16B9619E561, 0x6621743FAE5B4958,
				0xC61E07B8751EDF8E],
			[u64(0x50CEB10B699E0266), 0x1DDAAF328C1C110B, 0xC4AECBFD647C0C47, 0x0AC2FBC835DF0119,
				0xF9322A64232153A6],
		]
		pi:    [
			[u64(0xD63F69D4510E2B36), 0xC6866B7A708CA693, 0xFE607DD90B7E846F, 0x50CEB10B699E0266,
				0x73C6E783478230DD],
			[u64(0x27E4DB89E6310E9F), 0x1DDAAF328C1C110B, 0x7E027428BA903E3E, 0x0374B37F8C43BD07,
				0xFEA4B8D22FA5753F],
			[u64(0xDB55F2D44C6AF558), 0xD6233080EAF76701, 0x33D9C16B9619E561, 0x097B77B5668173AF,
				0xC4AECBFD647C0C47],
			[u64(0x6621743FAE5B4958), 0x2624128F483833BC, 0x0AC2FBC835DF0119, 0xE2F5552B489D5067,
				0xAC2BA7574C4FF37F],
			[u64(0xF9322A64232153A6), 0xFEA3FF403BF9148B, 0xE24BAE932C15EDCB, 0xC61E07B8751EDF8E,
				0x46FBB3DC0B8C19AA],
		]
		chi:   [
			[u64(0x0E2E49805944DA76), 0x04A77BFA126FC093, 0xFFB9FC9A0F77452E, 0x58C5F58B0B1E40CE,
				0x73CCA4AE07DA389D],
			[u64(0x03C4DFA24420069F), 0x3DDEAD3D8C1401B7, 0x76004EA89B563E26, 0xE1F0B375845FBD47,
				0xD6A59CD027A68607],
			[u64(0x4247F8944D4AE7FE), 0x0EA0DDC0D9366302, 0xD3D0C5789E1909A3, 0x0D7175255383FC27,
				0x867EDB7567FC04C7],
			[u64(0x602C35AFFE556148), 0x262012B5083C91AC, 0x16E2AA8036B5013D, 0xF235E528401D5007,
				0x9D2FE354084DD32A],
			[u64(0xD8F2B86D8510572F), 0xE7FB7B40B7E90583, 0xE249AEB39C95D7DB, 0xC52E05CCF15F628F,
				0xCADBAB8C23A95C88],
		]
		iota:  [
			[u64(0x0E2E4980D944DA77), 0x04A77BFA126FC093, 0xFFB9FC9A0F77452E, 0x58C5F58B0B1E40CE,
				0x73CCA4AE07DA389D],
			[u64(0x03C4DFA24420069F), 0x3DDEAD3D8C1401B7, 0x76004EA89B563E26, 0xE1F0B375845FBD47,
				0xD6A59CD027A68607],
			[u64(0x4247F8944D4AE7FE), 0x0EA0DDC0D9366302, 0xD3D0C5789E1909A3, 0x0D7175255383FC27,
				0x867EDB7567FC04C7],
			[u64(0x602C35AFFE556148), 0x262012B5083C91AC, 0x16E2AA8036B5013D, 0xF235E528401D5007,
				0x9D2FE354084DD32A],
			[u64(0xD8F2B86D8510572F), 0xE7FB7B40B7E90583, 0xE249AEB39C95D7DB, 0xC52E05CCF15F628F,
				0xCADBAB8C23A95C88],
		]
	},
	// round 23
	RoundValues{
		theta: [
			[u64(0x2205AD7B44E8659B), 0x288C9F018FC37F7F, 0xD392186192DBFAC2, 0x74EE117096B2FF22,
				0x5FE740559A768771],
			[u64(0xF58DDC9FF08CCA78), 0xCB97AE0038B8CD50, 0x80494D952FFAF2C1, 0x17B9B04830F371A0,
				0x20EC9FED930A4AE0],
			[u64(0x42E1FCCBACC90058), 0x0E06D99F38B584A4, 0xD376C1277F9AEE05, 0x0DD7717AB2001B81,
				0x86D8DF2A867FE361],
			[u64(0xD17FFC6F397A6216), 0x9773DB75CF1392F2, 0xA7B16340F19A0263, 0x43662CE887325359,
				0x2C7C2A94CF62D074],
			[u64(0x5B750C009CAC6AE8), 0x647CCF2DAE553844, 0x61CE1ADE8529EA1C, 0x46A9B1A1E8E35F48,
				0x495C1FE13A15614F],
		]
		rho:   [
			[u64(0x2205AD7B44E8659B), 0xFC37F7F288C9F018, 0x9C90C30C96DFD616, 0x65FE44E9DC22E12D,
				0x015669DA1DC57F9D],
			[u64(0xEB1BB93FE11994F1), 0x8CD50CB97AE0038B, 0x253654BFEBCB0601, 0x6E3402F73609061E,
				0x83B27FB64C292B80],
			[u64(0x10B87F32EB324016), 0x81B667CE2D612903, 0xD7702E9BB6093BFC, 0xB8BD59000DC086EB,
				0x30DB1BE550CFFC6C],
			[u64(0xF397A6216D17FFC6), 0x794BB9EDBAE789C9, 0x81E33404C74F62C6, 0x9D10E64A6B286CC5,
				0x742C7C2A94CF62D0],
			[u64(0x04E5635742DBA860), 0xF2DAE553844647CC, 0x94F50E30E70D6F42, 0xA9B1A1E8E35F4846,
				0x07F84E855853D257],
		]
		pi:    [
			[u64(0x2205AD7B44E8659B), 0xF397A6216D17FFC6, 0xEB1BB93FE11994F1, 0x04E5635742DBA860,
				0x10B87F32EB324016],
			[u64(0x8CD50CB97AE0038B), 0xF2DAE553844647CC, 0x81B667CE2D612903, 0xFC37F7F288C9F018,
				0x794BB9EDBAE789C9],
			[u64(0xD7702E9BB6093BFC), 0x9C90C30C96DFD616, 0x81E33404C74F62C6, 0x253654BFEBCB0601,
				0x94F50E30E70D6F42],
			[u64(0x9D10E64A6B286CC5), 0x6E3402F73609061E, 0xA9B1A1E8E35F4846, 0xB8BD59000DC086EB,
				0x65FE44E9DC22E12D],
			[u64(0x07F84E855853D257), 0x30DB1BE550CFFC6C, 0x015669DA1DC57F9D, 0x742C7C2A94CF62D0,
				0x83B27FB64C292B80],
		]
		chi:   [
			[u64(0x71258F79C0E15DEF), 0xFF97A42D7F8E6FD4, 0xEB5AA93F2317D635, 0x05E5635A21D9AE61,
				0x940C7922AE3A2614],
			[u64(0x84D5CCF933C0478A), 0x90FEE5A0A44647C4, 0xA9A6E6260D712103, 0x64BEFEF28CC970F2,
				0x1841F924A2C509E4],
			[u64(0xD598261EA65AA9EE), 0x8C5BDA0CD6192E76, 0x81A57C16DBCF555F, 0x613670957BC46611,
				0x16F53526E70465C2],
			[u64(0xBD1547306F80494D), 0xAD30A6F71B19059C, 0x43B831CD0347C826, 0xB87C5A554FD00ECB,
				0x75F644E97F30A13B],
			[u64(0x8B284E056253D057), 0x30935AB7D08FFC64, 0x01F22F1A11A5569F, 0x8C3EE88A1CCF32C8,
				0xEAF1FF7B5CECA249],
		]
		iota:  [
			[u64(0xF1258F7940E1DDE7), 0xFF97A42D7F8E6FD4, 0xEB5AA93F2317D635, 0x05E5635A21D9AE61,
				0x940C7922AE3A2614],
			[u64(0x84D5CCF933C0478A), 0x90FEE5A0A44647C4, 0xA9A6E6260D712103, 0x64BEFEF28CC970F2,
				0x1841F924A2C509E4],
			[u64(0xD598261EA65AA9EE), 0x8C5BDA0CD6192E76, 0x81A57C16DBCF555F, 0x613670957BC46611,
				0x16F53526E70465C2],
			[u64(0xBD1547306F80494D), 0xAD30A6F71B19059C, 0x43B831CD0347C826, 0xB87C5A554FD00ECB,
				0x75F644E97F30A13B],
			[u64(0x8B284E056253D057), 0x30935AB7D08FFC64, 0x01F22F1A11A5569F, 0x8C3EE88A1CCF32C8,
				0xEAF1FF7B5CECA249],
		]
	},
]

const second_24 = [
	// round 0
	RoundValues{
		theta: [
			[u64(0xAF463273CA4D877D), 0xA1F41927F522354E, 0xB5391435A9BB8CAF, 0x5B86DE50AB75F4FB,
				0xCA6FC42824967C8E],
			[u64(0xAF9FDF84CEC209D0), 0xBBB4F6DD5944099E, 0x82ECF55BF0736F59, 0x4FF4ED8F71CB3EA8,
				0x330BEA595FC747BE],
			[u64(0x28C573DB9CDDA7BA), 0x71068FC9EC9E2022, 0x7CF829D3E1485B0B, 0x9C6B255041436845,
				0xEBA860E3DD836B96],
			[u64(0xABBCDA349E794C02), 0xBB993BF3EAE000D3, 0x5511ACC9F2BECD69, 0xAED5C751BE290B84,
				0x635FD9ED8EC9A474],
			[u64(0xFD3CB094025A23B6), 0x4687A426B0860F85, 0x77E6D18B71ACA57E, 0xFA2A161B7CC6C129,
				0x9CE501EA3CE551A8],
		]
		rho:   [
			[u64(0xAF463273CA4D877D), 0x522354EA1F41927F, 0xA9C8A1AD4DDC657D, 0xEBE9F6B70DBCA156,
				0x10A09259F23B29BF],
			[u64(0x5F3FBF099D8413A1), 0x4099EBBB4F6DD594, 0xB3D56FC1CDBD660B, 0x67D509FE9DB1EE39,
				0xCC2FA9657F1D1EF8],
			[u64(0x8A315CF6E73769EE), 0x41A3F27B2788089C, 0x42D85BE7C14E9F0A, 0x92A820A1B422CE35,
				0xDD750C1C7BB06D72],
			[u64(0x49E794C02ABBCDA3), 0x69DDCC9DF9F57000, 0x93E57D9AD2AA2359, 0xEA37C5217095DAB8,
				0x74635FD9ED8EC9A4],
			[u64(0xA012D11DB7E9E584), 0x426B0860F854687A, 0xD652BF3BF368C5B8, 0x2A161B7CC6C129FA,
				0x407A8F39546A2739],
		]
		pi:    [
			[u64(0xAF463273CA4D877D), 0x49E794C02ABBCDA3, 0x5F3FBF099D8413A1, 0xA012D11DB7E9E584,
				0x8A315CF6E73769EE],
			[u64(0x4099EBBB4F6DD594), 0x426B0860F854687A, 0x41A3F27B2788089C, 0x522354EA1F41927F,
				0x69DDCC9DF9F57000],
			[u64(0x42D85BE7C14E9F0A), 0xA9C8A1AD4DDC657D, 0x93E57D9AD2AA2359, 0xB3D56FC1CDBD660B,
				0xD652BF3BF368C5B8],
			[u64(0xEA37C5217095DAB8), 0x67D509FE9DB1EE39, 0x2A161B7CC6C129FA, 0x92A820A1B422CE35,
				0xEBE9F6B70DBCA156],
			[u64(0x407A8F39546A2739), 0xDD750C1C7BB06D72, 0x10A09259F23B29BF, 0x74635FD9ED8EC9A4,
				0xCC2FA9657F1D1EF8],
		]
		chi:   [
			[u64(0xAD0622374A4F8D77), 0xE067354D2F33C8A6, 0xCD7BB2894DA630E0, 0x01C6FA1C77558184,
				0x1C336FD4E53FEC56],
			[u64(0xE8BE6FBB7FFC9524), 0x047E00326875E27A, 0x69B1F01F23C9003E, 0x520B54CA2F431A4B,
				0x40748C19F5615046],
			[u64(0x429051FFC524BA0B), 0x31E8A5AD2FDC643F, 0x8345FD9BE290235C, 0xD79630998431678B,
				0xD254B67B8169DB10],
			[u64(0x4533F563FA905AFC), 0x6757993E9DBA6EB8, 0x6509367CCB453BFA, 0x12B8A0A5A643EA35,
				0xE9F9A2258D9EC050],
			[u64(0x00E346B1514A77B9), 0xDF7D043CABF44D2A, 0x1020D22BD03321A3, 0x26425B3BE58EDBDF,
				0xADE3296C67DD0EF8],
		]
		iota:  [
			[u64(0xAD0622374A4F8D76), 0xE067354D2F33C8A6, 0xCD7BB2894DA630E0, 0x01C6FA1C77558184,
				0x1C336FD4E53FEC56],
			[u64(0xE8BE6FBB7FFC9524), 0x047E00326875E27A, 0x69B1F01F23C9003E, 0x520B54CA2F431A4B,
				0x40748C19F5615046],
			[u64(0x429051FFC524BA0B), 0x31E8A5AD2FDC643F, 0x8345FD9BE290235C, 0xD79630998431678B,
				0xD254B67B8169DB10],
			[u64(0x4533F563FA905AFC), 0x6757993E9DBA6EB8, 0x6509367CCB453BFA, 0x12B8A0A5A643EA35,
				0xE9F9A2258D9EC050],
			[u64(0x00E346B1514A77B9), 0xDF7D043CABF44D2A, 0x1020D22BD03321A3, 0x26425B3BE58EDBDF,
				0xADE3296C67DD0EF8],
		]
	},
	// round 1
	RoundValues{
		theta: [
			[u64(0xC7054E4D3E5539BA), 0x8A6459375B297C6A, 0xA778DEF339BC842C, 0x6BC59666034F3548,
				0x763003AE9125589A],
			[u64(0x9EAE41D6DF2C0E21), 0x726E2E5FC8A5797F, 0x1FA1DE7283199B3B, 0x241B7AA78F93814E,
				0x3664A27455B1CB43],
			[u64(0xADC6A7F92422CCD1), 0xDEBE53ABCEDA12E5, 0x6C130B9D03965586, 0x38C0C69F65371151,
				0x3D02407D606FADCA],
			[u64(0x38F3BFABA61D8721), 0x1A97D3F6C137B365, 0x18C97CB497C8E627, 0x6F78EA6DFACE37E8,
				0x9439E8EDD1131D8D],
			[u64(0x87117E67A39863A7), 0x588F3CEA59265934, 0x97D2EAFD22E135BD, 0xA1B063ED175CCFC1,
				0x2A1111BA950F1AE6],
		]
		rho:   [
			[u64(0xC7054E4D3E5539BA), 0xB297C6A8A6459375, 0x3BC6F799CDE42165, 0x9E6A90D78B2CCC06,
				0x0EBA44956269D8C0],
			[u64(0x3D5C83ADBE581C43), 0x5797F726E2E5FC8A, 0x8779CA0C666CEC7E, 0x7029C4836F54F1F2,
				0xD99289D156C72D0C],
			[u64(0x6B71A9FE4908B334), 0xAF94EAF3B684B977, 0xB2AC3360985CE81C, 0x634FB29B88A89C60,
				0x47A0480FAC0DF5B9],
			[u64(0xBA61D872138F3BFA), 0xB28D4BE9FB609BD9, 0x692F91CC4E3192F9, 0x4DBF59C6FD0DEF1D,
				0x8D9439E8EDD1131D],
			[u64(0x3D1CC31D3C388BF3), 0xCEA59265934588F3, 0x709ADECBE9757E91, 0xB063ED175CCFC1A1,
				0x446EA543C6B98A84],
		]
		pi:    [
			[u64(0xC7054E4D3E5539BA), 0xBA61D872138F3BFA, 0x3D5C83ADBE581C43, 0x3D1CC31D3C388BF3,
				0x6B71A9FE4908B334],
			[u64(0x5797F726E2E5FC8A), 0xCEA59265934588F3, 0xAF94EAF3B684B977, 0xB297C6A8A6459375,
				0xB28D4BE9FB609BD9],
			[u64(0xB2AC3360985CE81C), 0x3BC6F799CDE42165, 0x692F91CC4E3192F9, 0x8779CA0C666CEC7E,
				0x709ADECBE9757E91],
			[u64(0x4DBF59C6FD0DEF1D), 0x7029C4836F54F1F2, 0xB063ED175CCFC1A1, 0x634FB29B88A89C60,
				0x9E6A90D78B2CCC06],
			[u64(0x446EA543C6B98A84), 0x47A0480FAC0DF5B9, 0x0EBA44956269D8C0, 0x8D9439E8EDD1131D,
				0xD99289D156C72D0C],
		]
		chi:   [
			[u64(0x672D4E0D264D39AE), 0x8B23BDEA5F2F1AFE, 0x7D7792A1F6691ECB, 0x3874CB197C10E7F9,
				0x2B633DFC491DD734],
			[u64(0x1A84BFA087E4FB8B), 0x8E8C9267B1555861, 0x3FD486E0A64AF877, 0xD291F63B2EC58375,
				0x3CED4BFDF9681BDF],
			[u64(0xB2EC97619AECE89C), 0x3C46FF954DED256C, 0x67B7914C6C118AB9, 0x0BE9C36C033DEF63,
				0x310AD7CBBDB65F99],
			[u64(0xCEBE13CAC549DE27), 0xC86854F37CD6FBB0, 0x81276E3FC0DFC5A2, 0x5347708E98801482,
				0xBC0BB0F982245E36],
			[u64(0x54FC146106194E84), 0x03244A0A2C4D75B8, 0x8C3A2CC762ED79F4, 0x0F173D486F940319,
				0x491ECBD0E4A725C5],
		]
		iota:  [
			[u64(0x672D4E0D264DB92C), 0x8B23BDEA5F2F1AFE, 0x7D7792A1F6691ECB, 0x3874CB197C10E7F9,
				0x2B633DFC491DD734],
			[u64(0x1A84BFA087E4FB8B), 0x8E8C9267B1555861, 0x3FD486E0A64AF877, 0xD291F63B2EC58375,
				0x3CED4BFDF9681BDF],
			[u64(0xB2EC97619AECE89C), 0x3C46FF954DED256C, 0x67B7914C6C118AB9, 0x0BE9C36C033DEF63,
				0x310AD7CBBDB65F99],
			[u64(0xCEBE13CAC549DE27), 0xC86854F37CD6FBB0, 0x81276E3FC0DFC5A2, 0x5347708E98801482,
				0xBC0BB0F982245E36],
			[u64(0x54FC146106194E84), 0x03244A0A2C4D75B8, 0x8C3A2CC762ED79F4, 0x0F173D486F940319,
				0x491ECBD0E4A725C5],
		]
	},
	// round 2
	RoundValues{
		theta: [
			[u64(0x7186E7FB6B6A5B56), 0x9D88141C1208F884, 0x6BDC3B57BB4EFCB1, 0x2EDF62EF31370583,
				0x3DC8940A043A354E],
			[u64(0x3F17F23D36D59938), 0xAB1FDFFA00643AD2, 0x1A47CB7D177B9AC4, 0xF702BBA69FF4E1C6,
				0x197E06604859796C],
			[u64(0x263753621A737EA9), 0xA89D3B96CD72B359, 0xF36C554FEC8E1C8C, 0x9F32076F83A27956,
				0xA5D113C83D29C9AC],
			[u64(0x2697F6BC47C6E1BD), 0x2041B185FE59C42A, 0x690E8B494250FA38, 0xBB6E95F81A0F2B18,
				0x5422558F00AB61AC],
			[u64(0x389CD25711F0FFAC), 0x6F448C3C3BA4C490, 0xE05AEAF17504C8DC, 0x6377FB7E787DB231,
				0x257E0DE6F34E94ED],
		]
		rho:   [
			[u64(0x7186E7FB6B6A5B56), 0x208F8849D88141C1, 0x5EE1DABDDA77E58B, 0x6E0B065DBEC5DE62,
				0x502810E8D538F722],
			[u64(0x7E2FE47A6DAB3270), 0x43AD2AB1FDFFA006, 0x1F2DF45DEE6B1069, 0x9C38DEE05774D3FE,
				0x65F819812165E5B0],
			[u64(0x498DD4D8869CDFAA), 0x274EE5B35CACD66A, 0x70E4679B62AA7F64, 0x03B7C1D13CAB4F99,
				0x94BA227907A53935],
			[u64(0xC47C6E1BD2697F6B), 0x151020D8C2FF2CE2, 0x9284A1F470D21D16, 0xBF0341E563176DD2,
				0xAC5422558F00AB61],
			[u64(0xB88F87FD61C4E692), 0xC3C3BA4C4906F448, 0x82646E702D7578BA, 0x77FB7E787DB23163,
				0x8379BCD3A53B495F],
		]
		pi:    [
			[u64(0x7186E7FB6B6A5B56), 0xC47C6E1BD2697F6B, 0x7E2FE47A6DAB3270, 0xB88F87FD61C4E692,
				0x498DD4D8869CDFAA],
			[u64(0x43AD2AB1FDFFA006), 0xC3C3BA4C4906F448, 0x274EE5B35CACD66A, 0x208F8849D88141C1,
				0x151020D8C2FF2CE2],
			[u64(0x70E4679B62AA7F64), 0x5EE1DABDDA77E58B, 0x9284A1F470D21D16, 0x1F2DF45DEE6B1069,
				0x82646E702D7578BA],
			[u64(0xBF0341E563176DD2), 0x9C38DEE05774D3FE, 0x77FB7E787DB23163, 0x03B7C1D13CAB4F99,
				0x6E0B065DBEC5DE62],
			[u64(0x8379BCD3A53B495F), 0x94BA227907A53935, 0x502810E8D538F722, 0xAC5422558F00AB61,
				0x65F819812165E5B0],
		]
		chi:   [
			[u64(0x41C6A2F1696A0436), 0xD85C2EAA40187EE8, 0xEEAFE43E4DF93B64, 0xA7AFF3E947AEF6BA,
				0xCBE99AF8AB9C8FB2],
			[u64(0xCCAE2AD5FCEAA094), 0x43DBBE0C4C06E63C, 0x4235BBBB518CF60B, 0x201D89C9C8010E51,
				0x791B20D5507FAAA2],
			[u64(0x709CDB89E6827F69), 0x5E63FAA4DAF6CD8A, 0x9284A174F0DADB16, 0xB36DD6596D6BB009,
				0x839477F02C55592A],
			[u64(0xCF8502CD29577FD2), 0xDC7C92E2873C95B4, 0x59FC9A6A55313133, 0x133C44795C6F0B0B,
				0x660EC205385DC468],
			[u64(0x8150B4D331AEE95F), 0x9739B23D0EA3B935, 0x51681169C53C3328, 0xAC542A551701AA20,
				0x71E839816106C5F0],
		]
		iota:  [
			[u64(0xC1C6A2F1696A84BC), 0xD85C2EAA40187EE8, 0xEEAFE43E4DF93B64, 0xA7AFF3E947AEF6BA,
				0xCBE99AF8AB9C8FB2],
			[u64(0xCCAE2AD5FCEAA094), 0x43DBBE0C4C06E63C, 0x4235BBBB518CF60B, 0x201D89C9C8010E51,
				0x791B20D5507FAAA2],
			[u64(0x709CDB89E6827F69), 0x5E63FAA4DAF6CD8A, 0x9284A174F0DADB16, 0xB36DD6596D6BB009,
				0x839477F02C55592A],
			[u64(0xCF8502CD29577FD2), 0xDC7C92E2873C95B4, 0x59FC9A6A55313133, 0x133C44795C6F0B0B,
				0x660EC205385DC468],
			[u64(0x8150B4D331AEE95F), 0x9739B23D0EA3B935, 0x51681169C53C3328, 0xAC542A551701AA20,
				0x71E839816106C5F0],
		]
	},
	// round 3
	RoundValues{
		theta: [
			[u64(0x73F6AA5E1760A08F), 0x6A6C26053E125ADB, 0x5C9FEC9133F31F57, 0x159FFB4639A4D289,
				0x79D99257D596AB81],
			[u64(0x4ED968406F721901), 0xC1ACFC99DF9E5FA9, 0xC042F92EC2144F9E, 0xA26ACB5C5B99B7C4,
				0xFB6C6240C3E71337],
			[u64(0x9AB54584A14C4355), 0xB44A64A99D38F1B6, 0x78AD3F79B714E72A, 0x594448542AA58C35,
				0x69BDE9FD6B9B6516],
			[u64(0x767D2B9ABCABE621), 0x6584BBB512C00C47, 0xE004B33DC0CDA8C0, 0xAAC46D2EC99392F8,
				0xDFF6EB52ADA15D9B],
			[u64(0x88813A03BFB58D18), 0x9EE83CED80B8DD72, 0x58B99FB94B27576F, 0xA585A485991ACE67,
				0x7839B751EF1DA1B7],
		]
		rho:   [
			[u64(0x73F6AA5E1760A08F), 0xE125ADB6A6C26053, 0xE4FF64899F98FABA, 0x49A5122B3FF68C73,
				0x495F565AAE05E766],
			[u64(0x9DB2D080DEE43202), 0xE5FA9C1ACFC99DF9, 0x0BE4BB08513E7B01, 0x36F8944D596B8B73,
				0xEDB189030F9C4CDF],
			[u64(0x66AD5161285310D5), 0x12992A674E3C6DAD, 0xA73953C569FBCDB8, 0x242A1552C61AACA2,
				0xCD37BD3FAD736CA2],
			[u64(0xABCABE621767D2B9), 0x23B2C25DDA896006, 0x7B819B5181C00966, 0xA5D932725F15588D,
				0x9BDFF6EB52ADA15D],
			[u64(0x1DFDAC68C44409D0), 0xCED80B8DD729EE83, 0x93ABB7AC5CCFDCA5, 0x85A485991ACE67A5,
				0x6DD47BC7686DDE0E],
		]
		pi:    [
			[u64(0x73F6AA5E1760A08F), 0xABCABE621767D2B9, 0x9DB2D080DEE43202, 0x1DFDAC68C44409D0,
				0x66AD5161285310D5],
			[u64(0xE5FA9C1ACFC99DF9), 0xCED80B8DD729EE83, 0x12992A674E3C6DAD, 0xE125ADB6A6C26053,
				0x23B2C25DDA896006],
			[u64(0xA73953C569FBCDB8), 0xE4FF64899F98FABA, 0x7B819B5181C00966, 0x0BE4BB08513E7B01,
				0x93ABB7AC5CCFDCA5],
			[u64(0xA5D932725F15588D), 0x36F8944D596B8B73, 0x85A485991ACE67A5, 0x242A1552C61AACA2,
				0x49A5122B3FF68C73],
			[u64(0x6DD47BC7686DDE0E), 0xCD37BD3FAD736CA2, 0x495F565AAE05E766, 0x9BDFF6EB52ADA15D,
				0xEDB189030F9C4CDF],
		]
		chi:   [
			[u64(0x71F7E99B3752E08F), 0x8BEDDA621FF7C281, 0xF4B241905F243240, 0x173DBE60957812D0,
				0xF6A464C12C158C74],
			[u64(0xE53ABC28D9CD8DFC), 0xDCD89BC9974AEFC2, 0x96BD2EEF54320B2C, 0xC52FA9E420C2E4F1,
				0x6BB6C25EF9B96054],
			[u64(0xEF3D1A4049934BBA), 0x2DF84DBB3B889E3A, 0x33DAC91325C18924, 0x903159A1419B7A5C,
				0x37BB3EAC5CC79C29],
			[u64(0xB7FBB26A4815780C), 0x1430960D4B6F196A, 0x110405194A2E77A5, 0x200A1D52425AA422,
				0x4BA9424B1FB59C73],
			[u64(0xE9DC6FC7A0E4C37E), 0x8927BCB26D7B40A0, 0x4B567C3DAE1DAACB, 0x7BDFF77D702FC15E,
				0xECA30B1FDD142CDD],
		]
		iota:  [
			[u64(0xF1F7E99BB752608F), 0x8BEDDA621FF7C281, 0xF4B241905F243240, 0x173DBE60957812D0,
				0xF6A464C12C158C74],
			[u64(0xE53ABC28D9CD8DFC), 0xDCD89BC9974AEFC2, 0x96BD2EEF54320B2C, 0xC52FA9E420C2E4F1,
				0x6BB6C25EF9B96054],
			[u64(0xEF3D1A4049934BBA), 0x2DF84DBB3B889E3A, 0x33DAC91325C18924, 0x903159A1419B7A5C,
				0x37BB3EAC5CC79C29],
			[u64(0xB7FBB26A4815780C), 0x1430960D4B6F196A, 0x110405194A2E77A5, 0x200A1D52425AA422,
				0x4BA9424B1FB59C73],
			[u64(0xE9DC6FC7A0E4C37E), 0x8927BCB26D7B40A0, 0x4B567C3DAE1DAACB, 0x7BDFF77D702FC15E,
				0xECA30B1FDD142CDD],
		]
	},
	// round 4
	RoundValues{
		theta: [
			[u64(0x4EAA7FD8FE767F77), 0x34B04C2156D3DD79, 0x4BEFD7D316002DB8, 0xA8602823DC5C0D28,
				0x49F9F2826531938C],
			[u64(0x2720E72A022CF6B4), 0x1EC2C0CB4CAB948A, 0x54A775ED8FD37064, 0x0735F2E6FB239FB9,
				0xA9AC995C22581B1C],
			[u64(0x5C23843AA32BFB28), 0x9EE6D3C1D1302EA8, 0x80C45769CF7939B6, 0x232FC7DBAB23CACE,
				0x84A5A0D6B67F2CBB],
			[u64(0x98CCEDDA9FE04BF0), 0x3B07C9BD9C9A2A96, 0x3E335AA99DDB4459, 0x0F3D42E295AF97DE,
				0x649E1DFBC840AF8F],
			[u64(0xEED340302987F038), 0x8E289345E41873E6, 0x4C5953CA277E998D, 0x7CD0D88AF94CF218,
				0xEBAC24E854771F9B],
		]
		rho:   [
			[u64(0x4EAA7FD8FE767F77), 0x6D3DD7934B04C215, 0x5F7EBE98B0016DC2, 0xB81A5150C05047B8,
				0xCA0994C64E3127E7],
			[u64(0x4E41CE540459ED68), 0xB948A1EC2C0CB4CA, 0x9DD7B63F4DC19152, 0x73F720E6BE5CDF64,
				0xA6B2657089606C72],
			[u64(0x1708E10EA8CAFECA), 0xB9B4F0744C0BAA27, 0xC9CDB40622BB4E7B, 0xE3EDD591E5671197,
				0x7094B41AD6CFE597],
			[u64(0xA9FE04BF098CCEDD), 0x4B1D83E4DECE4D15, 0x533BB688B27C66B5, 0x5C52B5F2FBC1E7A8,
				0x8F649E1DFBC840AF],
			[u64(0x814C3F81C7769A01), 0x345E41873E68E289, 0xBF4CC6A62CA9E513, 0xD0D88AF94CF2187C,
				0x093A151DC7E6FAEB],
		]
		pi:    [
			[u64(0x4EAA7FD8FE767F77), 0xA9FE04BF098CCEDD, 0x4E41CE540459ED68, 0x814C3F81C7769A01,
				0x1708E10EA8CAFECA],
			[u64(0xB948A1EC2C0CB4CA), 0x345E41873E68E289, 0xB9B4F0744C0BAA27, 0x6D3DD7934B04C215,
				0x4B1D83E4DECE4D15],
			[u64(0xC9CDB40622BB4E7B), 0x5F7EBE98B0016DC2, 0x533BB688B27C66B5, 0x9DD7B63F4DC19152,
				0xBF4CC6A62CA9E513],
			[u64(0x5C52B5F2FBC1E7A8), 0x73F720E6BE5CDF64, 0xD0D88AF94CF2187C, 0xE3EDD591E5671197,
				0xB81A5150C05047B8],
			[u64(0x093A151DC7E6FAEB), 0x7094B41AD6CFE597, 0xCA0994C64E3127E7, 0x8F649E1DFBC840AF,
				0xA6B2657089606C72],
		]
		chi:   [
			[u64(0x0E2F6BDAFCC53546), 0xE2DEBAA7898DC39F, 0x0C4AC8DCB62DA9F8, 0x118E1FADC3B78B43,
				0xA348A50C88EB5EC8],
			[u64(0xAD5AA01CF54C154A), 0x14DF41E1303470AD, 0x3974F8050089B26F, 0x0F159613EB22C290,
				0x4B0F92B41E9E4FBD],
			[u64(0xC8E5B40B269D5638), 0x5F7E2A80F0824D51, 0x593AA28EB07D4136, 0x91D7BC335749D17A,
				0xB9ECE2862589CD51],
			[u64(0x1AD2DF32C3D1E2BC), 0xFA9D2043B75CD52C, 0xD498C0E94CBAD074, 0xE3E5F411E1518B97,
				0xA912D15EE0DAD530],
			[u64(0xB87A9539C7EE7A63), 0x6494F51AE0AFC597, 0x7BBDA4E6063325E0, 0xE3555E0FF3C800BB,
				0xEEA76790DF646D67],
		]
		iota:  [
			[u64(0x0E2F6BDAFCC5B5CD), 0xE2DEBAA7898DC39F, 0x0C4AC8DCB62DA9F8, 0x118E1FADC3B78B43,
				0xA348A50C88EB5EC8],
			[u64(0xAD5AA01CF54C154A), 0x14DF41E1303470AD, 0x3974F8050089B26F, 0x0F159613EB22C290,
				0x4B0F92B41E9E4FBD],
			[u64(0xC8E5B40B269D5638), 0x5F7E2A80F0824D51, 0x593AA28EB07D4136, 0x91D7BC335749D17A,
				0xB9ECE2862589CD51],
			[u64(0x1AD2DF32C3D1E2BC), 0xFA9D2043B75CD52C, 0xD498C0E94CBAD074, 0xE3E5F411E1518B97,
				0xA912D15EE0DAD530],
			[u64(0xB87A9539C7EE7A63), 0x6494F51AE0AFC597, 0x7BBDA4E6063325E0, 0xE3555E0FF3C800BB,
				0xEEA76790DF646D67],
		]
	},
	// round 5
	RoundValues{
		theta: [
			[u64(0x2D58AC3E9181F74E), 0xC1A97D43E4C9811C, 0x2F3D0F38DB69EB7B, 0x32F9D849AEF3C9C0,
				0x803F62E8E5AF1C4B],
			[u64(0x3213C67C54319382), 0x8B9627819149F665, 0xA63D9E65A1F434A7, 0x905CF0734A5F4458,
				0xD446F4D4BFE3C975],
			[u64(0xF04E9CFA65A97E1B), 0x67D50271B3B66572, 0x61918A7FF3496915, 0xA97C94C2147DF959,
				0x8147CA7766BDE572],
			[u64(0xA90B4736CCCE4B59), 0x4944B847B8437CC9, 0x674158ED43A57991, 0x503C6C15EE4E2272,
				0x1ACB495AEFC57CD5],
			[u64(0x62A1C9EFEE20D762), 0xBE4FA9CCC9616896, 0xA166F8302FFD88E1, 0x398E02D9DA06ADBA,
				0x347C3B46F6AAC066],
		]
		rho:   [
			[u64(0x2D58AC3E9181F74E), 0x4C9811CC1A97D43E, 0x79E879C6DB4F5BD9, 0xE7938065F3B0935D,
				0x8BA396BC712E00FD],
			[u64(0x64278CF8A8632704), 0x9F6658B962781914, 0xF6799687D0D29E98, 0xE88B120B9E0E694B,
				0x511BD352FF8F25D7],
			[u64(0xFC13A73E996A5F86), 0xF5409C6CED995C99, 0x4B48AB0C8C53FF9A, 0x4A610A3EFCACD4BE,
				0x5028F94EECD7BCAE],
			[u64(0x6CCCE4B59A90B473), 0x64A4A25C23DC21BE, 0xDA874AF322CE82B1, 0x82BDC9C44E4A078D,
				0xD51ACB495AEFC57C],
			[u64(0x7F7106BB13150E4F), 0x9CCC9616896BE4FA, 0xFEC470D0B37C1817, 0x8E02D9DA06ADBA39,
				0x0ED1BDAAB0198D1F],
		]
		pi:    [
			[u64(0x2D58AC3E9181F74E), 0x6CCCE4B59A90B473, 0x64278CF8A8632704, 0x7F7106BB13150E4F,
				0xFC13A73E996A5F86],
			[u64(0x9F6658B962781914), 0x9CCC9616896BE4FA, 0xF5409C6CED995C99, 0x4C9811CC1A97D43E,
				0x64A4A25C23DC21BE],
			[u64(0x4B48AB0C8C53FF9A), 0x79E879C6DB4F5BD9, 0xDA874AF322CE82B1, 0xF6799687D0D29E98,
				0xFEC470D0B37C1817],
			[u64(0x82BDC9C44E4A078D), 0xE88B120B9E0E694B, 0x8E02D9DA06ADBA39, 0x4A610A3EFCACD4BE,
				0xE7938065F3B0935D],
			[u64(0x0ED1BDAAB0198D1F), 0x5028F94EECD7BCAE, 0x8BA396BC712E00FD, 0xD51ACB495AEFC57C,
				0x511BD352FF8F25D7],
		]
		chi:   [
			[u64(0x6D500F3A1D8211C4), 0x0DEC8D75C894AF72, 0x6EA0CE6BAA25A524, 0xCD1080B8D35504CF,
				0x6653F7BE094A4787],
			[u64(0x1FD3187920701911), 0x1CCF941F8D6BC4F8, 0xF1400D64E9B86491, 0x449819F436BB9418,
				0x65B72279635CA2F6],
			[u64(0x47089F263C427788), 0x69C89082BB9ECF7D, 0xDB264CD753CC8275, 0x636357C6D2919FD8,
				0xEECC23C2BF733C95],
			[u64(0xA3B5C9D04FCA75CD), 0xC44F16BA8C0E691A, 0xEA06D19A8EEC9D39, 0x60000E8CFDBCDEBD,
				0x4B93A449F3D0C95D],
			[u64(0x9CF7ED2BD261850F), 0xC028EB4CEDBCFC26, 0x1AE386B834B65864, 0xD592DA0D526D154C,
				0x51BFD312DD1B05EF],
		]
		iota:  [
			[u64(0x6D500F3A9D8211C5), 0x0DEC8D75C894AF72, 0x6EA0CE6BAA25A524, 0xCD1080B8D35504CF,
				0x6653F7BE094A4787],
			[u64(0x1FD3187920701911), 0x1CCF941F8D6BC4F8, 0xF1400D64E9B86491, 0x449819F436BB9418,
				0x65B72279635CA2F6],
			[u64(0x47089F263C427788), 0x69C89082BB9ECF7D, 0xDB264CD753CC8275, 0x636357C6D2919FD8,
				0xEECC23C2BF733C95],
			[u64(0xA3B5C9D04FCA75CD), 0xC44F16BA8C0E691A, 0xEA06D19A8EEC9D39, 0x60000E8CFDBCDEBD,
				0x4B93A449F3D0C95D],
			[u64(0x9CF7ED2BD261850F), 0xC028EB4CEDBCFC26, 0x1AE386B834B65864, 0xD592DA0D526D154C,
				0x51BFD312DD1B05EF],
		]
	},
	// round 6
	RoundValues{
		theta: [
			[u64(0x09A6F3E43B163F06), 0x691A71AB6E0081B1, 0x0A5632B50CB18BE7, 0xA9E67C6675C12A0C,
				0x02A50B60AFDE6944],
			[u64(0x4A1E4DB476B97250), 0x4902C1D2DBA2AFB9, 0xA48D58A9BF710FD0, 0x11554C396072FF59,
				0x307A77B43595C9B7],
			[u64(0xD8A46DC3AB8FD403), 0xF66462672C536CF6, 0x448ABE32C40121FE, 0xFCCFA523455C3C53,
				0x7160D12728BE9F1E],
			[u64(0x5FDFED26FE028FDD), 0x3825324C3DC6930A, 0x166CF56C3F246729, 0x9C6A2A7A4C7424AD,
				0xB7F980BF4218334D],
			[u64(0x70263F5ADB7DA2B6), 0x2CF9393DE4A0DB9F, 0xF63254C93DAA7FDD, 0x3943087C5B7132F5,
				0xBD6E0163D4072256],
		]
		rho:   [
			[u64(0x09A6F3E43B163F06), 0xE0081B1691A71AB6, 0x52B195A8658C5F38, 0x82541953CCF8CCEB,
				0x2D82BF79A5100A94],
			[u64(0x943C9B68ED72E4A0), 0x2AFB94902C1D2DBA, 0x3562A6FDC43F4292, 0x5FEB222AA9872C0E,
				0xC1E9DED0D65726DC],
			[u64(0xF6291B70EAE3F500), 0x991899CB14DB3DBD, 0x090FF22455F19620, 0xD291A2AE1E29FE67,
				0xCE2C1A24E517D3E3],
			[u64(0x6FE028FDD5FDFED2), 0x851C1299261EE349, 0xD87E48CE522CD9EA, 0x4F498E8495B38D45,
				0x4DB7F980BF421833],
			[u64(0xD6DBED15B38131FA), 0x93DE4A0DB9F2CF93, 0xD53FEEFB192A649E, 0x43087C5B7132F539,
				0x8058F501C895AF5B],
		]
		pi:    [
			[u64(0x09A6F3E43B163F06), 0x6FE028FDD5FDFED2, 0x943C9B68ED72E4A0, 0xD6DBED15B38131FA,
				0xF6291B70EAE3F500],
			[u64(0x2AFB94902C1D2DBA), 0x93DE4A0DB9F2CF93, 0x991899CB14DB3DBD, 0xE0081B1691A71AB6,
				0x851C1299261EE349],
			[u64(0x090FF22455F19620), 0x52B195A8658C5F38, 0xD87E48CE522CD9EA, 0x3562A6FDC43F4292,
				0xD53FEEFB192A649E],
			[u64(0x4F498E8495B38D45), 0x5FEB222AA9872C0E, 0x43087C5B7132F539, 0xD291A2AE1E29FE67,
				0x82541953CCF8CCEB],
			[u64(0x8058F501C895AF5B), 0xCE2C1A24E517D3E3, 0x2D82BF79A5100A94, 0x4DB7F980BF421833,
				0xC1E9DED0D65726DC],
		]
		chi:   [
			[u64(0x08A291C06AF6AD06), 0x2FC1BD5D91F1EEFA, 0xD45ADB6CAF5624E2, 0xC3B949FCF79971FA,
				0xA60AF712F3C3F196],
			[u64(0x6CBB9810AC1F24FF), 0x9E94680F31F1EF95, 0x9A18ADDA35C919AC, 0x22991B148BA7A6D3,
				0x875C0399E2CE6B28],
			[u64(0x891F83251DF5B43A), 0xD2B58DAC219C8CD9, 0xF4FCCBEED62CD36E, 0x3844FFFD657D4282,
				0x9496287B0B2D468A],
			[u64(0x46EF8C60A6B19D41), 0x7E2B02F3B96F001E, 0xD3347C5B39501119, 0x40D9A6BB1EA8DFAF,
				0xB4541873E4581DEB],
			[u64(0xA201F111CC9CAFE3), 0x5E325824CD15D2E2, 0x2482BFFAB5991389, 0x6DB7EB82BF641237,
				0xC0FDDE59D24B2495],
		]
		iota:  [
			[u64(0x88A291C0EAF62D87), 0x2FC1BD5D91F1EEFA, 0xD45ADB6CAF5624E2, 0xC3B949FCF79971FA,
				0xA60AF712F3C3F196],
			[u64(0x6CBB9810AC1F24FF), 0x9E94680F31F1EF95, 0x9A18ADDA35C919AC, 0x22991B148BA7A6D3,
				0x875C0399E2CE6B28],
			[u64(0x891F83251DF5B43A), 0xD2B58DAC219C8CD9, 0xF4FCCBEED62CD36E, 0x3844FFFD657D4282,
				0x9496287B0B2D468A],
			[u64(0x46EF8C60A6B19D41), 0x7E2B02F3B96F001E, 0xD3347C5B39501119, 0x40D9A6BB1EA8DFAF,
				0xB4541873E4581DEB],
			[u64(0xA201F111CC9CAFE3), 0x5E325824CD15D2E2, 0x2482BFFAB5991389, 0x6DB7EB82BF641237,
				0xC0FDDE59D24B2495],
		]
	},
	// round 7
	RoundValues{
		theta: [
			[u64(0x66BD3845B1554BD6), 0xC1DE14D8CA5288AB, 0x3A4572E9F4F542B3, 0x2DA6E079AC3A17AB,
				0x48155E97A86097C7],
			[u64(0x7D39F4CC743F9C06), 0x8F1604D3E9D1576C, 0x8B9AC106EDE9A155, 0x331B77C853871E2A,
				0x96DE6F453AEED3D1],
			[u64(0x7A175E6C64473703), 0x21BD50E5582E0FE0, 0x07F416A7AF9E5057, 0xCB4C22B41CCFC1BB,
				0x679EF532729FC5B3],
			[u64(0xAE9DD8A890DAC290), 0x9659563B8F045FCF, 0x3B4628930F3B4EC8, 0xA8ABF27328C3807E,
				0x5C264CBBD233423A],
			[u64(0x90682F2EB0F42E07), 0x6C5B861BB17D5306, 0x16EB61C5C9F1926D, 0x5FDE35BDC30C93D3,
				0xF2940066AE23A571],
		]
		rho:   [
			[u64(0x66BD3845B1554BD6), 0xA5288ABC1DE14D8C, 0xD22B974FA7AA1599, 0x742F565B4DC0F358,
				0x7A5EA1825F1D2055],
			[u64(0xFA73E998E87F380C), 0x1576C8F1604D3E9D, 0x6B041BB7A685562E, 0xE3C546636EF90A70,
				0x5B79BD14EBBB4F46],
			[u64(0xDE85D79B1911CDC0), 0x6F5439560B83F808, 0xF282B83FA0B53D7C, 0x115A0E67E0DDE5A6,
				0x6CF3DEA64E53F8B6],
			[u64(0x890DAC290AE9DD8A), 0xE7CB2CAB1DC7822F, 0x261E769D90768C51, 0x4E6518700FD5157E,
				0x3A5C264CBBD23342],
			[u64(0x7587A1703C834179), 0x61BB17D53066C5B8, 0xF8C9368B75B0E2E4, 0xDE35BDC30C93D35F,
				0x0019AB88E95C7CA5],
		]
		pi:    [
			[u64(0x66BD3845B1554BD6), 0x890DAC290AE9DD8A, 0xFA73E998E87F380C, 0x7587A1703C834179,
				0xDE85D79B1911CDC0],
			[u64(0x1576C8F1604D3E9D), 0x61BB17D53066C5B8, 0x6F5439560B83F808, 0xA5288ABC1DE14D8C,
				0xE7CB2CAB1DC7822F],
			[u64(0xF282B83FA0B53D7C), 0xD22B974FA7AA1599, 0x261E769D90768C51, 0x6B041BB7A685562E,
				0xF8C9368B75B0E2E4],
			[u64(0x4E6518700FD5157E), 0xE3C546636EF90A70, 0xDE35BDC30C93D35F, 0x115A0E67E0DDE5A6,
				0x742F565B4DC0F358],
			[u64(0x0019AB88E95C7CA5), 0x6CF3DEA64E53F8B6, 0x7A5EA1825F1D2055, 0x3A5C264CBBD23342,
				0x5B79BD14EBBB4F46],
		]
		chi:   [
			[u64(0x843D084B31E54AB6), 0x1B0D2C238D61CD8B, 0xFA79AF11780B3C5D, 0x3F83B0739E87535B,
				0xC685C59B7921AD00],
			[u64(0x1913C8B16F0D3E9F), 0x407F57F57837CFD8, 0xB775B0140702AB06, 0xB5728EFC5DB9EC0C,
				0xE3ED6CFB15879337],
			[u64(0xF29A1BB740BD55FD), 0xDE190FCBA7A8E51F, 0x0654769DC37AAC51, 0x41003BBFBD87446E,
				0xF3999F8FD78BEEE2],
			[u64(0x28C108351FD4162C), 0x62C9666A6E510F78, 0x5E14F5DBACF1CB57, 0x54D98F57E4DCA59F,
				0xF0AB14D05DC073D8],
			[u64(0x115B6B38A95448AC), 0x0C41CD727E55F886, 0x7F5AB1C45C9DE055, 0xBA742CC0BAB23FC6,
				0x7A339534EF7D4D69],
		]
		iota:  [
			[u64(0x043D084B31E5CABF), 0x1B0D2C238D61CD8B, 0xFA79AF11780B3C5D, 0x3F83B0739E87535B,
				0xC685C59B7921AD00],
			[u64(0x1913C8B16F0D3E9F), 0x407F57F57837CFD8, 0xB775B0140702AB06, 0xB5728EFC5DB9EC0C,
				0xE3ED6CFB15879337],
			[u64(0xF29A1BB740BD55FD), 0xDE190FCBA7A8E51F, 0x0654769DC37AAC51, 0x41003BBFBD87446E,
				0xF3999F8FD78BEEE2],
			[u64(0x28C108351FD4162C), 0x62C9666A6E510F78, 0x5E14F5DBACF1CB57, 0x54D98F57E4DCA59F,
				0xF0AB14D05DC073D8],
			[u64(0x115B6B38A95448AC), 0x0C41CD727E55F886, 0x7F5AB1C45C9DE055, 0xBA742CC0BAB23FC6,
				0x7A339534EF7D4D69],
		]
	},
	// round 8
	RoundValues{
		theta: [
			[u64(0xD7373C9F5FBAA29A), 0xC80718F7E33EA5AE, 0x29739BC516545478, 0xEC8984A7F0D83B7E,
				0x158FF14F177EC525],
			[u64(0x35C1BB82D0E397D2), 0x6CAD24C6C7D96695, 0x9BA7C327B8EC024B, 0x99A0FDCFE2574541,
				0xCF3F1FC8AA693A7A],
			[u64(0x2AC0D6E6D06B780E), 0x0643C29A377EC8EC, 0xDE0EBBCC53AC81A2, 0x995AF6EE2D51699D,
				0x2BC352DE475DC311],
			[u64(0xF4809211ED11E5B2), 0xBE88FC4E9C94FCE6, 0x82556FFF5E3438C9, 0x8898157316195601,
				0x2CEA8EF4AF058046],
			[u64(0x99AA96198B6FC68C), 0x84B030535C6E76A6, 0xF7AB4CE57EA66E75, 0x3285D1E19889B1E6,
				0xF2C26815CD46C349],
		]
		rho:   [
			[u64(0xD7373C9F5FBAA29A), 0x33EA5AEC80718F7E, 0x4B9CDE28B2A2A3C1, 0xB076FDD913094FE1,
				0xC53C5DFB1494563F],
			[u64(0x6B837705A1C72FA4), 0x966956CAD24C6C7D, 0x9F0C9EE3B0092E6E, 0xE8A833341FB9FC4A,
				0x3CFC7F22A9A4E9EB],
			[u64(0x8AB035B9B41ADE03), 0x90F0A68DDFB23B01, 0x640D16F075DE629D, 0x7B7716A8B4CECCAD,
				0x25786A5BC8EBB862],
			[u64(0x1ED11E5B2F480921), 0x735F447E274E4A7E, 0xFEBC68719304AADF, 0xAE62C32AC0311302,
				0x462CEA8EF4AF0580],
			[u64(0xCC5B7E3464CD54B0), 0x0535C6E76A684B03, 0x53373AFBD5A672BF, 0x85D1E19889B1E632,
				0x9A057351B0D27CB0],
		]
		pi:    [
			[u64(0xD7373C9F5FBAA29A), 0x1ED11E5B2F480921, 0x6B837705A1C72FA4, 0xCC5B7E3464CD54B0,
				0x8AB035B9B41ADE03],
			[u64(0x966956CAD24C6C7D), 0x0535C6E76A684B03, 0x90F0A68DDFB23B01, 0x33EA5AEC80718F7E,
				0x735F447E274E4A7E],
			[u64(0x640D16F075DE629D), 0x4B9CDE28B2A2A3C1, 0xFEBC68719304AADF, 0x9F0C9EE3B0092E6E,
				0x53373AFBD5A672BF],
			[u64(0xAE62C32AC0311302), 0xE8A833341FB9FC4A, 0x85D1E19889B1E632, 0x7B7716A8B4CECCAD,
				0xB076FDD913094FE1],
			[u64(0x9A057351B0D27CB0), 0x25786A5BC8EBB862, 0xC53C5DFB1494563F, 0x462CEA8EF4AF0580,
				0x3CFC7F22A9A4E9EB],
		]
		chi:   [
			[u64(0xB7333CAF7A28A01A), 0x54590653BFCAA9E1, 0x058F3F75A1C3AF7A, 0x405FFA3754C574B0,
				0x8A900F3864BAEE82],
			[u64(0x1C0B97C0526D7D7F), 0xA515E7F367711709, 0x91B12705D7037F21, 0x53995AE484B74FFF,
				0xD31F817E2547473E],
			[u64(0x740826A1451C0E2D), 0x4ECC966372E0A3E1, 0xBE9074128700BAD2, 0x9B0476E5F0282F6E,
				0x5FBF38D97D02D2B5],
			[u64(0xEB50CFA48F199108), 0xF229273438B9FD4B, 0xAF52C39C28F2CFB2, 0xF3240298B48E9C9D,
				0x3276FD40071359E1],
			[u64(0x9A4D3111309630D5), 0x245CAAFF88CBFA60, 0x554CDD734AA4463E, 0x758CEA46749F8ECE,
				0x4DB33F64AAE0E997],
		]
		iota:  [
			[u64(0xB7333CAF7A28A090), 0x54590653BFCAA9E1, 0x058F3F75A1C3AF7A, 0x405FFA3754C574B0,
				0x8A900F3864BAEE82],
			[u64(0x1C0B97C0526D7D7F), 0xA515E7F367711709, 0x91B12705D7037F21, 0x53995AE484B74FFF,
				0xD31F817E2547473E],
			[u64(0x740826A1451C0E2D), 0x4ECC966372E0A3E1, 0xBE9074128700BAD2, 0x9B0476E5F0282F6E,
				0x5FBF38D97D02D2B5],
			[u64(0xEB50CFA48F199108), 0xF229273438B9FD4B, 0xAF52C39C28F2CFB2, 0xF3240298B48E9C9D,
				0x3276FD40071359E1],
			[u64(0x9A4D3111309630D5), 0x245CAAFF88CBFA60, 0x554CDD734AA4463E, 0x758CEA46749F8ECE,
				0x4DB33F64AAE0E997],
		]
	},
	// round 9
	RoundValues{
		theta: [
			[u64(0x3402B648D170706F), 0xD7688CB41492791E, 0x86BEB5920A9B7F85, 0xC36E70D0FF9DA44F,
				0x09A185DFCFE23E7D],
			[u64(0xB1FE729E7D9E94CC), 0x08E002AD4882FEBA, 0x3C44C25BF8F09692, 0xFE6CBFBAAB44A64C,
				0x7EEA64200AB4AE8D],
			[u64(0x32D203A55F6DDEA1), 0x0816B3676891736D, 0xF84A51169D716A5E, 0xDDDE53E1EA59FFE2,
				0x19651DDD67730239],
			[u64(0x0D7A6236EBC2AC68), 0x14038AA65C62C02B, 0x49786E0E4C29F2D2, 0x150EAF0AD055A1FD,
				0xD45C50D263C86481],
			[u64(0xB56104C9B4E52E2A), 0x0B709F270CB8E49F, 0x7A60E8ABCED758C1, 0x5AA0DF9EF0EC9031,
				0x629F0ABC2E93F768],
		]
		rho:   [
			[u64(0x3402B648D170706F), 0x492791ED7688CB41, 0x35F5AC9054DBFC2C, 0x3B489F86DCE1A1FF,
				0x177F3F88F9F42686],
			[u64(0x63FCE53CFB3D2999), 0x2FEBA08E002AD488, 0x13096FE3C25A48F1, 0x94C99FCD97F75568,
				0xFBA990802AD2BA35],
			[u64(0x4CB480E957DB77A8), 0x05ACD9DA245CDB42, 0x8B52F7C25288B4EB, 0x29F0F52CFFF16EEF,
				0x232CA3BBACEE6047],
			[u64(0x6EBC2AC680D7A623), 0x158A01C5532E3160, 0x1C9853E5A492F0DC, 0xE15A0AB43FA2A1D5,
				0x81D45C50D263C864],
			[u64(0x4DA7297155AB0826), 0xF270CB8E49F0B709, 0x6BAC60BD307455E7, 0xA0DF9EF0EC90315A,
				0xC2AF0BA4FDDA18A7],
		]
		pi:    [
			[u64(0x3402B648D170706F), 0x6EBC2AC680D7A623, 0x63FCE53CFB3D2999, 0x4DA7297155AB0826,
				0x4CB480E957DB77A8],
			[u64(0x2FEBA08E002AD488), 0xF270CB8E49F0B709, 0x05ACD9DA245CDB42, 0x492791ED7688CB41,
				0x158A01C5532E3160],
			[u64(0x8B52F7C25288B4EB), 0x35F5AC9054DBFC2C, 0x1C9853E5A492F0DC, 0x13096FE3C25A48F1,
				0x6BAC60BD307455E7],
			[u64(0xE15A0AB43FA2A1D5), 0x94C99FCD97F75568, 0xA0DF9EF0EC90315A, 0x29F0F52CFFF16EEF,
				0x3B489F86DCE1A1FF],
			[u64(0xC2AF0BA4FDDA18A7), 0x232CA3BBACEE6047, 0x177F3F88F9F42686, 0x81D45C50D263C864,
				0xFBA990802AD2BA35],
		]
		chi:   [
			[u64(0xB412E10883F0500C), 0x6B390ED694DCEE07, 0x7BECE7197BBF0905, 0x5FAF4773D5F90896,
				0x2690E0D1778B332F],
			[u64(0x4FE3A8BA2D08D59C), 0x7278D8C3CAD4B649, 0xA5EB55CA6C5CDA40, 0x61D701E14B29ED4F,
				0x05CA9EC79FAF9178],
			[u64(0x89F7F6C292D0ACC9), 0x16D18CA27CD3DC2B, 0x0BB872EDB5F6F658, 0x930D67B3C258C8F1,
				0xAB0D60BD12664FE7],
			[u64(0xD55ABEFC3F82C19D), 0xD859978997E6D348, 0xC05F5EC4EE993843, 0x65D3D40DFA796EED,
				0x3F5C9FEF89E8E477],
			[u64(0xC9460B22FDD09C27), 0xB36C62B3E5CE714F, 0x137F274AFDB4F4C4, 0x81D4CCDCF0630B25,
				0xEAA391842AF6BA75],
		]
		iota:  [
			[u64(0xB412E10883F05084), 0x6B390ED694DCEE07, 0x7BECE7197BBF0905, 0x5FAF4773D5F90896,
				0x2690E0D1778B332F],
			[u64(0x4FE3A8BA2D08D59C), 0x7278D8C3CAD4B649, 0xA5EB55CA6C5CDA40, 0x61D701E14B29ED4F,
				0x05CA9EC79FAF9178],
			[u64(0x89F7F6C292D0ACC9), 0x16D18CA27CD3DC2B, 0x0BB872EDB5F6F658, 0x930D67B3C258C8F1,
				0xAB0D60BD12664FE7],
			[u64(0xD55ABEFC3F82C19D), 0xD859978997E6D348, 0xC05F5EC4EE993843, 0x65D3D40DFA796EED,
				0x3F5C9FEF89E8E477],
			[u64(0xC9460B22FDD09C27), 0xB36C62B3E5CE714F, 0x137F274AFDB4F4C4, 0x81D4CCDCF0630B25,
				0xEAA391842AF6BA75],
		]
	},
	// round 10
	RoundValues{
		theta: [
			[u64(0x4EEB87A102C2733D), 0x91C0687F15EECDBE, 0x811581B0FA8D2ABC, 0xA55621DA54CB2B2F,
				0xDC698678F6B91096],
			[u64(0xCB2718D8F47F5AFA), 0xF6BC68A113A3392F, 0x212FE5A8B52B5526, 0xE513B183925E6229,
				0x810E2EA546D81E1E],
			[u64(0x5A3C34F1A70F2972), 0xC51A4E91490C5990, 0xD873B0DE802973E3, 0x40C6A580F7874D4A,
				0x78C6A28E27B9CA5C],
			[u64(0x7D809679CA3691C9), 0x7083BF0C6252831C, 0x688576411B2D6817, 0xCD09FC880FCD3EB9,
				0x9786B76A7C5CB423],
			[u64(0xE56469BA557F2454), 0x9F4E002B4D61C93C, 0x3F5D45D2551B4CB7, 0xADF6AE4458CCB356,
				0xC681F31C82590206],
		]
		rho:   [
			[u64(0x4EEB87A102C2733D), 0x5EECDBE91C0687F1, 0x08AC0D87D46955E4, 0x96565F4AAC43B4A9,
				0x19E3DAE4425B71A6],
			[u64(0x964E31B1E8FEB5F5), 0x3392FF6BC68A113A, 0xBF96A2D4AD549884, 0xCC453CA27630724B,
				0x0438BA951B60787A],
			[u64(0x968F0D3C69C3CA5C), 0x4693A45243166431, 0x4B9F1EC39D86F401, 0x52C07BC3A6A52063,
				0x8F18D451C4F7394B],
			[u64(0x9CA3691C97D80967), 0x8E3841DF86312941, 0x82365AD02ED10AEC, 0x9101F9A7D739A13F,
				0x239786B76A7C5CB4],
			[u64(0xD2ABF922A72B234D), 0x02B4D61C93C9F4E0, 0x8DA65B9FAEA2E92A, 0xF6AE4458CCB356AD,
				0x7CC720964081B1A0],
		]
		pi:    [
			[u64(0x4EEB87A102C2733D), 0x9CA3691C97D80967, 0x964E31B1E8FEB5F5, 0xD2ABF922A72B234D,
				0x968F0D3C69C3CA5C],
			[u64(0x3392FF6BC68A113A), 0x02B4D61C93C9F4E0, 0x4693A45243166431, 0x5EECDBE91C0687F1,
				0x8E3841DF86312941],
			[u64(0x4B9F1EC39D86F401), 0x08AC0D87D46955E4, 0x82365AD02ED10AEC, 0xBF96A2D4AD549884,
				0x8DA65B9FAEA2E92A],
			[u64(0x9101F9A7D739A13F), 0xCC453CA27630724B, 0xF6AE4458CCB356AD, 0x52C07BC3A6A52063,
				0x96565F4AAC43B4A9],
			[u64(0x7CC720964081B1A0), 0x8F18D451C4F7394B, 0x19E3DAE4425B71A6, 0x239786B76A7C5CB4,
				0x0438BA951B60787A],
		]
		chi:   [
			[u64(0x06E687211BC6973C), 0x94AB609FD3F80863, 0x166A6B31C43FBF39, 0x73B9D936067B3B49,
				0x9709173C41410A76],
			[u64(0xA3921E4F84B31004), 0xC6F5E63CB1D9D6EB, 0x321BA05A83343030, 0x1EAC82EA1EA7A792,
				0x9C68459F86703DC0],
			[u64(0x27591ED39D06E481), 0x0BB4CDD654AE5CE4, 0x8B77C0742C992BEE, 0x9E8126E0E50CC410,
				0x8D8EFB0ABD82A178],
			[u64(0x93297E86D57BE322), 0xDCE615AE6538726F, 0x70A265496417D2FC, 0x82E802C323A6032A,
				0x04D15A62CCC036AD],
			[u64(0x4DD758DC8489B1A2), 0x8D0C4251C4F6CDCB, 0x59725EA6415B31A6, 0x2FD3847E7278D804,
				0x0C08FA569D50597B],
		]
		iota:  [
			[u64(0x06E687219BC61735), 0x94AB609FD3F80863, 0x166A6B31C43FBF39, 0x73B9D936067B3B49,
				0x9709173C41410A76],
			[u64(0xA3921E4F84B31004), 0xC6F5E63CB1D9D6EB, 0x321BA05A83343030, 0x1EAC82EA1EA7A792,
				0x9C68459F86703DC0],
			[u64(0x27591ED39D06E481), 0x0BB4CDD654AE5CE4, 0x8B77C0742C992BEE, 0x9E8126E0E50CC410,
				0x8D8EFB0ABD82A178],
			[u64(0x93297E86D57BE322), 0xDCE615AE6538726F, 0x70A265496417D2FC, 0x82E802C323A6032A,
				0x04D15A62CCC036AD],
			[u64(0x4DD758DC8489B1A2), 0x8D0C4251C4F6CDCB, 0x59725EA6415B31A6, 0x2FD3847E7278D804,
				0x0C08FA569D50597B],
		]
	},
	// round 11
	RoundValues{
		theta: [
			[u64(0x17E5839A28D8029E), 0x85A8642460E61DC8, 0x07696F8A7721AA92, 0x62BADD8DB5652EE2,
				0x860A1387F25F1FDD],
			[u64(0xAA2EC1FD34F76C93), 0xCF49398E019DAA7C, 0x3BA77FE833704CA7, 0x17105D58AEE3DB05,
				0x95D49A2D36344157],
			[u64(0x80492C0FC5EB6461), 0xACA4FF0A0C43DC04, 0x2C67F2A87474AB0E, 0x3991143CBDE144F0,
				0x2A9EC9D6E56F2198],
			[u64(0x5358C41AB4DD8CA0), 0x1C97AF32049E1DED, 0xB0D3DFD505B1BD7E, 0x4299B85F42006CA8,
				0xC4A0E0FEAD66592F],
			[u64(0x35AD8B1729CCE534), 0xF576919A69B3995D, 0x21088D6DEC1E6530, 0x57A957B5DF3D8C92,
				0x7472299D30150DED],
		]
		rho:   [
			[u64(0x17E5839A28D8029E), 0x0E61DC885A864246, 0x3B4B7C53B90D5490, 0xCA5DC4C575BB1B6A,
				0x4E1FC97C7F761828],
			[u64(0x545D83FA69EED927), 0xDAA7CCF49398E019, 0x9DFFA0CDC1329CEE, 0x7B60A2E20BAB15DC,
				0x575268B4D8D1055E],
			[u64(0x60124B03F17AD918), 0x293FC28310F7012B, 0xA55871633F9543A3, 0x8A1E5EF0A2781CC8,
				0x0553D93ADCADE433],
			[u64(0xAB4DD8CA05358C41), 0xF68E4BD799024F0E, 0xAA0B637AFD61A7BF, 0x0BE8400D95085337,
				0x2FC4A0E0FEAD6659],
			[u64(0xB94E6729A1AD6C58), 0x19A69B3995DF5769, 0x0F3298108446B6F6, 0xA957B5DF3D8C9257,
				0x8A674C05437B5D1C],
		]
		pi:    [
			[u64(0x17E5839A28D8029E), 0xAB4DD8CA05358C41, 0x545D83FA69EED927, 0xB94E6729A1AD6C58,
				0x60124B03F17AD918],
			[u64(0xDAA7CCF49398E019), 0x19A69B3995DF5769, 0x293FC28310F7012B, 0x0E61DC885A864246,
				0xF68E4BD799024F0E],
			[u64(0xA55871633F9543A3), 0x3B4B7C53B90D5490, 0xAA0B637AFD61A7BF, 0x9DFFA0CDC1329CEE,
				0x0F3298108446B6F6],
			[u64(0x0BE8400D95085337), 0x7B60A2E20BAB15DC, 0xA957B5DF3D8C9257, 0x8A1E5EF0A2781CC8,
				0xCA5DC4C575BB1B6A],
			[u64(0x8A674C05437B5D1C), 0x0553D93ADCADE433, 0x4E1FC97C7F761828, 0x2FC4A0E0FEAD6659,
				0x575268B4D8D1055E],
		]
		chi:   [
			[u64(0x32BDB29904DD013C), 0x8904BC882D358CD1, 0xD65DA28284EE7FB3, 0x28D0476C209DF0F0,
				0x6922DB03F53E69E8],
			[u64(0xD007CCF81390F00D), 0x59861999977D5625, 0x286B5606107B116B, 0x0C6182B878CE4246,
				0x36C30F12E8BB4606],
			[u64(0x255F7D637DE64FAB), 0x3F58254B6D09B4B3, 0xEC032B5ABF13AF97, 0xB83F00CD9DB7FEFF,
				0x1A30B0200C06B2E2],
			[u64(0x1E68C397BD8851B5), 0xD16CA2220ABB1D9C, 0xB917B75D3D045350, 0x1A1419F9A37814C8,
				0xEA5DC7C65491C36A],
			[u64(0x42650061D07BBD1D), 0x15F1DA0B4C67B71B, 0x673D897D6F671820, 0x29E53860A4AF645F,
				0xC1DE6860D0D10358],
		]
		iota:  [
			[u64(0x32BDB29984DD0136), 0x8904BC882D358CD1, 0xD65DA28284EE7FB3, 0x28D0476C209DF0F0,
				0x6922DB03F53E69E8],
			[u64(0xD007CCF81390F00D), 0x59861999977D5625, 0x286B5606107B116B, 0x0C6182B878CE4246,
				0x36C30F12E8BB4606],
			[u64(0x255F7D637DE64FAB), 0x3F58254B6D09B4B3, 0xEC032B5ABF13AF97, 0xB83F00CD9DB7FEFF,
				0x1A30B0200C06B2E2],
			[u64(0x1E68C397BD8851B5), 0xD16CA2220ABB1D9C, 0xB917B75D3D045350, 0x1A1419F9A37814C8,
				0xEA5DC7C65491C36A],
			[u64(0x42650061D07BBD1D), 0x15F1DA0B4C67B71B, 0x673D897D6F671820, 0x29E53860A4AF645F,
				0xC1DE6860D0D10358],
		]
	},
	// round 12
	RoundValues{
		theta: [
			[u64(0xDCBFAC140A1F1210), 0x6706A205A3F79FF7, 0x385FBC0F0A2C6C95, 0xC6D259E1AE5FE3D6,
				0x8720C58E7BFC7ACE],
			[u64(0x54067B3A97AFAA65), 0xDD87AE5B13420C4D, 0xAC6AE1C494444B03, 0x8860357AFCF1182E,
				0xB2C2B8D06C841C6E],
			[u64(0xB2A3620082B86D1F), 0xA8A43A2892579607, 0x7BFF3439404D8D23, 0x2FC31FAE62E9DC4B,
				0x8DCCAF43F3589056],
			[u64(0xFB4706278DCFA364), 0x344367923AFCEF4D, 0x5C3872ED0D43A181, 0xFF3BDC49933FE619,
				0x0F72027664D631BB],
			[u64(0x9C13694F5CEFA35E), 0xCB87B325C0F3A958, 0xB94BE053E3F30663, 0xF793514E283B7A1C,
				0x1FA8014E5C451D1B],
		]
		rho:   [
			[u64(0xDCBFAC140A1F1210), 0x3F79FF76706A205A, 0xC2FDE078516364A9, 0xBFC7AD8DA4B3C35C,
				0x1639EFF1EB3A1C83],
			[u64(0xA80CF6752F5F54CA), 0x20C4DDD87AE5B134, 0xAB871251112C0EB1, 0x2305D10C06AF5F9E,
				0xCB0AE341B21071BA],
			[u64(0xECA8D88020AE1B47), 0x290E8A2495E581EA, 0x6C691BDFF9A1CA02, 0x8FD73174EE2597E1,
				0xD1B995E87E6B120A],
			[u64(0x78DCFA364FB47062), 0xA69A21B3C91D7E77, 0xDA1A874302B870E5, 0x893267FCC33FE77B,
				0xBB0F72027664D631],
			[u64(0x7AE77D1AF4E09B4A), 0x325C0F3A958CB87B, 0xF98331DCA5F029F1, 0x93514E283B7A1CF7,
				0x005397114746C7EA],
		]
		pi:    [
			[u64(0xDCBFAC140A1F1210), 0x78DCFA364FB47062, 0xA80CF6752F5F54CA, 0x7AE77D1AF4E09B4A,
				0xECA8D88020AE1B47],
			[u64(0x20C4DDD87AE5B134), 0x325C0F3A958CB87B, 0x290E8A2495E581EA, 0x3F79FF76706A205A,
				0xA69A21B3C91D7E77],
			[u64(0x6C691BDFF9A1CA02), 0xC2FDE078516364A9, 0xDA1A874302B870E5, 0xAB871251112C0EB1,
				0xF98331DCA5F029F1],
			[u64(0x893267FCC33FE77B), 0x2305D10C06AF5F9E, 0x93514E283B7A1CF7, 0x8FD73174EE2597E1,
				0xBFC7AD8DA4B3C35C],
			[u64(0x005397114746C7EA), 0xD1B995E87E6B120A, 0x1639EFF1EB3A1C83, 0xBB0F72027664D631,
				0xCB0AE341B21071BA],
		]
		chi:   [
			[u64(0x9096AE138B1F5812), 0xB87D1A760FD734E2, 0x7A1CF3362D4724CF, 0xFA617D1BF5E495EB,
				0xB5A9C8CC044E1AC7],
			[u64(0xA1D6B9F878FB944D), 0x135C1E3E9300A36D, 0x284FC20CACA78DF8, 0x3B29DE529E6BB11A,
				0xA0DEADB2C91EBC7B],
			[u64(0x6C288BDEFDE1CA82), 0x1245E498292364A9, 0xDE322692C2B870E5, 0x9B8F5053016C4EA1,
				0xB98B739CB7F01953],
			[u64(0x559E4FF8CB26F76B), 0x0B41BB1A073B3FFE, 0x3B555E2C3F3F5CBF, 0xCF373C6C6EA59EAB,
				0x9B67B50DA41DC919],
			[u64(0x2013C6D937A666CE), 0xD3B990E0EE639A13, 0x173BE7F17B9A9DA3, 0xBE17F066766EF621,
				0xC918C2727B01158A],
		]
		iota:  [
			[u64(0x9096AE130B1FD899), 0xB87D1A760FD734E2, 0x7A1CF3362D4724CF, 0xFA617D1BF5E495EB,
				0xB5A9C8CC044E1AC7],
			[u64(0xA1D6B9F878FB944D), 0x135C1E3E9300A36D, 0x284FC20CACA78DF8, 0x3B29DE529E6BB11A,
				0xA0DEADB2C91EBC7B],
			[u64(0x6C288BDEFDE1CA82), 0x1245E498292364A9, 0xDE322692C2B870E5, 0x9B8F5053016C4EA1,
				0xB98B739CB7F01953],
			[u64(0x559E4FF8CB26F76B), 0x0B41BB1A073B3FFE, 0x3B555E2C3F3F5CBF, 0xCF373C6C6EA59EAB,
				0x9B67B50DA41DC919],
			[u64(0x2013C6D937A666CE), 0xD3B990E0EE639A13, 0x173BE7F17B9A9DA3, 0xBE17F066766EF621,
				0xC918C2727B01158A],
		]
	},
	// round 13
	RoundValues{
		theta: [
			[u64(0x016C019B847C353E), 0x2987B5FE80B4D945, 0xEBE65CBEA224C968, 0x6B9BD2937A87784C,
				0x245367448B2DF760],
			[u64(0xB85F9F4BE113C1AC), 0x0AD5388D0AE8F68C, 0x31C6E4BF354FD819, 0x22A0F8E10783E4FB,
				0xB9578B0150F6E99A],
			[u64(0x0EAEDAAA9EFDFA0B), 0x70C3B5EC4A3F5420, 0xBCB477E6A1A4406C, 0xF909012762707E28,
				0xDB0D22E8D4EC29DA],
			[u64(0xF0F8225A35A17BFC), 0xAE27D6B8F9BCB369, 0x9E33338EC1B8D028, 0x6A5151CE9022123C,
				0x3E01D8AF5A9A458E],
			[u64(0x2BB6007FBE762A66), 0xD81C564667B3D6BB, 0x1C9E2157F24AD10B, 0xB5B236C0FFBEBA89,
				0xC2BD04D4F2D15922],
		]
		rho:   [
			[u64(0x016C019B847C353E), 0x0B4D9452987B5FE8, 0x5F32E5F511264B47, 0x0EF098D737A526F5,
				0x9D122CB7DD80914D],
			[u64(0x70BF3E97C2278359), 0x8F68C0AD5388D0AE, 0x1B92FCD53F6064C7, 0x7C9F64541F1C20F0,
				0xE55E2C0543DBA66A],
			[u64(0xC3ABB6AAA7BF7E82), 0x30ED7B128FD5081C, 0x220365E5A3BF350D, 0x8093B1383F147C84,
				0x5B61A45D1A9D853B],
			[u64(0xA35A17BFCF0F8225), 0xB4D713EB5C7CDE59, 0x1D8371A0513C6667, 0x39D20442478D4A2A,
				0x8E3E01D8AF5A9A45],
			[u64(0xFDF3B153315DB003), 0x64667B3D6BBD81C5, 0x2568858E4F10ABF9, 0xB236C0FFBEBA89B5,
				0x41353CB45648B0AF],
		]
		pi:    [
			[u64(0x016C019B847C353E), 0xA35A17BFCF0F8225, 0x70BF3E97C2278359, 0xFDF3B153315DB003,
				0xC3ABB6AAA7BF7E82],
			[u64(0x8F68C0AD5388D0AE), 0x64667B3D6BBD81C5, 0x30ED7B128FD5081C, 0x0B4D9452987B5FE8,
				0xB4D713EB5C7CDE59],
			[u64(0x220365E5A3BF350D), 0x5F32E5F511264B47, 0x1D8371A0513C6667, 0x1B92FCD53F6064C7,
				0x2568858E4F10ABF9],
			[u64(0x39D20442478D4A2A), 0x7C9F64541F1C20F0, 0xB236C0FFBEBA89B5, 0x8093B1383F147C84,
				0x0EF098D737A526F5],
			[u64(0x41353CB45648B0AF), 0x5B61A45D1A9D853B, 0x9D122CB7DD80914D, 0x8E3E01D8AF5A9A45,
				0xE55E2C0543DBA66A],
		]
		chi:   [
			[u64(0x216F24DB244B103F), 0xB84A937FDF0DC827, 0x7DBD3E37920FE53A, 0xED61D9D6165D9004,
				0xC28332AEA4BF5F22],
			[u64(0x96B8C0AF17889A8C), 0x44EB7B3D65A5A175, 0x92D9FB4D2157818C, 0x8B4C957A986F47E8,
				0xBE470BBA6CD9DA5D],
			[u64(0x62265D51B3FF8588), 0x5C5265FC11A7CE4C, 0x10835DA0103C762F, 0x15BEFC15BF2AE686,
				0xC466A18E0F4A2BF3],
			[u64(0x399A0549C7B94F3A), 0xDC8577F6DA1E22F4, 0xD29BD2FFBC9D8BA5, 0xF152013B2F115C86,
				0x0C510A7D93817E75],
			[u64(0xCF35FC9005C8702F), 0x1F45CC5D3A2D84FB, 0x9D526DB7D0509949, 0x8C3205D82778D5AD,
				0xD10A2D441B9B2633],
		]
		iota:  [
			[u64(0xA16F24DB244B10B4), 0xB84A937FDF0DC827, 0x7DBD3E37920FE53A, 0xED61D9D6165D9004,
				0xC28332AEA4BF5F22],
			[u64(0x96B8C0AF17889A8C), 0x44EB7B3D65A5A175, 0x92D9FB4D2157818C, 0x8B4C957A986F47E8,
				0xBE470BBA6CD9DA5D],
			[u64(0x62265D51B3FF8588), 0x5C5265FC11A7CE4C, 0x10835DA0103C762F, 0x15BEFC15BF2AE686,
				0xC466A18E0F4A2BF3],
			[u64(0x399A0549C7B94F3A), 0xDC8577F6DA1E22F4, 0xD29BD2FFBC9D8BA5, 0xF152013B2F115C86,
				0x0C510A7D93817E75],
			[u64(0xCF35FC9005C8702F), 0x1F45CC5D3A2D84FB, 0x9D526DB7D0509949, 0x8C3205D82778D5AD,
				0xD10A2D441B9B2633],
		]
	},
	// round 14
	RoundValues{
		theta: [
			[u64(0x5A76ED02B885C137), 0x43535AA643C319A4, 0x86A4F7EE0EC134B9, 0x1678100F8A934187,
				0x399AFB7738718EA1],
			[u64(0x239CD368C82A893E), 0xF1CF68FABA07B2C7, 0x27FDE88AFEF5923E, 0x3E6886BD47CD545A,
				0x0B63187DB37BC9EF],
			[u64(0x82A8D5422F662B79), 0xBCDCEDEF8D3E60BD, 0xF00DD5B38CA5D8DE, 0xF530740623B34877,
				0x24E8299D93D38502],
			[u64(0xE681D612631083A2), 0x039EA4AD7EB7EE6C, 0x0D8001A41834473D, 0x2E49D2608BB8901E,
				0xD34AD9263728B2ED],
			[u64(0x93469241AF3551A9), 0x4336A28C90D0A57D, 0xC12103667AADB8CF, 0xD0416B098D85F42B,
				0x8D794395B16607B5],
		]
		rho:   [
			[u64(0x5A76ED02B885C137), 0x3C319A443535AA64, 0x3527BF707609A5CC, 0x26830E2CF0201F15,
				0xEDDCE1C63A84E66B],
			[u64(0x4739A6D19055127C), 0x7B2C7F1CF68FABA0, 0xF7A22BFBD648F89F, 0xAA8B47CD10D7A8F9,
				0x2D8C61F6CDEF27BC],
			[u64(0x60AA35508BD98ADE), 0x373B7BE34F982F6F, 0x2EC6F7806EAD9C65, 0x3A0311D9A43BFA98,
				0x449D0533B27A70A0],
			[u64(0x2631083A2E681D61), 0x3601CF5256BF5BF7, 0x4830688E7A1B0003, 0x4C11771203C5C93A,
				0xEDD34AD9263728B2],
			[u64(0x0D79AA8D4C9A3492), 0x28C90D0A57D4336A, 0x56DC67E09081B33D, 0x416B098D85F42BD0,
				0x50E56C5981ED635E],
		]
		pi:    [
			[u64(0x5A76ED02B885C137), 0x2631083A2E681D61, 0x4739A6D19055127C, 0x0D79AA8D4C9A3492,
				0x60AA35508BD98ADE],
			[u64(0x7B2C7F1CF68FABA0), 0x28C90D0A57D4336A, 0x373B7BE34F982F6F, 0x3C319A443535AA64,
				0x3601CF5256BF5BF7],
			[u64(0x2EC6F7806EAD9C65), 0x3527BF707609A5CC, 0x4830688E7A1B0003, 0xF7A22BFBD648F89F,
				0x56DC67E09081B33D],
			[u64(0x4C11771203C5C93A), 0xAA8B47CD10D7A8F9, 0x416B098D85F42BD0, 0x3A0311D9A43BFA98,
				0x26830E2CF0201F15],
			[u64(0x50E56C5981ED635E), 0x449D0533B27A70A0, 0xEDDCE1C63A84E66B, 0xEDD34AD9263728B2,
				0x2D8C61F6CDEF27BC],
		]
		chi:   [
			[u64(0x5EB46D82B0A5D572), 0x3317BA4A0E6199E5, 0x0F39A6DDA056127C, 0xCEFB8B368ED26409,
				0x207615F00BD92AD6],
			[u64(0x3B3D7F0EF7CFEABA), 0xA2414D8757023B5B, 0x36707AE2CA7C04BF, 0x34308A441506A864,
				0x1602C75E369F57F7],
			[u64(0x3E22FFC9EE85BE21), 0x7133BF42D421F5CC, 0xE4A488CC401BC428, 0x327261FBD44CF8BD,
				0x5FD006329D4E9395],
			[u64(0x4603F6103BC5491B), 0x88AB4FC51CD7A5B8, 0x434A0F9C05A53BC4, 0x3A2BB1DDECB3EE98,
				0x66A11A2CF2309757],
			[u64(0x71ED7E45C7E749DE), 0x4C550033E3EE52AA, 0xDDDEB8E4750CCB68, 0xDDD35A991712A2D6,
				0x3B8DABF499C9769D],
		]
		iota:  [
			[u64(0xDEB46D82B0A555FB), 0x3317BA4A0E6199E5, 0x0F39A6DDA056127C, 0xCEFB8B368ED26409,
				0x207615F00BD92AD6],
			[u64(0x3B3D7F0EF7CFEABA), 0xA2414D8757023B5B, 0x36707AE2CA7C04BF, 0x34308A441506A864,
				0x1602C75E369F57F7],
			[u64(0x3E22FFC9EE85BE21), 0x7133BF42D421F5CC, 0xE4A488CC401BC428, 0x327261FBD44CF8BD,
				0x5FD006329D4E9395],
			[u64(0x4603F6103BC5491B), 0x88AB4FC51CD7A5B8, 0x434A0F9C05A53BC4, 0x3A2BB1DDECB3EE98,
				0x66A11A2CF2309757],
			[u64(0x71ED7E45C7E749DE), 0x4C550033E3EE52AA, 0xDDDEB8E4750CCB68, 0xDDD35A991712A2D6,
				0x3B8DABF499C9769D],
		]
	},
	// round 15
	RoundValues{
		theta: [
			[u64(0xC2F0509FFD2B0437), 0x2F53875743EFC829, 0x137D9BC0EDD843B0, 0xD2BFB62BC35C35C5,
				0x3C3228ED46577B1A],
			[u64(0xBB05CFC10B2DB3DC), 0x2279FD48ABE0623D, 0xB648CA2D369E5DD9, 0xB4083A8BE9E4F102,
				0x963A7791CA7D0E91],
			[u64(0x11CCC1C8DFC4C9BD), 0x5EDD8143E5608250, 0xCB4AB6CD715AB3B4, 0x1D9C5FFAE50D8F21,
				0x703E3833AC0FE409],
			[u64(0x8C643661B7C4A558), 0x42CC8FB490D649FB, 0x892DCFED89A4D787, 0xF04C71AC60B202DB,
				0xACC6DA5D7E317B14],
			[u64(0xB8AABC5ACCE0C60C), 0x8512C22CE8E9DD78, 0x14997AFB7E0B44BA, 0x149498861C152D04,
				0xF2CA69EB92CEF94F],
		]
		rho:   [
			[u64(0xC2F0509FFD2B0437), 0x3EFC8292F5387574, 0x9BECDE076EC21D80, 0xB86B8BA57F6C5786,
				0xA3B5195DEC68F0C8],
			[u64(0x760B9F82165B67B9), 0x0623D2279FD48ABE, 0x2328B4DA797766D9, 0x9E20568107517D3C,
				0x58E9DE4729F43A46],
			[u64(0x4473307237F1326F), 0xB76050F958209417, 0xD59DA65A55B66B8A, 0x2FFD7286C7908ECE,
				0x2E07C7067581FC81],
			[u64(0x1B7C4A5588C64366), 0xFDA16647DA486B24, 0xDB1349AF0F125B9F, 0x358C16405B7E098E,
				0x14ACC6DA5D7E317B],
			[u64(0xD667063065C555E2), 0x22CE8E9DD788512C, 0x05A25D0A4CBD7DBF, 0x9498861C152D0414,
				0x9A7AE4B3BE53FCB2],
		]
		pi:    [
			[u64(0xC2F0509FFD2B0437), 0x1B7C4A5588C64366, 0x760B9F82165B67B9, 0xD667063065C555E2,
				0x4473307237F1326F],
			[u64(0x0623D2279FD48ABE), 0x22CE8E9DD788512C, 0xB76050F958209417, 0x3EFC8292F5387574,
				0xFDA16647DA486B24],
			[u64(0xD59DA65A55B66B8A), 0x9BECDE076EC21D80, 0xDB1349AF0F125B9F, 0x2328B4DA797766D9,
				0x05A25D0A4CBD7DBF],
			[u64(0x358C16405B7E098E), 0x9E20568107517D3C, 0x9498861C152D0414, 0x2FFD7286C7908ECE,
				0xB86B8BA57F6C5786],
			[u64(0x9A7AE4B3BE53FCB2), 0x2E07C7067581FC81, 0xA3B5195DEC68F0C8, 0x14ACC6DA5D7E317B,
				0x58E9DE4729F43A46],
		]
		chi:   [
			[u64(0x136C74C7BD096537), 0x825C1A57A0844FE6, 0x3E18968411492C31, 0xD76732786D82576B,
				0x4471297A334426F4],
			[u64(0x2623C227959C8ABA), 0x26CE8E1DD6993110, 0xB3E8D6E9480D9017, 0x3229C09673B8FD72,
				0x45E8E4E2E9086924],
			[u64(0x5FEF46E9F1B79FBA), 0xBBEB5F011E429D01, 0xF83650EEE752AB57, 0x33283082611957E8,
				0x452209484C2D55FF],
			[u64(0x750C064C1A56098B), 0x8F585ED08F177E5A, 0xC092009E073E0325, 0xEDBE72A6E711CA4E,
				0xBC79AB95696D57AF],
			[u64(0x9E796693BC87763A), 0x0E85438E2289EC89, 0x22D55924A44860CE, 0x3C344658CD46116F,
				0xE1699842E1FC7346],
		]
		iota:  [
			[u64(0x936C74C7BD09E534), 0x825C1A57A0844FE6, 0x3E18968411492C31, 0xD76732786D82576B,
				0x4471297A334426F4],
			[u64(0x2623C227959C8ABA), 0x26CE8E1DD6993110, 0xB3E8D6E9480D9017, 0x3229C09673B8FD72,
				0x45E8E4E2E9086924],
			[u64(0x5FEF46E9F1B79FBA), 0xBBEB5F011E429D01, 0xF83650EEE752AB57, 0x33283082611957E8,
				0x452209484C2D55FF],
			[u64(0x750C064C1A56098B), 0x8F585ED08F177E5A, 0xC092009E073E0325, 0xEDBE72A6E711CA4E,
				0xBC79AB95696D57AF],
			[u64(0x9E796693BC87763A), 0x0E85438E2289EC89, 0x22D55924A44860CE, 0x3C344658CD46116F,
				0xE1699842E1FC7346],
		]
	},
	// round 16
	RoundValues{
		theta: [
			[u64(0x7591ABAB888402B7), 0x64A1C53B9509A865, 0xD8E549E824C4CBB2, 0x319AED14580FB0E8,
				0xA28CF61606C9C177],
			[u64(0x4E6DC0A98CB92A30), 0x4E808C93CFBC919A, 0xDBA6D4675128309D, 0x5A67C2186A9D5DF8,
				0x2DA6E66CF02DC9AE],
			[u64(0x4D28FA2C5808F27B), 0xA92CE3C4B7FDF0C0, 0xEAF1EC2B4EEDC696, 0x21EF8C47C8A63A29,
				0x57E5B58DE592383E],
			[u64(0xC1DD32C6123C92D8), 0x3B896A5A877DE509, 0x744334140F549876, 0x596F462CEF7B511D,
				0x08A89F1F6107CCFC],
			[u64(0x8D05218E048171D6), 0x1DF904939A8FEB65, 0x31A91E391C4E6722, 0x2F48014575401683,
				0xF215DF5F59FA74AA],
		]
		rho:   [
			[u64(0x7591ABAB888402B7), 0x509A86564A1C53B9, 0xC72A4F4126265D96, 0x1F61D06335DA28B0,
				0xD8581B2705DE8A33],
			[u64(0x9CDB815319725460), 0xC919A4E808C93CFB, 0x9B519D44A0C2776E, 0xABBF0B4CF8430D53,
				0xB69B99B3C0B726B8],
			[u64(0xD34A3E8B16023C9E), 0x4B38F12DFF7C302A, 0x6E34B7578F615A77, 0xC623E4531D1490F7,
				0xCAFCB6B1BCB24707],
			[u64(0x6123C92D8C1DD32C), 0x849DC4B52D43BEF2, 0x281EA930ECE88668, 0xC59DEF6A23AB2DE8,
				0xFC08A89F1F6107CC],
			[u64(0x70240B8EB468290C), 0x4939A8FEB651DF90, 0x27339118D48F1C8E, 0x480145754016832F,
				0x77D7D67E9D2ABC85],
		]
		pi:    [
			[u64(0x7591ABAB888402B7), 0x6123C92D8C1DD32C, 0x9CDB815319725460, 0x70240B8EB468290C,
				0xD34A3E8B16023C9E],
			[u64(0xC919A4E808C93CFB), 0x4939A8FEB651DF90, 0x4B38F12DFF7C302A, 0x509A86564A1C53B9,
				0x849DC4B52D43BEF2],
			[u64(0x6E34B7578F615A77), 0xC72A4F4126265D96, 0x281EA930ECE88668, 0x9B519D44A0C2776E,
				0x27339118D48F1C8E],
			[u64(0xC59DEF6A23AB2DE8), 0xABBF0B4CF8430D53, 0x480145754016832F, 0xC623E4531D1490F7,
				0x1F61D06335DA28B0],
			[u64(0x77D7D67E9D2ABC85), 0xCAFCB6B1BCB24707, 0xD8581B2705DE8A33, 0xFC08A89F1F6107CC,
				0xB69B99B3C0B726B8],
		]
		chi:   [
			[u64(0x53B5B8BC0FA440B3), 0xE7218E2C8C3BD32A, 0xBCDD894319F2D220, 0xFB65128E14AA0D4A,
				0xF0682F83C68E3C92],
			[u64(0x4890ECC028431973), 0x61ACA8F26E10DFD1, 0x0B39B568FF6A312D, 0x14B8E6455708D328,
				0x9CDD84D60C139EC2],
			[u64(0x5C76A7431361CA72), 0x876AFBF022961F92, 0xB846B332E9208E78, 0xA35995C8A2A37066,
				0x87A9988814AA1A86],
			[u64(0xC59DC6EB232F2FDA), 0x8ABC4240F84E9D7B, 0x4C82C5255836D76F, 0xC607E753BD1CB8F7,
				0x5E21F66B23DA30B6],
			[u64(0xFFDFD23E9D6380CD), 0xC2E496638EF24B97, 0x9B786B0BE3D2AA39, 0xFC922CCF5575557D,
				0xB20E5987E9F6A4D8],
		]
		iota:  [
			[u64(0xD3B5B8BC0FA4C0B1), 0xE7218E2C8C3BD32A, 0xBCDD894319F2D220, 0xFB65128E14AA0D4A,
				0xF0682F83C68E3C92],
			[u64(0x4890ECC028431973), 0x61ACA8F26E10DFD1, 0x0B39B568FF6A312D, 0x14B8E6455708D328,
				0x9CDD84D60C139EC2],
			[u64(0x5C76A7431361CA72), 0x876AFBF022961F92, 0xB846B332E9208E78, 0xA35995C8A2A37066,
				0x87A9988814AA1A86],
			[u64(0xC59DC6EB232F2FDA), 0x8ABC4240F84E9D7B, 0x4C82C5255836D76F, 0xC607E753BD1CB8F7,
				0x5E21F66B23DA30B6],
			[u64(0xFFDFD23E9D6380CD), 0xC2E496638EF24B97, 0x9B786B0BE3D2AA39, 0xFC922CCF5575557D,
				0xB20E5987E9F6A4D8],
		]
	},
	// round 17
	RoundValues{
		theta: [
			[u64(0x6FABC531872124BC), 0x5B3FF3A104BE3727, 0x00C3F4CE9177362D, 0x477B6F039C2FE947,
				0x4C76520E4E0BD89F],
			[u64(0x4481AB9CBDB68BE0), 0x6DBDEFAEFBE54D42, 0x0728F2346A9FA3BE, 0x18A9A119C2FD41BB,
				0x90CCC38A99E60C51],
			[u64(0xC11C95E6CA60AB08), 0x1A00C955FB977EE8, 0x252C81973021EF02, 0x3E33A76D7BA2111C,
				0x1AC3AA2DCDAB7BFC],
			[u64(0x53899016D4703F2F), 0x1CA814BD0F118D8E, 0xDA9693D8AF69C79A, 0x5013B1AE4A43A802,
				0xC835A096D4852043],
			[u64(0x62D3873410618D85), 0x5FE8C36903F046DF, 0x06743E016ED0A771, 0x619E79C5D8775835,
				0x2F020C8D64F4A990],
		]
		rho:   [
			[u64(0x6FABC531872124BC), 0x4BE37275B3FF3A10, 0x061FA6748BB9B168, 0x5FD28E8EF6DE0738,
				0x4839382F627D31D9],
			[u64(0x890357397B6D17C0), 0x54D426DBDEFAEFBE, 0xA3C8D1AA7E8EF81C, 0xA83763153423385F,
				0x43330E2A67983146],
			[u64(0x30472579B2982AC2), 0x8032557EE5DFBA06, 0x0F781129640CB981, 0xD3B6BDD1088E1F19,
				0x83587545B9B56F7F],
			[u64(0x6D4703F2F5389901), 0xC70E540A5E8788C6, 0xB15ED38F35B52D27, 0x35C94875004A0276,
				0x43C835A096D48520],
			[u64(0xA0830C6C2B169C39), 0x36903F046DF5FE8C, 0x6853B8833A1F00B7, 0x9E79C5D877583561,
				0x8323593D2A640BC0],
		]
		pi:    [
			[u64(0x6FABC531872124BC), 0x6D4703F2F5389901, 0x890357397B6D17C0, 0xA0830C6C2B169C39,
				0x30472579B2982AC2],
			[u64(0x54D426DBDEFAEFBE), 0x36903F046DF5FE8C, 0x8032557EE5DFBA06, 0x4BE37275B3FF3A10,
				0xC70E540A5E8788C6],
			[u64(0x0F781129640CB981), 0x061FA6748BB9B168, 0xB15ED38F35B52D27, 0xA3C8D1AA7E8EF81C,
				0x6853B8833A1F00B7],
			[u64(0x35C94875004A0276), 0xA83763153423385F, 0x9E79C5D877583561, 0xD3B6BDD1088E1F19,
				0x5FD28E8EF6DE0738],
			[u64(0x8323593D2A640BC0), 0x83587545B9B56F7F, 0x4839382F627D31D9, 0x43C835A096D48520,
				0x43330E2A67983146],
		]
		chi:   [
			[u64(0x6483D411A72534BD), 0x6D48838277309861, 0xB84FD5B86B4D12E1, 0x008B8DE667165C35,
				0x18168DF892802AF3],
			[u64(0x64556E8FDEB8EDC8), 0x9EB07E0559F7F69B, 0x8E13512EA797AA46, 0x1BD55E24B3FF3D11,
				0xD08E52069A478FCE],
			[u64(0x8D5A00214E28B001), 0x0557B234022DF648, 0xF15EEBA835902DBF, 0xA380D18AE8DE783C,
				0x6872B8A33B1F30F1],
			[u64(0x5941CC75854B264A), 0xC43061A7702BA85F, 0x1F7B82C86E583361, 0x73B5B59D218C0700,
				0x6F96AFDF66DE0DB8],
			[u64(0x93777BF772BEC0C2), 0x91C84941B17009F3, 0x48093869E6EF99DF, 0x08A847B1063DA720,
				0x843B5E282B9FB142],
		]
		iota:  [
			[u64(0xE483D411A725343D), 0x6D48838277309861, 0xB84FD5B86B4D12E1, 0x008B8DE667165C35,
				0x18168DF892802AF3],
			[u64(0x64556E8FDEB8EDC8), 0x9EB07E0559F7F69B, 0x8E13512EA797AA46, 0x1BD55E24B3FF3D11,
				0xD08E52069A478FCE],
			[u64(0x8D5A00214E28B001), 0x0557B234022DF648, 0xF15EEBA835902DBF, 0xA380D18AE8DE783C,
				0x6872B8A33B1F30F1],
			[u64(0x5941CC75854B264A), 0xC43061A7702BA85F, 0x1F7B82C86E583361, 0x73B5B59D218C0700,
				0x6F96AFDF66DE0DB8],
			[u64(0x93777BF772BEC0C2), 0x91C84941B17009F3, 0x48093869E6EF99DF, 0x08A847B1063DA720,
				0x843B5E282B9FB142],
		]
	},
	// round 18
	RoundValues{
		theta: [
			[u64(0x5DFC5C5BBD667524), 0xD4370BC86D73D978, 0x01305DF2710E53F8, 0xB9F405AC7D551D2C,
				0xA16905B288C36BEA],
			[u64(0x280E8D93C4DE63C4), 0xD2EB9D1943917897, 0xC248B232BDF1244A, 0x578EBD38A999B31D,
				0x9CD5B11A802101C2],
			[u64(0x0EA52716FF9DDC52), 0x86A89503B3989A1B, 0x72A1CC9F842541EC, 0x207FF6BD596B146F,
				0xEB8D9F948AAA5CA2],
			[u64(0x67AADA6D3E198868), 0xFADB77BFCB79067D, 0x219094D0D50A9D43, 0x4D5EA3859ADEA922,
				0x517DB9C7DD8CA39A],
			[u64(0x5F6D4AC53349E7F8), 0x5DD27873F0872EC9, 0x8413095BA718BEE5, 0xC4B2768347CA801A,
				0x48216F1A6A689678],
		]
		rho:   [
			[u64(0x5DFC5C5BBD667524), 0xD73D978D4370BC86, 0x0982EF9388729FC0, 0xAA3A5973E80B58FA,
				0x16CA230DAFAA85A4],
			[u64(0x501D1B2789BCC788), 0x17897D2EB9D19439, 0x22C8CAF7C4912B09, 0x3663AAF1D7A71533,
				0x7356C46A0084070A],
			[u64(0x83A949C5BFE77714), 0xAA2540ECE62686E1, 0x2A0F63950E64FC21, 0xFB5EACB58A37903F,
				0x5D71B3F291554B94],
			[u64(0xD3E19886867AADA6), 0x3EFD6DBBDFE5BC83, 0xA1AA153A86432129, 0x70B35BD52449ABD4,
				0x9A517DB9C7DD8CA3],
			[u64(0x299A4F3FC2FB6A56), 0x873F0872EC95DD27, 0x8C5F72C20984ADD3, 0xB2768347CA801AC4,
				0x5BC69A9A259E1208],
		]
		pi:    [
			[u64(0x5DFC5C5BBD667524), 0xD3E19886867AADA6, 0x501D1B2789BCC788, 0x299A4F3FC2FB6A56,
				0x83A949C5BFE77714],
			[u64(0x17897D2EB9D19439), 0x873F0872EC95DD27, 0xAA2540ECE62686E1, 0xD73D978D4370BC86,
				0x3EFD6DBBDFE5BC83],
			[u64(0x2A0F63950E64FC21), 0x0982EF9388729FC0, 0xA1AA153A86432129, 0x22C8CAF7C4912B09,
				0x8C5F72C20984ADD3],
			[u64(0x70B35BD52449ABD4), 0x3663AAF1D7A71533, 0xB2768347CA801AC4, 0xFB5EACB58A37903F,
				0xAA3A5973E80B58FA],
			[u64(0x5BC69A9A259E1208), 0x5D71B3F291554B94, 0x16CA230DAFAA85A4, 0x9A517DB9C7DD8CA3,
				0x7356C46A0084070A],
		]
		chi:   [
			[u64(0x75FA5ECABB421D24), 0xDB617F078618AF66, 0x51970E3589FDE680, 0x095A074D467A695F,
				0x03AB5B85BFE77644],
			[u64(0x4739656E99D897ED), 0xB15E0812BB10DD14, 0xB871C2A9AEA69C25, 0x0E2BB38D49562CB0,
				0x1CDD648A3FEEECAB],
			[u64(0x214BE39F0FF2EC29), 0x4092FE918822D544, 0xA5223532A369A409, 0x22C99BFF81592789,
				0xDD1BF6CA0900AAD3],
			[u64(0x748B1F94BC29CEF0), 0xB4E3A2F5D18DB111, 0xF2639B65CA9458CC, 0xDAD4AEB38A15F26B,
				0x2A9350F6576828EE],
			[u64(0x59C7BBBE250F9211), 0x596FB382F9D01B95, 0xBCEA63C5C9A885C5, 0x4C74ED39C6DD1823,
				0x4F02E05040848F89],
		]
		iota:  [
			[u64(0x75FA5ECABB429D2E), 0xDB617F078618AF66, 0x51970E3589FDE680, 0x095A074D467A695F,
				0x03AB5B85BFE77644],
			[u64(0x4739656E99D897ED), 0xB15E0812BB10DD14, 0xB871C2A9AEA69C25, 0x0E2BB38D49562CB0,
				0x1CDD648A3FEEECAB],
			[u64(0x214BE39F0FF2EC29), 0x4092FE918822D544, 0xA5223532A369A409, 0x22C99BFF81592789,
				0xDD1BF6CA0900AAD3],
			[u64(0x748B1F94BC29CEF0), 0xB4E3A2F5D18DB111, 0xF2639B65CA9458CC, 0xDAD4AEB38A15F26B,
				0x2A9350F6576828EE],
			[u64(0x59C7BBBE250F9211), 0x596FB382F9D01B95, 0xBCEA63C5C9A885C5, 0x4C74ED39C6DD1823,
				0x4F02E05040848F89],
		]
	},
	// round 19
	RoundValues{
		theta: [
			[u64(0x730EC9FFDDC02B4B), 0xDD95E832E09A1903, 0x57639900EF7F50E5, 0x0FAE907820F8DF3A,
				0x055FCCB0D965C021],
			[u64(0xC4969C4D8D227C42), 0x32F1F131AFEA36BB, 0x3BDE3B8ABA5C778A, 0x8D844AAE5DACC71F,
				0x9F729DA92B140704],
			[u64(0xF9322BCF01BF01BF), 0x98EB36C1866F38D2, 0x7D5BFD62AD24499F, 0xFAB053AF8F14CA1F,
				0x05623E9A074D4745],
			[u64(0x31CA97BC3694E919), 0xF1A22ADD5B3096F8, 0xB722134D40297F25, 0x9F95269B00A8D582,
				0x6FD2D8DEDDD50F07],
			[u64(0x7071859FC537F81E), 0x70D98DA319E8719A, 0x955C5DE42990EFCA, 0x65C2D31826E5722C,
				0x66B4DE71A0BCE586],
		]
		rho:   [
			[u64(0x730EC9FFDDC02B4B), 0x09A1903DD95E832E, 0xBB1CC8077BFA872A, 0xF1BE741F5D20F041,
				0x32C365970084157F],
			[u64(0x892D389B1A44F885), 0xA36BB32F1F131AFE, 0x78EE2AE971DE28EF, 0x98E3F1B08955CBB5,
				0x7DCA76A4AC501C12],
			[u64(0xFE4C8AF3C06FC06F), 0x3ACDB0619BCE34A6, 0x224CFBEADFEB1569, 0x29D7C78A650FFD58,
				0xA0AC47D340E9A8E8],
			[u64(0xC3694E91931CA97B), 0x7C78D1156EAD984B, 0x9A8052FE4B6E4426, 0xD360151AB053F2A4,
				0x076FD2D8DEDDD50F],
			[u64(0xFE29BFC0F3838C2C), 0xDA319E8719A70D98, 0xC877E54AAE2EF214, 0xC2D31826E5722C65,
				0x379C682F396199AD],
		]
		pi:    [
			[u64(0x730EC9FFDDC02B4B), 0xC3694E91931CA97B, 0x892D389B1A44F885, 0xFE29BFC0F3838C2C,
				0xFE4C8AF3C06FC06F],
			[u64(0xA36BB32F1F131AFE), 0xDA319E8719A70D98, 0x3ACDB0619BCE34A6, 0x09A1903DD95E832E,
				0x7C78D1156EAD984B],
			[u64(0x224CFBEADFEB1569), 0xBB1CC8077BFA872A, 0x9A8052FE4B6E4426, 0x78EE2AE971DE28EF,
				0xC877E54AAE2EF214],
			[u64(0xD360151AB053F2A4), 0x98E3F1B08955CBB5, 0xC2D31826E5722C65, 0x29D7C78A650FFD58,
				0xF1BE741F5D20F041],
			[u64(0x379C682F396199AD), 0xA0AC47D340E9A8E8, 0x32C365970084157F, 0x076FD2D8DEDDD50F,
				0x7DCA76A4AC501C12],
		]
		chi:   [
			[u64(0x730A813F1D282E4A), 0xE2650E91F1442B59, 0x092D7A055A64B885, 0x8E679500D303A4ED,
				0x7E4BAEB9406DA27B],
			[u64(0x724BB73F3F03F87A), 0xDAD2AF3799A2450D, 0x7A9EB8613FDE1CE7, 0x08B0553FDD5F563E,
				0x4DF0C1003FAD980A],
			[u64(0x06D093CFD6CB1C60), 0x9B10CE443B52A762, 0xAA80376F4BEA553C, 0x7EC63AB9EB0E28E8,
				0xC437E7EA0E7EFE06],
			[u64(0x936294CA74D3D0E6), 0xDBA2F9B01A41CAA6, 0x4BFF002EFF32C4E5, 0xD1D7EA8A440DF578,
				0x73BAFC4C1D0F302C],
			[u64(0xB7FD5A2F3B728919), 0xB8BCD7D5484AAC68, 0x0003E5F7810E115D, 0x06EFD2E5D681D60D,
				0x7DFA27A082D00412],
		]
		iota:  [
			[u64(0xF30A813F9D282E40), 0xE2650E91F1442B59, 0x092D7A055A64B885, 0x8E679500D303A4ED,
				0x7E4BAEB9406DA27B],
			[u64(0x724BB73F3F03F87A), 0xDAD2AF3799A2450D, 0x7A9EB8613FDE1CE7, 0x08B0553FDD5F563E,
				0x4DF0C1003FAD980A],
			[u64(0x06D093CFD6CB1C60), 0x9B10CE443B52A762, 0xAA80376F4BEA553C, 0x7EC63AB9EB0E28E8,
				0xC437E7EA0E7EFE06],
			[u64(0x936294CA74D3D0E6), 0xDBA2F9B01A41CAA6, 0x4BFF002EFF32C4E5, 0xD1D7EA8A440DF578,
				0x73BAFC4C1D0F302C],
			[u64(0xB7FD5A2F3B728919), 0xB8BCD7D5484AAC68, 0x0003E5F7810E115D, 0x06EFD2E5D681D60D,
				0x7DFA27A082D00412],
		]
	},
	// round 20
	RoundValues{
		theta: [
			[u64(0xA9D374DBCC55173A), 0xB8BCFB75A0391223, 0x53F48FE10B1981FF, 0xD4BE60E4827E9D97,
				0x24925B5D11109B01],
			[u64(0x814616431C6332D1), 0x29DF0E4BBAC28FA6, 0x8993191D1CBED64C, 0xFBBDF443FE3F9C95,
				0xBEFD607C1CCD52A1],
			[u64(0xD33350BC3C024427), 0x4EF30D37D19BFF25, 0x7F63F41CA1230D7B, 0xAB25F9CA01C770AF,
				0x11D42499E4B7A641],
			[u64(0xF67C19EC7B1F2450), 0xBEBC7496158D3E10, 0x2EE18D08F0FE3053, 0xB4C967AC4BC101CE,
				0x16A4716A12C3C49A],
			[u64(0xC672BD98B91DE4FD), 0xC9333062CA25C18C, 0x718C024003617CB9, 0x7760355254EEBBE9,
				0x0C75C01700BF69F6],
		]
		rho:   [
			[u64(0xA9D374DBCC55173A), 0x0391223B8BCFB75A, 0x9FA47F0858CC0FFA, 0xFD3B2FA97CC1C904,
				0x6D7444426C049249],
			[u64(0x028C2C8638C665A3), 0x28FA629DF0E4BBAC, 0x4C647472FB593226, 0xF392BF77BE887FC7,
				0xFBF581F073354A86],
			[u64(0xF4CCD42F0F009109), 0xBCC34DF466FFC953, 0x186BDBFB1FA0E509, 0xFCE500E3B857D592,
				0x223A84933C96F4C8],
			[u64(0xC7B1F2450F67C19E), 0x085F5E3A4B0AC69F, 0x11E1FC60A65DC31A, 0xF589782039D6992C,
				0x9A16A4716A12C3C4],
			[u64(0xC5C8EF27EE3395EC), 0x062CA25C18CC9333, 0xB0BE5CB8C6012001, 0x60355254EEBBE977,
				0x7005C02FDA7D831D],
		]
		pi:    [
			[u64(0xA9D374DBCC55173A), 0xC7B1F2450F67C19E, 0x028C2C8638C665A3, 0xC5C8EF27EE3395EC,
				0xF4CCD42F0F009109],
			[u64(0x28FA629DF0E4BBAC), 0x062CA25C18CC9333, 0xBCC34DF466FFC953, 0x0391223B8BCFB75A,
				0x085F5E3A4B0AC69F],
			[u64(0x186BDBFB1FA0E509), 0x9FA47F0858CC0FFA, 0x11E1FC60A65DC31A, 0x4C647472FB593226,
				0xB0BE5CB8C6012001],
			[u64(0xF589782039D6992C), 0xF392BF77BE887FC7, 0x60355254EEBBE977, 0xFCE500E3B857D592,
				0xFD3B2FA97CC1C904],
			[u64(0x7005C02FDA7D831D), 0x223A84933C96F4C8, 0x6D7444426C049249, 0x9A16A4716A12C3C4,
				0xFBF581F073354A86],
		]
		chi:   [
			[u64(0xB9D2EDB9C355533B), 0x5E31AF454F67CD56, 0x03AC9C86B8C667AB, 0x89ACBB679E2395C8,
				0x446CD4AF8B01B109],
			[u64(0xCD7A429DD0B2A388), 0x663E222BBECCE336, 0xDCD74FE02E5DE136, 0xB31022BA8BC972CA,
				0x455E7D3B73CA0F9B],
			[u64(0x186F5BF4DD89E718), 0x9F8C7F8858DA8FF2, 0x1CA1F862A659D112, 0x4E76D062B9593062,
				0xB27ADCE8C5352283],
			[u64(0x7C5B4CF03DD68D0E), 0x3613CD33BDE97ED1, 0x62BD7AD0FE798CD5, 0xB92D4BE53C76C1BA,
				0xF9337BA670C1580D],
			[u64(0x702DC22BEADD2B99), 0x2236848B2C1EE6E9, 0xD13705322A3D1A19, 0x9807A4696BDEE1D6,
				0xF3E68BE0333F0C10],
		]
		iota:  [
			[u64(0x39D2EDB94355D3BA), 0x5E31AF454F67CD56, 0x03AC9C86B8C667AB, 0x89ACBB679E2395C8,
				0x446CD4AF8B01B109],
			[u64(0xCD7A429DD0B2A388), 0x663E222BBECCE336, 0xDCD74FE02E5DE136, 0xB31022BA8BC972CA,
				0x455E7D3B73CA0F9B],
			[u64(0x186F5BF4DD89E718), 0x9F8C7F8858DA8FF2, 0x1CA1F862A659D112, 0x4E76D062B9593062,
				0xB27ADCE8C5352283],
			[u64(0x7C5B4CF03DD68D0E), 0x3613CD33BDE97ED1, 0x62BD7AD0FE798CD5, 0xB92D4BE53C76C1BA,
				0xF9337BA670C1580D],
			[u64(0x702DC22BEADD2B99), 0x2236848B2C1EE6E9, 0xD13705322A3D1A19, 0x9807A4696BDEE1D6,
				0xF3E68BE0333F0C10],
		]
	},
	// round 21
	RoundValues{
		theta: [
			[u64(0xD2A5600D870B50A6), 0xB54622F18B394E4A, 0xE8DB11327C98E4B7, 0x62DB36D35A7D16D4,
				0xAF1B591B4F5F3215],
			[u64(0xAA685207CFA9A83C), 0x012C32B1A1D7E882, 0xBBC55F7A3146EA82, 0xD402322094D2797E,
				0x224C6DA16CD1042F],
			[u64(0x4865BD83004AF6BB), 0xCF8699FF85199E51, 0x4CAB1E157B9AC0B1, 0x1E7C3615649A21C1,
				0xE2703A9F18F63320],
			[u64(0xCA8F44530A8E5348), 0x80C7C5908AB1A097, 0xD4697273C9215293, 0x0FF943460B2E1FFC,
				0x4FE773054799864B],
			[u64(0x4BD96A1E9B807629), 0x19C22CBE5D43BB59, 0xEAC3AD075B6047A9, 0xA3F30C5C1A83BC66,
				0xC81223D5426251A0],
		]
		rho:   [
			[u64(0xD2A5600D870B50A6), 0xB394E4AB54622F18, 0x46D88993E4C725BF, 0xFA2DA8C5B66DA6B4,
				0x646D3D7CC856BC6D],
			[u64(0x54D0A40F9F535079), 0x7E882012C32B1A1D, 0x157DE8C51BAA0AEF, 0x4F2FDA804644129A,
				0x8931B685B34410BC],
			[u64(0xD2196F60C012BDAE), 0xE1A67FE146679473, 0xD6058A6558F0ABDC, 0x1B0AB24D10E08F3E,
				0x1C4E0753E31EC664],
			[u64(0x30A8E5348CA8F445), 0x4BC063E2C84558D0, 0xE79242A527A8D2E4, 0x68C165C3FF81FF28,
				0x4B4FE77305479986],
			[u64(0xF4DC03B14A5ECB50), 0xCBE5D43BB5919C22, 0xB023D4F561D683AD, 0xF30C5C1A83BC66A3,
				0x88F5509894683204],
		]
		pi:    [
			[u64(0xD2A5600D870B50A6), 0x30A8E5348CA8F445, 0x54D0A40F9F535079, 0xF4DC03B14A5ECB50,
				0xD2196F60C012BDAE],
			[u64(0x7E882012C32B1A1D), 0xCBE5D43BB5919C22, 0xE1A67FE146679473, 0xB394E4AB54622F18,
				0x4BC063E2C84558D0],
			[u64(0xD6058A6558F0ABDC), 0x46D88993E4C725BF, 0xE79242A527A8D2E4, 0x157DE8C51BAA0AEF,
				0xB023D4F561D683AD],
			[u64(0x68C165C3FF81FF28), 0x4F2FDA804644129A, 0xF30C5C1A83BC66A3, 0x1B0AB24D10E08F3E,
				0xFA2DA8C5B66DA6B4],
			[u64(0x88F5509894683204), 0x1C4E0753E31EC664, 0x646D3D7CC856BC6D, 0x4B4FE77305479986,
				0x8931B685B34410BC],
		]
		chi:   [
			[u64(0x52A0EA689FDBF166), 0x34B0ECB4CCEED5D8, 0x52C0A40BBEDB12FD, 0xF0B50BF541D6CBB7,
				0x623AFB75E1803E83],
			[u64(0x56484590642A4E3D), 0xC2C2863BB7918E22, 0xF1AA63FBC673B070, 0xB996F6A35422AA08,
				0x01CC4BE25E6C7CC0],
			[u64(0x56319A7D5898ABD8), 0x56988CC045DDE1DB, 0xE3F363C16FEA4AA8, 0x5538ADF71EAD1A6F,
				0xB133C2F560D693A5],
			[u64(0x3AC145C6FC82BF8A), 0x6F8F3AA44AE4229B, 0xE39CDC1994BD26B3, 0xAF9AB2CD5AF8CD6E,
				0xA825E1A5F67F0BB6],
			[u64(0xA4FD508AD448381D), 0xD70B1758D20FCE46, 0xC54B669C8872386F, 0x484F03791167BD8E,
				0x80F1B607BB0150EC],
		]
		iota:  [
			[u64(0xD2A0EA689FDB71E6), 0x34B0ECB4CCEED5D8, 0x52C0A40BBEDB12FD, 0xF0B50BF541D6CBB7,
				0x623AFB75E1803E83],
			[u64(0x56484590642A4E3D), 0xC2C2863BB7918E22, 0xF1AA63FBC673B070, 0xB996F6A35422AA08,
				0x01CC4BE25E6C7CC0],
			[u64(0x56319A7D5898ABD8), 0x56988CC045DDE1DB, 0xE3F363C16FEA4AA8, 0x5538ADF71EAD1A6F,
				0xB133C2F560D693A5],
			[u64(0x3AC145C6FC82BF8A), 0x6F8F3AA44AE4229B, 0xE39CDC1994BD26B3, 0xAF9AB2CD5AF8CD6E,
				0xA825E1A5F67F0BB6],
			[u64(0xA4FD508AD448381D), 0xD70B1758D20FCE46, 0xC54B669C8872386F, 0x484F03791167BD8E,
				0x80F1B607BB0150EC],
		]
	},
	// round 22
	RoundValues{
		theta: [
			[u64(0x1657447A84851FFF), 0xF04742A6D7B0BBC1, 0x96370A19A5857CE4, 0x3442A5E75A88A5AE,
				0xA6CD5567FADE509A],
			[u64(0x7EB5233B303B1F08), 0xEA3FE090E380DF17, 0xD95705509262E145, 0x916B90080033FB3D,
				0x29312D490A7D2DF5],
			[u64(0xE990674B5BA6F78A), 0xE93971F646E3BD89, 0x5C529EF76CD416FA, 0xEA9950C11D93463D,
				0x0E923FC363E8CFF7],
			[u64(0xC19777D8B8F07047), 0x94D908BA0E96ED56, 0x18CAEE07D0CFE97E, 0x54CC80D31E8A02A3,
				0x5373D3BBB20DC47B],
			[u64(0x592E0437C164C289), 0x2AD843E5C72334D2, 0x389832219D5EC2FB, 0xB59C57C4044B471A,
				0x7D22E2BAAE2DAA78],
		]
		rho:   [
			[u64(0x1657447A84851FFF), 0x7B0BBC1F04742A6D, 0xB1B850CD2C2BE724, 0x114B5C68854BCEB5,
				0x559FEB79426A9B35],
			[u64(0xFD6A467660763E10), 0x0DF17EA3FE090E38, 0x5C1542498B851765, 0x7F67B22D72010006,
				0xA4C4B52429F4B7D4],
			[u64(0xBA6419D2D6E9BDE2), 0x4E5C7D91B8EF627A, 0xA0B7D2E294F7BB66, 0xA8608EC9A31EF54C,
				0xE1D247F86C7D19FE],
			[u64(0x8B8F07047C19777D), 0xAB4A6C845D074B76, 0x0FA19FD2FC3195DC, 0x1A63D140546A9990,
				0x7B5373D3BBB20DC4],
			[u64(0xBE0B26144AC97021), 0x3E5C72334D22AD84, 0xAF617D9C4C1910CE, 0x9C57C4044B471AB5,
				0xB8AEAB8B6A9E1F48],
		]
		pi:    [
			[u64(0x1657447A84851FFF), 0x8B8F07047C19777D, 0xFD6A467660763E10, 0xBE0B26144AC97021,
				0xBA6419D2D6E9BDE2],
			[u64(0x0DF17EA3FE090E38), 0x3E5C72334D22AD84, 0x4E5C7D91B8EF627A, 0x7B0BBC1F04742A6D,
				0xAB4A6C845D074B76],
			[u64(0xA0B7D2E294F7BB66), 0xB1B850CD2C2BE724, 0x0FA19FD2FC3195DC, 0x5C1542498B851765,
				0xAF617D9C4C1910CE],
			[u64(0x1A63D140546A9990), 0x7F67B22D72010006, 0x9C57C4044B471AB5, 0xA8608EC9A31EF54C,
				0x114B5C68854BCEB5],
			[u64(0xB8AEAB8B6A9E1F48), 0xE1D247F86C7D19FE, 0x559FEB79426A9B35, 0x7B5373D3BBB20DC4,
				0xA4C4B52429F4B7D4],
		]
		chi:   [
			[u64(0xB651C43A8473AEB9), 0x0A2F07C85C10355D, 0xFCCBC4342466AB94, 0xBA1F6454C1486521,
				0xBE4508CAD6F1AD6A],
			[u64(0x17B17FA3BE010EA8), 0x701BD0131F22AD86, 0xDE0A3D95BBA9685B, 0xDB6B309F246ECA65,
				0xBB406CE4DC458547],
			[u64(0x003BF869BE63BD2E), 0x3128151D2057FEDC, 0x4E29B4ABFC1914DC, 0x0F06335B93251FE5,
				0x0BE5DC9864AD218E],
			[u64(0x1C329530D06B9927), 0x756AB22962016607, 0x3437C0026B533EB5, 0x2C688ACDE357856D,
				0x0B6B54BA5342C697],
			[u64(0xB10E910A10961F48), 0xD58237CB6D5F917E, 0x578BD2F8DAE3DB5F, 0x3A53EBD8BF860788,
				0xA5CED12020F2F5C0],
		]
		iota:  [
			[u64(0xB651C43A0473AEB8), 0x0A2F07C85C10355D, 0xFCCBC4342466AB94, 0xBA1F6454C1486521,
				0xBE4508CAD6F1AD6A],
			[u64(0x17B17FA3BE010EA8), 0x701BD0131F22AD86, 0xDE0A3D95BBA9685B, 0xDB6B309F246ECA65,
				0xBB406CE4DC458547],
			[u64(0x003BF869BE63BD2E), 0x3128151D2057FEDC, 0x4E29B4ABFC1914DC, 0x0F06335B93251FE5,
				0x0BE5DC9864AD218E],
			[u64(0x1C329530D06B9927), 0x756AB22962016607, 0x3437C0026B533EB5, 0x2C688ACDE357856D,
				0x0B6B54BA5342C697],
			[u64(0xB10E910A10961F48), 0xD58237CB6D5F917E, 0x578BD2F8DAE3DB5F, 0x3A53EBD8BF860788,
				0xA5CED12020F2F5C0],
		]
	},
	// round 23
	RoundValues{
		theta: [
			[u64(0xA9DC1646F96E0136), 0x15A2D5B4A10D9AD3, 0xE3461648D97B041A, 0xA592B6283C55CAAF,
				0xA1C8DAB62BEC02E4],
			[u64(0xA4ED78C2FEF72418), 0xC347D7725FD48736, 0x6D563AF4FB5F42EB, 0x683737FE6498E0D5,
				0x081C6B859CB3AFF7],
			[u64(0x2D6844EF8E9B3DA7), 0x1C7BA99B10AF7E55, 0x637A082DCCE19455, 0x22558FDDA3DD9F6C,
				0x26B6601E5455A107],
			[u64(0x3EDFBEAE3573BE21), 0x578799B787194101, 0x16DAEB9C8E4B19B3, 0x0E85A153064FA26B,
				0x29867F24B65AE191],
			[u64(0x42BC7ED6AEC36D53), 0x2630D817D30AE365, 0xA4393D2464B6A944, 0xC9E1040401D37593,
				0x567C3EFC9EA787DB],
		]
		rho:   [
			[u64(0xA9DC1646F96E0136), 0x10D9AD315A2D5B4A, 0x1A30B246CBD820D7, 0xAB955F4B256C5078,
				0x6AD8AFB00B928723],
			[u64(0x49DAF185FDEE4831), 0x48736C347D7725FD, 0x58EBD3ED7D0BADB5, 0x1C1AAD06E6FFCC93,
				0x2071AE1672CEBFDC],
			[u64(0xCB5A113BE3A6CF69), 0x1EEA66C42BDF9547, 0x0CA2AB1BD0416E67, 0xC7EED1EECFB6112A,
				0xE4D6CC03CA8AB420],
			[u64(0xE3573BE213EDFBEA), 0x80ABC3CCDBC38CA0, 0x391C9633662DB5D7, 0x2A60C9F44D61D0B4,
				0x9129867F24B65AE1],
			[u64(0xB5761B6A9A15E3F6), 0x817D30AE3652630D, 0x5B54A2521C9E9232, 0xE1040401D37593C9,
				0x0FBF27A9E1F6D59F],
		]
		pi:    [
			[u64(0xA9DC1646F96E0136), 0xE3573BE213EDFBEA, 0x49DAF185FDEE4831, 0xB5761B6A9A15E3F6,
				0xCB5A113BE3A6CF69],
			[u64(0x48736C347D7725FD), 0x817D30AE3652630D, 0x1EEA66C42BDF9547, 0x10D9AD315A2D5B4A,
				0x80ABC3CCDBC38CA0],
			[u64(0x0CA2AB1BD0416E67), 0x1A30B246CBD820D7, 0x391C9633662DB5D7, 0x58EBD3ED7D0BADB5,
				0x5B54A2521C9E9232],
			[u64(0x2A60C9F44D61D0B4), 0x1C1AAD06E6FFCC93, 0xE1040401D37593C9, 0xC7EED1EECFB6112A,
				0xAB955F4B256C5078],
			[u64(0x0FBF27A9E1F6D59F), 0xE4D6CC03CA8AB420, 0x6AD8AFB00B928723, 0x9129867F24B65AE1,
				0x2071AE1672CEBFDC],
		]
		chi:   [
			[u64(0xAD5C954D796E4B34), 0xF957B9A2DA65FB38, 0x68CE61B6B9CE68A1, 0xFD5449A6BF174743,
				0x900E3129E7BADD7B],
			[u64(0x6A332CD07057B56D), 0x85773DAE1275AF0D, 0xDEEA66C4BA8F974F, 0x97DDAD33D8994B40,
				0x202A9EC5FAA3CCE8],
			[u64(0x093D8D1270D76B6C), 0xFAF4F247C3D810F7, 0x33C43D836EAFB1F5, 0x48EAD5FC5D0BE774,
				0x5B3402464E1C3DB6],
			[u64(0x8A20D9B25569D094), 0x1F1B9EE6F79A8759, 0xE00654042719DBD9, 0xE3B8C8EE55B7B03C,
				0x609F4E62A44C1059],
			[u64(0x4F9C4F99E5E7F156), 0xE4FECC0FEE98B425, 0x7CF8A9F009831265, 0x91A0226E649E42E9,
				0x20D06CD26A8FBF5C],
		]
		iota:  [
			[u64(0x2D5C954DF96ECB3C), 0xF957B9A2DA65FB38, 0x68CE61B6B9CE68A1, 0xFD5449A6BF174743,
				0x900E3129E7BADD7B],
			[u64(0x6A332CD07057B56D), 0x85773DAE1275AF0D, 0xDEEA66C4BA8F974F, 0x97DDAD33D8994B40,
				0x202A9EC5FAA3CCE8],
			[u64(0x093D8D1270D76B6C), 0xFAF4F247C3D810F7, 0x33C43D836EAFB1F5, 0x48EAD5FC5D0BE774,
				0x5B3402464E1C3DB6],
			[u64(0x8A20D9B25569D094), 0x1F1B9EE6F79A8759, 0xE00654042719DBD9, 0xE3B8C8EE55B7B03C,
				0x609F4E62A44C1059],
			[u64(0x4F9C4F99E5E7F156), 0xE4FECC0FEE98B425, 0x7CF8A9F009831265, 0x91A0226E649E42E9,
				0x20D06CD26A8FBF5C],
		]
	},
]

const first_state_as_bytes = [u8(0xE7), 0xDD, 0xE1, 0x40, 0x79, 0x8F, 0x25, 0xF1, 0x8A, 0x47, 0xC0,
	0x33, 0xF9, 0xCC, 0xD5, 0x84, 0xEE, 0xA9, 0x5A, 0xA6, 0x1E, 0x26, 0x98, 0xD5, 0x4D, 0x49, 0x80,
	0x6F, 0x30, 0x47, 0x15, 0xBD, 0x57, 0xD0, 0x53, 0x62, 0x05, 0x4E, 0x28, 0x8B, 0xD4, 0x6F, 0x8E,
	0x7F, 0x2D, 0xA4, 0x97, 0xFF, 0xC4, 0x47, 0x46, 0xA4, 0xA0, 0xE5, 0xFE, 0x90, 0x76, 0x2E, 0x19,
	0xD6, 0x0C, 0xDA, 0x5B, 0x8C, 0x9C, 0x05, 0x19, 0x1B, 0xF7, 0xA6, 0x30, 0xAD, 0x64, 0xFC, 0x8F,
	0xD0, 0xB7, 0x5A, 0x93, 0x30, 0x35, 0xD6, 0x17, 0x23, 0x3F, 0xA9, 0x5A, 0xEB, 0x03, 0x21, 0x71,
	0x0D, 0x26, 0xE6, 0xA6, 0xA9, 0x5F, 0x55, 0xCF, 0xDB, 0x16, 0x7C, 0xA5, 0x81, 0x26, 0xC8, 0x47,
	0x03, 0xCD, 0x31, 0xB8, 0x43, 0x9F, 0x56, 0xA5, 0x11, 0x1A, 0x2F, 0xF2, 0x01, 0x61, 0xAE, 0xD9,
	0x21, 0x5A, 0x63, 0xE5, 0x05, 0xF2, 0x70, 0xC9, 0x8C, 0xF2, 0xFE, 0xBE, 0x64, 0x11, 0x66, 0xC4,
	0x7B, 0x95, 0x70, 0x36, 0x61, 0xCB, 0x0E, 0xD0, 0x4F, 0x55, 0x5A, 0x7C, 0xB8, 0xC8, 0x32, 0xCF,
	0x1C, 0x8A, 0xE8, 0x3E, 0x8C, 0x14, 0x26, 0x3A, 0xAE, 0x22, 0x79, 0x0C, 0x94, 0xE4, 0x09, 0xC5,
	0xA2, 0x24, 0xF9, 0x41, 0x18, 0xC2, 0x65, 0x04, 0xE7, 0x26, 0x35, 0xF5, 0x16, 0x3B, 0xA1, 0x30,
	0x7F, 0xE9, 0x44, 0xF6, 0x75, 0x49, 0xA2, 0xEC, 0x5C, 0x7B, 0xFF, 0xF1, 0xEA]

const first_state_as_words = [
	[u64(0xF1258F7940E1DDE7), 0xFF97A42D7F8E6FD4, 0xEB5AA93F2317D635, 0x05E5635A21D9AE61,
		0x940C7922AE3A2614],
	[u64(0x84D5CCF933C0478A), 0x90FEE5A0A44647C4, 0xA9A6E6260D712103, 0x64BEFEF28CC970F2,
		0x1841F924A2C509E4],
	[u64(0xD598261EA65AA9EE), 0x8C5BDA0CD6192E76, 0x81A57C16DBCF555F, 0x613670957BC46611,
		0x16F53526E70465C2],
	[u64(0xBD1547306F80494D), 0xAD30A6F71B19059C, 0x43B831CD0347C826, 0xB87C5A554FD00ECB,
		0x75F644E97F30A13B],
	[u64(0x8B284E056253D057), 0x30935AB7D08FFC64, 0x01F22F1A11A5569F, 0x8C3EE88A1CCF32C8,
		0xEAF1FF7B5CECA249],
]
const second_state_as_bytes = [u8(0x3C), 0xCB, 0x6E, 0xF9, 0x4D, 0x95, 0x5C, 0x2D, 0x6D, 0xB5,
	0x57, 0x70, 0xD0, 0x2C, 0x33, 0x6A, 0x6C, 0x6B, 0xD7, 0x70, 0x12, 0x8D, 0x3D, 0x09, 0x94, 0xD0,
	0x69, 0x55, 0xB2, 0xD9, 0x20, 0x8A, 0x56, 0xF1, 0xE7, 0xE5, 0x99, 0x4F, 0x9C, 0x4F, 0x38, 0xFB,
	0x65, 0xDA, 0xA2, 0xB9, 0x57, 0xF9, 0x0D, 0xAF, 0x75, 0x12, 0xAE, 0x3D, 0x77, 0x85, 0xF7, 0x10,
	0xD8, 0xC3, 0x47, 0xF2, 0xF4, 0xFA, 0x59, 0x87, 0x9A, 0xF7, 0xE6, 0x9E, 0x1B, 0x1F, 0x25, 0xB4,
	0x98, 0xEE, 0x0F, 0xCC, 0xFE, 0xE4, 0xA1, 0x68, 0xCE, 0xB9, 0xB6, 0x61, 0xCE, 0x68, 0x4F, 0x97,
	0x8F, 0xBA, 0xC4, 0x66, 0xEA, 0xDE, 0xF5, 0xB1, 0xAF, 0x6E, 0x83, 0x3D, 0xC4, 0x33, 0xD9, 0xDB,
	0x19, 0x27, 0x04, 0x54, 0x06, 0xE0, 0x65, 0x12, 0x83, 0x09, 0xF0, 0xA9, 0xF8, 0x7C, 0x43, 0x47,
	0x17, 0xBF, 0xA6, 0x49, 0x54, 0xFD, 0x40, 0x4B, 0x99, 0xD8, 0x33, 0xAD, 0xDD, 0x97, 0x74, 0xE7,
	0x0B, 0x5D, 0xFC, 0xD5, 0xEA, 0x48, 0x3C, 0xB0, 0xB7, 0x55, 0xEE, 0xC8, 0xB8, 0xE3, 0xE9, 0x42,
	0x9E, 0x64, 0x6E, 0x22, 0xA0, 0x91, 0x7B, 0xDD, 0xBA, 0xE7, 0x29, 0x31, 0x0E, 0x90, 0xE8, 0xCC,
	0xA3, 0xFA, 0xC5, 0x9E, 0x2A, 0x20, 0xB6, 0x3D, 0x1C, 0x4E, 0x46, 0x02, 0x34, 0x5B, 0x59, 0x10,
	0x4C, 0xA4, 0x62, 0x4E, 0x9F, 0x60, 0x5C, 0xBF, 0x8F, 0x6A, 0xD2, 0x6C, 0xD0, 0x20]

const second_state_as_words = [
	[u64(0x2D5C954DF96ECB3C), 0xF957B9A2DA65FB38, 0x68CE61B6B9CE68A1, 0xFD5449A6BF174743,
		0x900E3129E7BADD7B],
	[u64(0x6A332CD07057B56D), 0x85773DAE1275AF0D, 0xDEEA66C4BA8F974F, 0x97DDAD33D8994B40,
		0x202A9EC5FAA3CCE8],
	[u64(0x093D8D1270D76B6C), 0xFAF4F247C3D810F7, 0x33C43D836EAFB1F5, 0x48EAD5FC5D0BE774,
		0x5B3402464E1C3DB6],
	[u64(0x8A20D9B25569D094), 0x1F1B9EE6F79A8759, 0xE00654042719DBD9, 0xE3B8C8EE55B7B03C,
		0x609F4E62A44C1059],
	[u64(0x4F9C4F99E5E7F156), 0xE4FECC0FEE98B425, 0x7CF8A9F009831265, 0x91A0226E649E42E9,
		0x20D06CD26A8FBF5C],
]
