from Cryptodome.PublicKey import RSA
from Cryptodome.Util.number import bytes_to_long


class SlaKey:
    vendor = None
    da_codes = None
    name = None
    d = None
    n = None
    e = None

    def __init__(self, vendor, da_codes, name, d, n, e):
        self.vendor = vendor
        self.da_codes = da_codes
        self.name = name
        self.d = d
        self.n = n
        self.e = e
        if isinstance(d,int):
            d_da = d
        else:
            d_da = bytes_to_long(bytes.fromhex(self.d))
            # print(d_da)
            # print()
        if isinstance(n, int):
            n_da = n
        else:
            n_da = bytes_to_long(bytes.fromhex(self.n))
            # print(n_da)
            # print()
        if isinstance(e, int):
            e_da = e
        else:
            e_da = bytes_to_long(bytes.fromhex(self.e))
        self.key = RSA.construct((n_da,d_da,e_da))


da_sla_keys = [
    SlaKey(vendor="Tecno/Infinix",
           da_codes=[0x1208, 0x0907],
           name="",
           d=9290628951699481328475309461169541034500121089516027862181856913840022010635102049268261812308791093363805172235451657938017199713542837325185858058111121571929962551864114834247812282948579529878600586872468535262951625779773541021322192898257574112748018473977365002405392986459715833892890734625164420475426867615539641514771368751432993404832959237372126550681525089955938432067666681988697582732135230478286539347776195752967079337713571948645749249279454582422057071616583285917136995567674538604652543551748384178795961199805990614137083900886365213418511238694820457083547695155896842049543908113268220300113,
           # "499890AEF768030B56AD8BF7355262A57643A8C3DF318647B72E0B72416D91D25882D0077FEBE9E8275E9F42C3F89FFF5E99E0163487461B7CC3E97C1CC4F4E2D587AB7D8DB80335AA8E3254F318AAC42136FF1043649918685A0F5CF5548318090C26D951A2B93F79FB0D9B5E1DB1C928091E1908A5DA7D3B9194A22B18B78C6EF88932EFB3D88DB8BA7117257063DE165FA8534D50C6B35F6F7C0AF5F4A96BB89756BB7AC94110E8D1B5868A5DAADF815FE8FA38960A039D681F319C7B7ED7A55649C2F2F75B26C27807AFED8BC4EF57619FF9DA152E337FE379E1B1B0020C1EF2ABDCE4A66AAF3802EE8AB105ACA6F3388EA4C20184E572778C8EB3D30B51"
           n=23517958656142483886608202246298501768174369866497177211425737990047644747431158092039168574479769868126060238423900938829039521731419734638111455370970589975263574961819949551645379435596796317097483455460060656451450780174933123133039496175091024898693197633412691006668298190560463368321297028780587123395366122040994661213714293758307447142632041162424850688916650692868864183456962026794287131811104377642465209593830700089704297187466650314062477718754580735763729755653167228639937193809612984219268164517411638252392029819687772109456100509264575849184468749970644097321576462038422455506034789381388781421911,
           # "BA4C5178EE5477B7A30FC99F9FE78E7C011E58B5DC05832591AADEE0D26A2667738FFE851782016CAE7DB8DC1958C627BC60E3A96D46E225536DCC76CBFADFE6CA2CBE4BE982AB46CBB0F1C5CB2CB13D7BBAC96E467D76AF819DCA28FB90A03AF92DDC05B34506E24F02A049D73B8143A1CF4B081372D49F4BCA21DA9EE8A19679669C71C836FC5B8180BD98256BA2AE82B702DDC61EDFD9C47871D027297D6C72A984046081953D141656F0DC4BC4E2D4006147BE096D7E855AA7057460EAC7C57660F1F28BE278C38700BAC5EBADB39F6508CF2F1E4044421664A856623FFB2002C1204CED786D576E3DDAA7C185FD698B1185A98A03B875F0A93831270557"
           e="010001"),
    # lk/files/pbp/keys/toolauth/da_prvk.pem
    SlaKey(vendor="KaiOS",
           da_codes=[],
           name="",
           d=18500403869873106546662264807562301040137423940158585796110736261465364581178532284364991395764358723481475804411163950566650079387758219436409648391022659012945310208666268874406273503184187428325757088371617680709294737946463913708967026017384327161470155867421017619666677054409753414769650487804067919678053322220136481017488820383754824027271573236903207240215813952143393730614312341525682537838112535154105120603167562072952347634916817138379722981386236619165976381933690988967930599301041229614636857465977450960379457793842083986729225548210668678888661576320089040317053848348371293331381379138138545735233,
           # "928d2a63d56bc42c3c02e836c025f6db39f06d57480dc705f3eaf238120a2b0d8fc00ec3cdf209615ca41bf73ae499dd9accb09b99fbb4046087008aab48d96f437292c160092dfc4ae33f94acee374e584b9d70d90ca46664d31cf72bba302e6eafab68e13a7f1ed9fa0ddc2604f3164bc97bc8c1b0be9db1d76d1d970ea36f4af8ad2ce0ab5b7ca6b4f99c133fa3f8ff9c92da874394d0e8f1bdbb83c8e811a344a5b5a7251c9b4fb4ad0ac494a2c50a1a79fa9b3992d749535b691dc2f016cee493e41cd2033190c4c0497f689e48bae5d3ad28cb0e8d17dae7f4e8c034ab60ab330f678fbfdd2c9716bfa6a617339d248b46df1593d5317b3af44a864641"
           n=19657278995134140462850273162989794475802307924224598529826675135662380002540490763949877449808832322646003239197380866217356456758033567237418598031816666727162723656701706602171918718842122105872002955587073734446255662140035757340216796110592033871291660263994345521078064431174643475093362257120868989169226273020788837168307608295092717642766533481448995742730435386038144249613029233359569015729552319967621381491984227976537005736623579051612985506531004145994441261056261762723559290821017092156084831250996426787503165913875158361190113190564515282401830177770104414561937368442616914765277879526870000704829,
           # "9BB734774443D77557A76E24B10733787750D90D09C869CD606D54F28978EA6220DC9948B3C9E89284F8551D6166F3754B6A3B890AC9CDA9E37DFAA0C1317E351CE5107C4273795949C6CCE638314AB1A345385D7642CB8D055A1F410C7D7E24A6F0A2AAB8184E773D21B3754A947541680F2C1A8D6BA5BEFD3B6E1FC28EC0B61D55B1454383F2C3E8BD27170A25978608F6788B90A2FC34F0CE35056BF7520795C8C60232CBBC0B0399367AF937869CA45CF737A8A066127893E93166C433298DD6FD009E6790E743B3392ACA8EA99F61DFC77BD99416DDA4B8A9D7E4DA24217427F3584119A4932016F1735CC63B12650FDDDA73C8FCFBC79E058F36219D3D"
           e="010001"),
    # lk/files/pbp/keys/toolauth/da_prvk.pem
    SlaKey(vendor="Rowan",
           da_codes=[],
           name="",
           d=1210800584417750177622225735984718598213674162709141457567915149885298365559226058492240670180442503126882927910291291284113126758448528324019586541382153137814459223153459086679768913232854381546605790792132077332337818665318907462930356255182550564542013995493412305584438476568679413322808438037530835256522423428140358831386651465369440522117078360972314369088338159176728990749392227098932428794339939357216839572105087971607687215784983988883165794965027840715493415779758904882971873983231477951930598633936202022375642415578335791138329660663054442196042932573329831780799189015084095052298427519465699207809,
           # "09976537029b4362591c5b13873f223de5525d55df52dde283e52afa67f6c9dbf1408d2fb586a624efc93426f5f3be981f80e861ddd975a1e5e662db84f5164804a3ae717605d7f15866df9ed1497c38fdd6197243163ef22f958d7b822c57317203e9a1e7d18dad01f15054facdbddb9261a1272638da661fe4f9f0714ecf00e6541cc435afb1fd75a27d34b17ad400e9474ba850dafce266799caff32a058ff71e4c2daacaf8ba709e9ca4dc87584a7ffe8aa9a0a160ed069c3970b7dae3987ded71bd0bc824356987bd74363d46682c71913c3edbdb2a911f701f23aee3f8dd98180b5a138fd5ad74743682d2d2d1bb3d92786710248f316dd8391178ea81",
           n=26433123884405760623193806815200034234220374284300134478557113650244103592156894801934036243043191314932220668373337893699840768943518054887165770540493727579262562993940122639484348856276341640046603501380399717564764697491340852231201451664190145352561615996885996759189655042931892974995635111147787591674779244030701705123316907319188026210527181467872456001364726795668953855056904091524265897097261663776276950667041868655357490620574806649721566811023101021331448255446685668287778876931169746561035660219876374055024555771796510750743718584214662132902386657813231996634762104306913796862511886867431850810159,
           # "D16403466C530EF9BB53C1E8A96A61A4E332E17DC0F55BB46D207AC305BAE9354EAAC2CB3077B33740D275036B822DB268200DE17DA3DB7266B27686B8970B85737050F084F8D576904E74CD6C53B31F0BB0CD60686BF67C60DA0EC20F563EEA715CEBDBF76D1C5C10E982AB2955D833DE553C9CDAFD7EA2388C02823CFE7DD9AC83FA2A8EB0685ABDAB56A92DF1A7805E8AC0BD10C0F3DCB1770A9E6BBC3418C5F84A48B7CB2316B2C8F64972F391B116A58C9395A9CE9E743569A367086D7771D39FEC8EBBBA3DD2B519785A76A9F589D36D637AF884543FD65BAC75BE823C0C50AA16D58187B97223625C54C66B5A5E4DBAEAB7BE89A4E340A2E241B09B2F",
           e="010001"),
    SlaKey(vendor="Generic",
           da_codes=[],
           name="CodeSigKey",
           d=105272702820396244791430990822943687686650086706439182094802684161245153568523716607068427530539551658757632522202595397519445555014122015054152775286121112130187992901001873246518147517018501287877817853621214487446840967119669740935174587493300146381016525579857068336413102530392063882695413776304208134376238500307127729175219728318565706049656683957153622908123450652588151395896344612266884344907045752063472647897505368344357997392550599587482493801228444254687701608742201543008236287929806777162844058771451824914994585348743265306827467585898005704580988826668663037116670670863615530939993573888591619697,
           # "00D57BCF5934CE1A035F4598B42DAD4BC8AE7577FB6D81FA232317E8EE7C4FBB33772EFE378DF7DFE5369BB9ACAB1008FA1CFBA737890012A883B372B15932335B689B46A32F1B383B75F0DE2E1B5B0B9F4E1C3E780C2AB0CD3671EB4E34F30BB4C630A60D168CA124810D0F91A1ACC8EFDCEE52D4762BB35813BCC93878E1D15B750561B78006B4C13A8F76B5F10C941E5776C21192357A9B9D7E02C0FC812D2B154671863DE97CE3ED07F90624A0CADD04079E145168A3558A64786820192F5B638354DA69520288B976296961C337FB18A90120F2B6B365C0E1A57CE4119AE8BC718E08FDA33F1F42AA1C91AED090EC6B5656A66C246F89FDE5FD41A76671",
           n=224227538260596987061523378906147761055607485861740865057268143647031870695191225430707645197080522508368779050589602995457502789787140124820566493351312012924571175233285442066919946498841090672548621946529738823614794645978803204903264225049511251368997368680395615312928775733182901318866694811584675761448416076889428520254576903960381794612562439076330552193376575956152023696022439185516230723442327773005261938381955447895106863245842902039875795002919666692374456222372070722652524404104776699644603726821181300168745869913739418430839025033044410903231979835956848455112995362733801202062095736833308697041,
           # "01C6B6A1DDF05E818E3DFE16101C5DF65939F352EAB8AACA91CB5BFEC15A1989DD7553343683BC30BB38E45F15BF17BCCAB16A41D695A4318F26504675FE83E92EE21C991C0FBA705395B4A34C331842D8A6F69846B58CC67306E3DE27B05666A6C4372E3FC0D92F314805EDD5B1CB7D25BF3CF9CA9C33C36D97B0B37DA8A44A7A1CA651679D8D680557740C7C1CA25D84BDD12136C2930432808F28265D1E33E667389E4806D865F3CC06329534F7A11861EB688545DCCCEF0B04E96735A08368FA31A1F3260073B31299B216192E620B8D1EA468925ADBD627C49EFC3623658F3CF8AD6D8556272E48FA7711E650287DA19196610F036B6C0D394E42C121D1",
           e="010001"),
    SlaKey(vendor="Generic",
           da_codes=[],
           name="SecureROSigKey",
           d=510231269816482056114158705710715578210522531269866181473478530660801160283428181516720179960443600504675262246972061621100981874241540360945641487505504177879951051736662983210412007640969480317903112292222961260987966083610620930428697436828260752704440930479394161146643302948051836529866310212545389341912155586297877782660840511328852133201003027962736992951300799979961572437117497671573813926699087634852245512941314312002761804251620975257408266812756158371659525152186084961012696079041848960756100870466702430500113742466120262275160313924521512238480642900255357029759026041834923375393351752244509844993,
           # "040AB412E994921780E7D3AC4E665B5018BAD2221D93A236FCD3D4245BB14EF5E715B687254BFC5D5C058FF5C33AF644E6B03748A6ABFDFAFF808265B9C12B42C2164826B3A8CD5D6B3295E025618AED68D33E02D75FB8C69FDE6753AF454EAA92F448961C5D11DFD8D2D5125E54C71DE5792EAD4B4AD2A47ED2F144C664A2EC2B5C527D4C4570162EBADF6FA6AA19D86C927257BDA4AC4B471AD94AC16C8A97E9201101EF268E35E66835FE9831F3D18495BA15B3FCD9089B4569F770896674173647EFED86F4570CE6118E48A7EDD6CE2A2ED72A2E6FD615A323708F0881443B6C6DEF3800B392385E060AAC6CE086DDBA9227027F80B0DFDF691A1ED8A601",
           n=520176509395227187349163385852827559722217272289111130844803721982405040779899084252089018014865164679778835509299483580083614104476507850091694763341550421655751852368640438221964590641002688563163715237001683345177156574288925143387243231868174428317947627105159217520184838300455373197962920359647613423306750748753397961136126336976164308663028027388192335570858616384011792407209649047322230641351282661605831394398348188059714837662017796582047719555003732537795302614503005588125125378561344969794073462580047040722136668697394875680386727057349025648863851773493977168557210641267404539971942103942711769547,
           # "041EDF18B2702830A0D1D6D2C0C9833C46D55636929ABC8CBF62D03A8E352697F536DDF250E6CA5F1153A9227FEA4F6C4A91CF118FD821ABA9020FFC6AA05D4C361177AA384ACEE201E1326D5A0E1D566E74DA51CD735841FB5638E03F4E9A5AC58D0123F8E057686541E424F83508D1CEADDE7A893F57B852ECFF27953F53030953859EA265C0C7155F6B730E902BC23FEB58077E8D439606B164635D5AA0C53657BF2143EEE86F06781573BED22DD3E792591A263F2357CC42AEF4B8DB585987A311A022C4442A4DFA1C4891D8ADA1B231A92096E16D9C718FB09DFC0A5B008BB8243BF32C537A6B19542E37311085197B6BE8DA54EE1D6BC28BA94E0079CB",
           e="010001"),
    SlaKey(vendor="Motorola",
           da_codes=[],
           name="Moto G24",
           d=21285935122662264864057010353628894046000309676608514009863068975210666417913992390203952016184103585209090567456567655640013653254389492000330698369108534400952734580691061611305270427006293274003837251061472898178055883452926583157069288429704943815305237986749864472793502417868363277703857993800596854717455227000386605194799101756888463582260204208854094963305819533909836121398593369779835885060955127693070927806187215287927106022661033162137977188155754455157501898266271848816663792930331047190853544282318635289788145216933852870788132800493768471635534748961829891319455548129601994556070502373744923475337,
           # "00a89df958cec69e5e82f12cc64f21b577a99916043912cc47ed278f88cb79ba847e7601abd8c502beef0bc706038a9c5269486c191b65da800dcd465028ccd5e530beb93e02053ac49d1ff4f17be3245b0bbd0ca7ea51558c439783648502e9ff92ac3696cadf09603d1f89c1d1d09095ee5ee68cace1b3a401ef401de86d3911ea96021dc5b5af36e6babf3d48d6a58a9075d5deeacadbfd09f93748929ef466a9d339d92370334e0e50afc0c43cfb1b9f2bff5e3b5a7012b93e92d8644f032993033245eec56d899837d1080c5aba7e09fe2e2eac06921159775392d64e819daa905a2931352ae02e3a21318b207e0fbfa113e8a32a37987243da2bb57d7d89",
           n=27716841114500344765566038575545425862643744318131811064043948092197282887898697011320983157192537336111902575339275485241294128734436513028266336433137947170436497709458387570684339829820815050334574753587565304243810940252616667269969799046662551962481559713449481789622057339608569750866816501553907454007165785963469328609654526247959103207964930235031378798604098615371009080202164164858220545960117335763391558057794040598744405105128408738639336292387025499581025311622360049763406211499993117726494678511901460658618003333913774822703760038655009487487524407468957728357988965151989430864537201197625439572663,
           # "00db8f46cf8da80af8cca1aec9ff7b358cfe4cc5659ade5ef9c196905caaf979658349284723bef9524532b21f460c0897468be95d0aa92682144d1bfcb84afc7712ff3b5dc34153e5efe64b465a6d8cf2bd8c2fb1bf27d9c77f26e90baa3ddada18525d3f689441ef7b6dc5c4b8c496b0a9c92f29d26dac8ff8b137d6a93cf26ad391bf6124207ff9eb26e10b65269c6bad38eff0c50aab604a0128b874f24263037c605bc9f855252f78173141d166b632dbb549370af71efdc522532cb55c48b9a39a21ee8e0cc8bb34c394aec92155a16f95b646aa9e5f88c989eaf2d7f615bf5afe619e27dfab5adbbd7999db9590ab0f30c95c98da39616cad6494be52b7",
           e="010001"),
]

"""
    SlaKey(vendor="Generic",
           da_codes=[],
           name="VERIFIED_BOOT_IMG_AUTH_KEY.ini",
           d=0xDACD8B5FDA8A766FB7BCAA43F0B16915CE7B47714F1395FDEBCF12A2D41155B0FB587A51FECCCB4DDA1C8E5EB9EB69B86DAF2C620F6C2735215A5F22C0B6CE377AA0D07EB38ED340B5629FC2890494B078A63D6D07FDEACDBE3E7F27FDE4B143F49DB4971437E6D00D9E18B56F02DABEB0000B6E79516D0C8074B5A42569FD0D9196655D2A4030D42DFE05E9F64883E6D5F79A5BFA3E7014C9A62853DC1F21D5D626F4D0846DB16452187DD776E8886B48C210C9E208059E7CAFC997FD2CA210775C1A5D9AA261252FB975268D970C62733871D57814098A453DF92BC6CA19025CD9D430F02EE46F80DE6C63EA802BEF90673AAC4C6667F2883FB4501FA77455,
           n=0x8BC9B1F7A559BCDD1717F3F7BFF8B858743892A6338D21D0BE2CE78D1BCB8F61A8D31822F694C476929897E4B10753DDBE45A2276C0EFEE594CF75E47016DA9CDB3D8EB6C3E4C5D69B8BCCE1AE443CF299C22B905300C85875E8DBB8231F4E9949D8CF9D8E0F40E93F29F843420F22CD9D080A45A4407F58F3609D03A7DB950D3D847B8B4E7D50DB6359D37A2DD730D3CE77F8FB2A33C095B0A6CF3E08593E4F70254DCDF671790F530EC07C3CD1E80199CB42F24ACA92DB5996F2119003F502E16D88EB4E4A8DEAE4036558D2A52F5C9960B0FBBC6F6FA75EFF6F5A173CE1A82539A35973D568B8918ED12F7610748BEB0239A5006257E19574C77F4133A269,
           e="010001"),
    """,

"""SlaKey(vendor="Generic",
       da_codes=[],
       name="AuthGen_SV5.ini",
       d="9a3c3d4da0650cef38ed96ef833904c9c13835199367c7b9cb03a55e7aa482016a820dfe597cd54dd1f81fd879cf070ec0c25899ac5a49822db09675a92acf6a01e0f8f538bbe66de48ca9bdca313b616470d9ec2914356d03c95f7d9236549e5a21457e4dd5fcaf09046c47ca7436f06cd7b82cb6d2a936fca88b707f6ce28f33110fea1ec363e8482419db901cb0d38e574fe0c02ad117166b40ec78f59aaa7f3eafa425010a95614e046651273a6cb1371380c4e6ce81bdb892db6ff4892cc4d8c613a8fb3fec1e72c279052896872fc23da07fba63783374f3be8e16a15e0a04a139108dd6ac239f191135f4a895e27c670de065d2248e3f9c7e920fd001",
       n="008C8BF38EB2FC7FC06D567DBF70E9C34BE4281C4239ED9C58A6B598C3AE7821815D94D0B463463EEBBD69FF6AF990AE0499B6C3B3CADCD91D54499CD66E5314DB610FC0C6CAEEB1F16B6F2D451E3F2B2D515008917FCEC50ADA4CE0699BCF247D5AE2A1DDD34C48624A657CCB11CE5F8C6CE92CAB6038EFC2A89E42E029488C02C3CF21947C86D51BBA8EF540A2A7CE85356F431891261D860B518E89DD73B2D240461ACB66BCC213403145DE83F6963147E65274EA1E45DB2D231E0774ECC86E4F2328F8A90835C4FDEF1088DDBA1D8F7CA0CA732A64BDA6816162C0F88F02CF97634D85530968CBF8B7CE6A8B67D53BBFB4910843EA413135D56FB5074445",
       e="010001"),
"""

brom_sla_keys = [
    SlaKey(vendor="Generic",
           da_codes=[],
           name="IMG_AUTH_KEY.ini",
           d=53263609703361838109020175246245674288689696180717044730191379394296157153471314070234975025293094736924576539329759162789285198048982284746792708084633974177752907657638262795669080787563606500577176119733003521032115800905758129538382994083393168709839480493483958362740062831235997441166047655264922702317,
           # "4BD992E9A2230CD2ABEF49E4F6A7E11D7E2ADD24847787B320239829C560D5EAB94B8304317C938E9358E94758AE60D9B13F2913DD1A749A9941FACAFAB574D70EBBFBCC0133A4BE2134CBA3CE7EE18A6D3CC98D33DAB06AEEE512F405A3248EA316ABC31A2758D4C5A7B9DFCC02C2508A492EF3760A0D4CDA827CFFCADD11ED",
           n=67410872084066677947412379064812829613152127505081840767868845593418443337911081054032975266644140929915724710002402839768290443283242608490248801918441527464704825031470913531037102725978151632284726605760530003998763143197899397314949514726901846049117611827351641628287203049431807690418659042420210896337,
           # "5FFF0B70D5DE3FC5BF41CB824B4BFD14820571CE57EDD3E7C668CC570E718DB07DCC7A6CACD0E80DADC38AA33DB37816839D97980DF3E577A6E0B1169D708071E17DD259CFE538DBDA804A2FC07D795841F2F59DEE023A9919360D0A3F4647FDF5657D9FC5944C8BFA2802336BA23AFDCDE8D546E8806EB532AA7F95A01D8DD1",
           e="010001"),
    SlaKey(vendor="Generic",
           da_codes=[],
           name="ROWAN / 0_2048_key.pem / CHIP_TEST_KEY.ini / lk/files/pbp/keys/toolauth/sla_prvk.pem",
           d=1210800584417750177622225735984718598213674162709141457567915149885298365559226058492240670180442503126882927910291291284113126758448528324019586541382153137814459223153459086679768913232854381546605790792132077332337818665318907462930356255182550564542013995493412305584438476568679413322808438037530835256522423428140358831386651465369440522117078360972314369088338159176728990749392227098932428794339939357216839572105087971607687215784983988883165794965027840715493415779758904882971873983231477951930598633936202022375642415578335791138329660663054442196042932573329831780799189015084095052298427519465699207809,
           # "09976537029b4362591c5b13873f223de5525d55df52dde283e52afa67f6c9dbf1408d2fb586a624efc93426f5f3be981f80e861ddd975a1e5e662db84f5164804a3ae717605d7f15866df9ed1497c38fdd6197243163ef22f958d7b822c57317203e9a1e7d18dad01f15054facdbddb9261a1272638da661fe4f9f0714ecf00e6541cc435afb1fd75a27d34b17ad400e9474ba850dafce266799caff32a058ff71e4c2daacaf8ba709e9ca4dc87584a7ffe8aa9a0a160ed069c3970b7dae3987ded71bd0bc824356987bd74363d46682c71913c3edbdb2a911f701f23aee3f8dd98180b5a138fd5ad74743682d2d2d1bb3d92786710248f316dd8391178ea81",
           n=26433123884405760623193806815200034234220374284300134478557113650244103592156894801934036243043191314932220668373337893699840768943518054887165770540493727579262562993940122639484348856276341640046603501380399717564764697491340852231201451664190145352561615996885996759189655042931892974995635111147787591674779244030701705123316907319188026210527181467872456001364726795668953855056904091524265897097261663776276950667041868655357490620574806649721566811023101021331448255446685668287778876931169746561035660219876374055024555771796510750743718584214662132902386657813231996634762104306913796862511886867431850810159,
           # "D16403466C530EF9BB53C1E8A96A61A4E332E17DC0F55BB46D207AC305BAE9354EAAC2CB3077B33740D275036B822DB268200DE17DA3DB7266B27686B8970B85737050F084F8D576904E74CD6C53B31F0BB0CD60686BF67C60DA0EC20F563EEA715CEBDBF76D1C5C10E982AB2955D833DE553C9CDAFD7EA2388C02823CFE7DD9AC83FA2A8EB0685ABDAB56A92DF1A7805E8AC0BD10C0F3DCB1770A9E6BBC3418C5F84A48B7CB2316B2C8F64972F391B116A58C9395A9CE9E743569A367086D7771D39FEC8EBBBA3DD2B519785A76A9F589D36D637AF884543FD65BAC75BE823C0C50AA16D58187B97223625C54C66B5A5E4DBAEAB7BE89A4E340A2E241B09B2F",
           e="010001"),
    SlaKey(vendor="Generic",
           da_codes=[],
           name="SetRsaKey in libsla_challenge.so, secure_chip_tools/keys/toolauth/sla_prvk.pem V5",
           d=17927221772803595589677548665100382532460666845948522915567191138741093137994346758223790428815724337119399848617323634517831321838857002928203340809368584412532275387119052050346117829587547335823645786993417933752612674263704610122642616766124615147357260453258785716304089937180892099709020291991047354911782870897115280093452505641278117790282224075714432226317608734573868880945885289415362825737676921878537194428184324625476692278084544343444435636989350942762677823240040394539138366047050728975383606379913663863336393801544062712098911021002990407295958911986185315713867112814558339220670999677054756992713,
           # "8E02CDB389BBC52D5383EBB5949C895B0850E633CF7DD3B5F7B5B8911B0DDF2A80387B46FAF67D22BC2748978A0183B5B420BA579B6D847082EA0BD14AB21B6CCCA175C66586FCE93756C2F426C85D7DF07629A47236265D1963B8354CB229AFA2E560B7B3641DDB8A0A839ED8F39BA8C7CDB94104650E8C7790305E2FF6D18206F49B7290B1ADB7B4C523E10EBF53630D438EF49C877402EA3C1BD6DD903892FD662FBDF1DFF5D7B095712E58E728BD7F6A8B5621175F4C08EBD6143CDACD65D9284DFFECAB64F70FD63182E4981551522727A2EE9873D0DB78180C26553AD0EE1CAAA21BCEBC5A8C0B331FE7FD8710F905A7456AF675A04AF1118CE71E36C9",
           n=24768553458927569182264098384414119743435748352122448536260714781483609591751565373536682899405420594113926591732550627920208945701553102158862914443419182389461952820609704105060261920473892479137225127905294215755138122340474523740015372088152395971589602157167092642340489980247161917120903612864283134792211128271400444531978419218384701537034538417051660863427537718871156919447450593998355621418942546492747019880888003141225040123766139214470940969962244251360705865766394055854812601692583755861695306804617391695946006993172614987479125081044934005567736503206356076476324144050995732879986002921180459717263,
           # "C43469A95B143CDC63CE318FE32BAD35B9554A136244FA74D13947425A32949EE6DC808CDEBF4121687A570B83C51E657303C925EC280B420C757E5A63AD3EC6980AAD5B6CA6D1BBDC50DB793D2FDDC0D0361C06163CFF9757C07F96559A2186322F7ABF1FFC7765F396673A48A4E8E3296427BC5510D0F97F54E5CA1BD7A93ADE3F6A625056426BDFE77B3B502C68A18F08B470DA23B0A2FAE13B8D4DB3746255371F43306582C74794D1491E97FDE504F0B1ECAC9DDEF282D674B817B7FFA8522672CF6281790910378FEBFA7DC6C2B0AF9DA03A58509D60AA1AD6F9BFDC84537CD0959B8735FE0BB9B471104B458A38DF846366926993097222F90628528F",
           e="010001"),
    SlaKey(vendor="Generic",
           da_codes=[],
           name="bootloader/preloader/platform/mt6781/flash/custom/oemkey.h V6",
           d=12180287029548245287298775942804593340279288451052798442577648206382973334211669991249360439869082278033823658376213185025907624559029574959059240134802920683752164636101953508132652625318345384583854228833745190764920406472569093546914564415986941509597382031696394373785092963578708071300200586229902950898835011339655604046121001255529998310701229563671143830565068871289127484312410431872593242113529332768682489754110502444810388796263412820875744519208161068087537993525013618898005167221382314457476527462596572937835500557389408306844112183603013797538939696905222870787212681990900437039173959784328388619905,
           # "607C8892D0DE8CE0CA116914C8BD277B821E784D298D00D3473EDE236399435F8541009525C2786CB3ED3D7530D47C9163692B0D588209E7E0E8D06F4A69725498B979599DC576303B5D8D96F874687A310D32E8C86E965B844BC2ACE51DC5E06859EA087BD536C39DCB8E1262FDEAF6DA20035F14D3592AB2C1B58734C5C62AC86FE44F98C602BABAB60A6C8D09A199D2170E373D9B9A5D9B6DE852E859DEB1BDF33034DCD91EC4EEBFDDBECA88E29724391BB928F40EFD945299DFFC4595BB8D45F426AC15EC8B1C68A19EB51BEB2CC6611072AE5637DF0ABA89ED1E9CB8C9AC1EB05B1F01734DB303C23BE1869C9013561B9F6EA65BD2516DE950F08B2E81",
           n=22503931863314821588681209939207956521816749188561323058502800250950634962906805824777552298931552922178132078794595272526017929373170987090997559221768691160663780270529254822465315040186299094200272175098025444495957055677513551076458609836787781509767806332072806666518765154320528610419520912825584959772992850209334881093031096920109728031982275244804724362688387461024726412729022872380997164853683726352645069098177129002823368703377814735777096536077336746410578873620055970129075368735411176640443142113244692593524081582031473788163061195037485533616694214309200334237324365629236276801067976705094852955351,
           # "B243F6694336D527C5B3ED569DDD0386D309C6592841E4C033DCB461EEA7B6F8535FC4939E403060646A970DD81DE367CF003848146F19D259F50A385015AF6309EAA71BFED6B098C7A24D4871B4B82AAD7DC6E2856C301BE7CDB46DC10795C0D30A68DD8432B5EE5DA42BA22124796512FCA21D811D50B34C2F672E25BCC2594D9C012B34D473EE222D1E56B90E7D697CEA97E8DD4CCC6BED5FDAECE1A43F96495335F322CCE32612DAB462B024281841F553FF7FF33E0103A7904037F8FE5D9BE293ACD7485CDB50957DB11CA6DB28AF6393C3E78D9FBCD4567DEBCA2601622F0F2EB19DA9192372F9EA3B28B1079409C0A09E3D51D64A4C4CE026FAD24CD7",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_U91",
           d=93625883854376320788947095432214306675379825104215883927353347605892438899776045946824279934666282364153668560368409844912618364570632276153791567052817668248317664538581936972440869233573201583262690183102487448915402791919214599869844397441399024224589063472601020536184898374803388767212584931070027381441,
           # "8553e31d7a73f6c9294e961815c23f31f2b5ea1116e3c613ae12b26cf285e4c5ca0e2dc8e17d52f96b30cef6ad544e43205933f20ad17eb8712097aaa23116c68eb6328980b8ba26706105656fa65315688b8232758607b8936d0abc27dbc97d94e95b4f1957fd1965082e5849c4185ebba8afc7d558d4f5f001ac5363423ac1",
           n=108558783308521654144319482561498691068687725095626417765594924844261929304954225314338183965150280374005501865483607728071162905751132788689201336292780027290016561766748432968982210515432193354734933807016713020558178133671416298957043155677851645276045491982363990169674968469943338407175689570277245867213,
           # "9a97c44b0768424b6bbb0b6aa987a2d373448c6fee1f61fb81f8cf53d70856f0f77e76c06a6901de90ed3b4d9ad4b9e04eaed42e5657bf2fccf390fe9f5abe1abe8575f07916da69acef95d38874223ec51cb501148a1feea2be2b8ccda08672aa423a4099203c6aa4777fed7353c57696b8e0d4020bd6930b828b9846a454cd",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_OTMINI",
           d=75667112208908967404790107558958939148253538629251049419671255455469452157072004362359261704523868684815749092788174604846859566259140723765092442303504020114731094250207768005911228678554496364511007939863181801117502287023553782799258861099692592585685682517270969044712437092884431318726649225709058742337,
           # "6bc0e84b4f38415bc575dd0d5248c2d182ec55e2ba7a11dfe86815155c709a25bbe34fafa6a9c19344adcfb32eb3d2eca465c2dc0fd7528a00cc268c6657cdff0b0da1b2ac6a95b94865facb7e1494cedf44358e29ec7e8f091172e4ef29856d1f45032aa644efc273f141c10cb8281a12cebe202b65f176e1a145c326d75841",
           n=132356354467538553842253964264325726504900103054560707444337552213550630848961164488641709619915679984967297843276979717560750404193698604462403271979201532958160128512184014029648876556663718854604319078490890745724660376017563685067709902074309990052237617002932003168977488795959842461857350712903088042701,
           # "bc7b5107bcf46c2cd7758f4bd4d4e9f06b731d9cff383dffe48156d1ad91ff74a7925fa3027669766b3d4c6e28c1c9310194c34a59e672c8ced38588e998d7b162889dcf06668345f93e4efca34b5fee5bb57dfc38d7623a48f31b382de2db656ec1f3b5267a9a8f5e441c61448a283e4717ace6983d01b163e34f959c9972cd",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="ST513",
           d=111392911285336717582229537954257206772373107161595075101307719716793885987898960342281529394568741429335356994722925172709728613102105408382169050197289273081964184215237327814335151345535076621000243555715248239099088266370499457838495897712120325212506478191879178079420748364979373067822199773515423102977,
           # "9ea0f7256bcca9099e5db80757a5f3ddeb3292475c01d2e6eaff8da905d9537a5875e874d26872a8c04b552dd310f194ef5a5ea445a50d5c1e6670e5126ef01e5fb1af24a67d07b5a9f72197bc66d5743faab54759fbedcf1fd8ac1aabede2c6fb29601b4734334db92a92fc25f7ed8700d307b74a2c435c9ce5b5caba4b3801",
           n=156956424727105101053202984797646941869646893791912978089834967902466479746924109132097191948999217557928089339593144716285635623427775244004605543575415984164076702495971922168636965212810398977029455808472026209983831470481571455968441357575493287458112621918694187804044694980332596702339034118335370236009,
           # "df836c16bc8e129dac8e6efcd3f41636981687c29c465b481cbe874ffb14d592de024b70f4fa20ad96c96e4e3eded3625f314dedb4d8635782f6d668d04ab1167982229e03ede17a7857a22cbf72444a6bee2bba54f32099e0eabe654c3da4933926db4d97dcaeb68236df4b3e51bd3c4bfa8b2d47c2534405e4f1c1d43e1069",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MPK_U7",
           d=138696367262317347423966071230074992282855092111249865557711798627787735553684974485001617876136157032356654276965094209402108454567077317151504850959649649666145231422562356699632485913983193443613290821102108993580215446956669285045159111495732744761495417618077088736545150662302572308117729887324786476417,
           # "c5829b5bc34253f090db831f5085cd5a6f88da7f6f90e3a3cb6fff6e53218c5a616719971b3f64ef02de526719a7b709978bf1ed48c821981b32ea77c9e536bbda206fad74946d02a20d17120f89419b0daee2d8a47275768930ad53c876afebffb6805483c1ddcf6c19f3566f0de494838afb51b18080beff66364de5294581",
           n=153818388013512234712905953399760124026829697967559861222048907616159421006901752679181491026979648076446983642154218350451585301163612991744647060908765027854147233663938214928326292829749520256611472551366266499923042438885074888386940283590700434197084822888503495750159956884122714906459211015799254258491,
           # "db0b6e89fabdc24e6e7379d25b0c402686537ab6375d8b2407beaf44cbbef27e04e90b556801bbce5eee2a7ec636ac825667dae3578eb7bbb66701bc62ee86f28fc14d57e8637a2ddcee00cc3ab87dff4155250c2dbde9ae62f3d7a9d5e4a265fb0a8b23c082be263d7788e44d59780b47a31b25dc588f81902be419f917933b",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_U8",
           d=83418066194813900180248883887574708601626096602946796282943281140509135480181253528180777921963743017193992523629430770578226987739273963253460607144870609204588522019327944707362234222686446692358659870090822057807725378300405302775171223402436494192103250375171142161994478927891994814483338398858889663325,
           # "76ca90a16bcf7552db2b716b8531fe5617bfe86635627647e3d27291fdf47e67ba8f953ac362dbbce2977f05a9f24aff4250f8f3a14d3ef09b7b99c9384aad0c53104f87b47d7daea3ca725beb233d127ec342ce0619b16bd3d5e44371cffce9f23178ff48dd42fc4450ccdb3e2d63437ef9dfc0296b12840ae85d472cf0135d",
           n=109385337926119341645750637334480155217686818325043001820669757620311475058817479590955874320649429282655569099443806486952225176483499024184998655644438387343091998867423653359144406810782471389052010412107386111913034467864879293926497452621867540950733818835741118446270825541140076305757038537144797861097,
           # "9bc517a0dfa87a7e240000c5f42cf31905ab93d4bcb95694dee85282867d5c83270aea0b0948d66eb39d8500aa6c8b1069b8ee784f75948958f7bbf627d6ed5f286fd3bd4df60a6c9490cb319448b22765aba9329820eec50f62f1ca0b6b3322aa27747b26855a1f1719cf0c4060c9f5a6a3a60ec60fe6e04e7b044e5da994e9",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MPK_U91",
           d=93625883854376320788947095432214306675379825104215883927353347605892438899776045946824279934666282364153668560368409844912618364570632276153791567052817668248317664538581936972440869233573201583262690183102487448915402791919214599869844397441399024224589063472601020536184898374803388767212584931070027381441,
           # "8553e31d7a73f6c9294e961815c23f31f2b5ea1116e3c613ae12b26cf285e4c5ca0e2dc8e17d52f96b30cef6ad544e43205933f20ad17eb8712097aaa23116c68eb6328980b8ba26706105656fa65315688b8232758607b8936d0abc27dbc97d94e95b4f1957fd1965082e5849c4185ebba8afc7d558d4f5f001ac5363423ac1",
           n=108558783308521654144319482561498691068687725095626417765594924844261929304954225314338183965150280374005501865483607728071162905751132788689201336292780027290016561766748432968982210515432193354734933807016713020558178133671416298957043155677851645276045491982363990169674968469943338407175689570277245867213,
           # "9a97c44b0768424b6bbb0b6aa987a2d373448c6fee1f61fb81f8cf53d70856f0f77e76c06a6901de90ed3b4d9ad4b9e04eaed42e5657bf2fccf390fe9f5abe1abe8575f07916da69acef95d38874223ec51cb501148a1feea2be2b8ccda08672aa423a4099203c6aa4777fed7353c57696b8e0d4020bd6930b828b9846a454cd",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[0x6577],
           name="MTK_6577_HUIZHOU",
           d=7755741263210084883367096370736007977050044495749727943522009135365928255136269140711318519350303572086234846700996218015875990470004016470158007458955435030160031513806627235934895985463098611857308473038059281697464451826650324895885792023119234075084381381154308139629775896324842062722275371796808420067632298166095712108351593771078813561672983140819649883416935488401219046917270299938522364840660079634364761203077009496983205491276173458216413570665412581404410480602542317250189822585156372585506174550140677434505383152251838015264154734823308542140990659162634889442897660075192996883610247445997929838097,
           # "3d6ff33ae0ec1d029db4a6fb9ca3e41890f5cb5a53bfc0ab3cb2053d85243c7715a07ebfad719bea67c252a223ad0fe65074a5d26ea14ba63ff8d92e553e879b6ce51e065f05b23e5d27deed116ec751c9556ea0cec11e80f3bd206da9e9072fbe1695b19a8a9fcb576f00f7a268df8d6d262127ab3f3246941004f25534ac8d2f418815d15f4a5a663a2f1383115cb3e8bd263ebcd92c5bd1b92644497e15a1b41e77e648cac179182d83c496728fb52b9a1c600954ad0c3eac5d4633d519c88daf775fe090c2f2568c7c91a8938a2859245f100fce764033147d84d79075a81331ecdd170d2541832ab9161dc473cadc1dfbc17df2be89fa6d6c13d9db3611",
           n=21733784374524279856383007433571033257447674610593488700329324526637740627565128946628369684126260110480847186395997312186533748981598889443055771789351449333886260970639454789475575242626035563573456424402244458229218274227698394095081333724683167724753286123774314471583213875975592177903525934983000531319934994784644932160326418349874471860674688796966819097947123283194013934739991177528841688559537220210159750694347629717820019308132468353116981595454845181474798252931760308688791097184614507568098981573130876647599178617397795979925575486008926466171354290300050046400654182867462218349779803692058223423211,
           # "ac2a2c19bf4beef4272df8899cb648f90453e53faa1dd8143327978620ec74e6068a8fd051fac856a59ff0a2f3051b7512f55fcd6eea57262a5a24e141b2a9c105509b79976b952a4cfa0367535aa1db83290f18f62e2f604bfd5fee3fb6fa863ca5546e359e0348937e5b62e47f645e9552ebd2e7e516c13a192a6075c55351192dd545dd90c34fa28c695d6643a2449c0c7acc9d003b9bb4f9d249bc19beb8ffdc2d6115260499156461eea896361aac9a24ace3bf6c81db3e8c32fd6d74d876882382618c7ae920ce63b0c33a3ed6a59642acdcdccd68f2e84f6b1dfe8e4dd33fd78208c750f877a8eddbf32b7f6cd28bc7f62a79e1281cad49b29ea1aeeb",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_S_2019",
           d=13231399456461146478930253731090437983706697024112619616667998880719099694011148882178214374616790772353315320035731397195515107806341252802378101686802134865814271612464111972101015318369156566274856567773588866905775151442683709554280795980500807887668932023490879569132627711589434751046934306580740959122452958066436599914648172185929515929537370473961879198329027560425162708465917434582670394218300387599745385140177097369928100225393873907934234864635775919903249249130878298568085965110740369402657841254932115263138005173226045946011057525704233958193203337223037942532942400026671180419082955189663871571681,
           # "68d01875ee507057075dd8cf2e3007aebeaf767f350c130684911c483eb918a5e235ab71c2eaec62aa7bbeecdac518cb8962272e83a2943cb0e486b66da8e244fbf3e3d8e4a065198032fdb045f011784127cdfd63d285f7f20dcc37b0ebbdc8b49020b9a16333f196e8e3e8246835b1e76615985ba6e221241d096cc5bdd7336d8b22704dc1576ae0ac252fea8dab129756a609f347d60e25d8d085cf0c8775631d3c0e54e50fc67dff2c55148b4e78cf36987febb23e14ffc1da9cb0adfc139d509826aa98f6fe0e25ec6ab6442e5a7cebbe6454ff06b897467512cdd8f0460201125d0bc9cc2bae259840722ae56d16b06f9e0515a2d128a23b5b0a1896e1",
           n=28829916423236058141819670150092228011775576928960328207246846287841200766221446382449419358642882400682200483050127288317124596725320389816791463868872648204696785646188593201528833064697101332666908533884490111390510875061478897302310676447040409818942708891000790422310160992567218075648744386275085452424311113791117678029418559326848958424608166600574931221767825212851758867780060773806321764368227493873109161247249726949832795882994266103597866374386049781952542624932909515630102388499676280443008483364051025814131844991840486210463900802019254954434189872220830389528453385770163287848934677989140140719951,
           # "e4607e6cf78f5e4857bdaea441f8ddd35a7576f552b4ad2c8b4ee7f578c0590d747b049bb5014e06f8350dc6b78d5e0ddff1b4bb8af695e4a338a154596555738cccbe6b58eb43ae221df9babfe9dda6ca770c25ab42ff986f946756b46ec553daf7616f2843dcd6a48f48d9011c050e7ed11c99f61624f057695d622088f868bf6a3966f25bd8ad58db81623fd63f2b91f3ded1a5be0efb69a64bb40d8bbfc251d9c32fbf0a1bad516751e9e04439392c59ba6f856b5c0bebe0dcc67d7d4f25da5342aba94680583ed76d94823c6f62e5e7484f7e2d2a467d167ad3f5647f958dbba3eb66f756c851a55138d1ce465333592969470fa8652df2e38bc380ff4f",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[0x6577],
           name="MTK_6577_SHENZHEN",
           d=13776037267019760086765350767904162461239166343410337323745453741947686534058956786933899829274739416864008137348962739829540467623607966623470699642645456074743735286745484908321692339690492844565965054488518373409397926637671631537227659521290480787426902435432749334330955461363109002168213069040463724696439504338820378227323401764114526971849517423048148252194706876147207471330651074021511095609055087307498982843410112450672871290562784148952894860037039960454364256139285482028199703047595153538650907759323274772849198022173214478196228400986819061556923620612069670612169684130051857656025967855665104843457,
           # "6d209285b39ee78c7cfa17a34473855463c8a42d7b494ff0d6885c16d672aed0219193ef388b5aafb3ab10bef394d6fb7831b122ce47564abb084f68f3f7be113bcfc4e8ad3774fbc8eaa8a6fe030e96a56022cd0891f59eb2564ffa2700056e50a8cce72357d3f7ac7ef7b4fdaa69e0ceae1ab3d0f5b90e00414a3cd7bd17afc3b6463ef43bfd22788b68fcfcc2964421b1b622907d8c75e8d83193a579e50c26b0beb93e53e2888cfddafefa03c368c68e6d357087f1bf0800e1bb4f0fc97c092a7e7098cb60cad71e292b506c0cd1f428aff3192da6818351a780aa1b4cce0dccd15adad815b610f445a6571d3c65d2c44da9057b5c8970cded0dfc3072c1",
           n=20522825840349927641533478661486977069063267063286126504507769637343733687548232654693739614274836269390264168426963336065843644904718933183360593800692290752193130102915049200688278570337693888759766588834652405849511545691332190308585404665548605186524752339310694970086557284764367945878754771451738296177558742661365767115380707041044713312570238666313817877173613509368031125797706448344475147208454133162056435893334130763512975781089013439446062070571280906489140469414463598276628528621949651115296598454325052215517159403385800826297303513527429612764864262194782841301329000667576565899107971299037523361911,
           # "a29274e3085c260de63f571646cd2c69737ba5a0bf604ad31cf6a86d6a9e08dc931ecdddb7404f4c9255c72b5debdb69114146bdd7edf6b38505b19c4d18eb0e71516d4faa871cbe1d2e24e15c1877b33587a8bdd1e7dfe1b17235d1ac431c27cae07804014c287fbf2479e6b4b80665898f7cbaa7edcf23daa8dd95f63039fe7eb641ad7c05e221d29adc62cf84893ffc6acfd44a9d2cd60d5e0f94d1c29d317bbddb3f5a324648069c72857cfc708fc9bd8a3f7a98051fa9835af1f9c71d80236334ea51cbd52e57e5a7950beb394d9c97bcc32591d9700106b0abfe1dd2db9617fb7dd2eaa3885630c3ce1dfcf087c814b480f30c411f3071f12aedee4077",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_S_2022",
           d=7560870156650794587598602000865124794652389856763606576357196011242296520465919156012259790051986496406756867133263700108656570180103863100521714510383827099663445871653440103449942162455778103712749547433303734653695479290637743495822872437892421428986602957226013123721479154515057025413184966706523221701307406766136439589930132250893464907717261703687625513979036960151497748157904806668904729977718123078989078661570770789783944829971705313535975326990948541078634967444780034936102685527826275587082557866213379656565819637472891390428368448515955229335236971347202859635167916697308403281054519785568721073793,
           # "3be4c4d89124e53d12cdc922c0c6571224e8925fba160186068855c5032de6655be49233899432008faef8ba5037f1a0b237e169f6f9f05be2694bf53d04b44507fceb1480007d2f49c8191ced7528e6b4fb06070851c85f2025ccb60271631def9f831822b351ed17ca9a165aae97516a6c3940971d17e927f3befb43432c1b689cc660a896237f090d7b311d9e39aa1eee5a4e3af00843c965c30ca9aa5dd7767809d27d4f66777661779d2a1fb90b014329a1973e67b8989de924e8ac98673667e4f734382f87f0dd0300d360142afa772d5beca2ef248e90a7bd32240c4a5b5f41aed3f4b63f90642f138186fe17afd713a3242eea7b2dd0f32b06b67681",
           n=22155901965411273189691463417424443714157553053553162718297409120893645743875472556564966235664522200536983000282258131635190057674646406260625602676817566583082640379635658576337977174194693922782135796563220516789592695116053020142443263624509529317750726492632292514613663284124895697496262157972072988179979988433323606505232817289344825812599823903096521254828599415501829918963049690524551846377284415707043230980500813568313099689325279644796975809875507480011104254716579662627201106576553072636334055927351644238896391941466538196803480223610084375734167869734775977880476749119121354117217442810741264073077,
           # "af823063550d6e5adcca01a1ae1fe357f73d7e5c60cfa25e4beac24304b70623654fd13547de869899be532f45f3c5ff26b50292dccd112dda1478721c05304445058499bd00f6b104e16fcf2d0af55781be147787227eff54a25dca42b9d6f1fc8f4b821c099f483c402addd178330167aa9b1021dae121bb2bdcb0127ac47ae866a1579f2399c70e69293ddd3b0bacec2df9dc518aa0c58c2d7561c5783ac32e57b91d16d6c57764755894963733b85f19f9a3bcbf624199cdd1b31cbecc5448b132c3799e2d0e569f0ba61245796db5876820ef125f4a230039c5cd16b2414855bf3a3b565f81787a4e9b264c9bc855b4fe7ac17caca1bc5f070594a9c175",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_B3G",
           d=13340647395324487193196337701686639933472937662349792852199775152034262121195527649662464128649043343433414034226426985908261232867544266918630424503976232433231927065119099107251427316794166050303660329435015956161484550910117053431824110680625162793447269954057753377816666678800722770483578272504611065473,
           # "12ff6a160cda225ddc898cc6ef7dd3c69d05dc24d23b7a0334568dc85191f3b63d278ab1c8449507dea8533496e04c77225a12a27b7abcf34d10c3cd67b1b41d7c19c44114e344a74396541d998d7b76ca06d0322bf3333684652528df22021c190bc38acdac2a3be6e2d0bce7f1e3c77a71750ff17895cff9c6225275a3ce81",
           n=145717668057896819530084730659239553553335985929569729025769444023144906106131882595988151933212058933099275926849557562578286069740041436841047021786181748318405046209865262233020782439663943569614975074468080902427591764234767605998371798512022888959080014675802696225580544100478037898749965094814621565979,
           # "cf8243d13128ed39fadad9ca97c15585d634f4d9b38dd59e4eec4b0b93e4eb2fd2d96c425855e69706d5c11021a8c2e08bff87b424bed2dc3efa9360bf1bcf80c96cd4ba9c39eb79bfa2bf9d4efc5a56798ccd9c6599ede595aea644086605fbe55b2f7719fccbafe0c95956fcffb0ce77a9637c9ed66e067165cbe901eb041b",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_E8",
           d=105122852609409170000076227126135544095202506637734467020636483605832833975276605841771512566939337400164173521600585386403967704289778616449684481432243364899342179204726268287177272149065613305673395884168658380888232353720237208980245894034711631080798431551836775573845660740490838269599082264459557174825,
           # "95b32d61a10e6c2a54fa4e5e020d590f6bf0f295fa87fa03b3d00dcdc4982dc997ad5c7ff872255141ec1b77f714c14587ffb87c985531c937b245062ee03514aa796ad79698c40c49a8b3c54ec66fc20deb874a8bbce87239c414f541367a350d525fa6bdea77e4cd3078cf7ddf22a8aefb0c595a6c76285d837008c0a77e29",
           n=133376628943796197279881436039136275087452795088806360884567569328134635076425889903050676019272531753485876022866332993935742274287358601197641457769484156099478005361448301176231402277096509816934367931593418904836395783519080274699581705867644633415974927176055820691255507670896269003069557853699449331437,
           # "bdef438901dfa726cfc2cca59d12f009108b8e1fd7dd9b91a5cc71fa7b1e36c8783f9de5850050e6505fc715c50bddd59a3064b05214c4365360cb98d080cc38658a94695184b564e8e8dcc28f70eb0122a4bb7662e3a1f34c057ea523819ed02ed46bae0cd9530b0536cbe7a1ba3f33a45feb2f92ff5104dc32ebe94f249eed",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_C3G",
           d=47371979803011856691766041615217421091852437161071009971980252623175881008895315381278810480605866811797914321874570789759949496801929194912608564738874059636696189575490593265629929928301224592178236560626992944634692052232594284887383177076081969383564143974264199951677634531956878004018123479491374400033,
           # "4375be875664fad432cb6476f1c7aeecaea3166a51eadeaa32e96d0d79dd159b6287f4cd42685330fc15391eb4ee83dc6fd22a913c5fd5023d8fd6b71af8b530209b5355acf1cc6e6397aa6e5d2dc92b7d37635d391cd22a3aa337d8fc0a274cdd7d6630395d13517e32c91daab2f5378ed7a1be86c81c2e775c249201f2c221",
           n=135042080919964682557993521937212010704512099792305819074974763643544050181816976256758129728902422498686381509904034182187812534663246352631040778916554579491364814722992672710492874657317605292584783562231632515260252376294570553691383111866063613066647031842637709722803913887712490723909802831493464186353,
           # "c04e6a1be49c5a57accaebc837099b40890180fc046c3dca58745749d0979cadb63b8b4573fafc129c2f89ebb64c4ec81339e862f5638ae145e2c8bc291097e6b90434ff3f3a1e620fa77dcb6d963f53b79abaf4eefb8a5d4378cdf4ab3060a9901909fd455cf850ae5adbdf035cb3cbcf572ac4dce4bc1321562273a461ddf1",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_K6",
           d=55754939170681846109173845271744373200493044171707270723489910705892856215715544140322828168626229373252883937479291472564873606212582360361099003896388582935593560837305101875421331644743119424827671152735084980339681868866669716326279066951191351490398366217737178633388347884200462942657410196045547992065,
           # "4f65cda0c3ac66753c58d748db46bfb8cb8dbd1f849c7444afcf37dc6bb218904c5a2fe08808680d2a6e7587681256a6ed9751046fa42ce44874bf2061f40dca4953c345c2f156e8ee7e2f497ebc59b3ddccda98584dfc999d213d6782f2b0faff59a9671cee801defeb5a51178a7b95c487aa735b463e8b1321b6ebe58c7401",
           n=117894155269696591225944579517787668143534640120061282874277546555207463309329212632262282689786965039519721707768610867828744967747112671839238091835762380909333133683831826504392973933221968787891401574804807411228692802991287368655217479883745324492767503325844043672096650096731009738536227001548548259409,
           # "a7e3089840bb7a9a7a972e8c88d7c464fe40dc4771a2df0da981079cc800f5d3cd45ed9eb34efac6bf7d2aa6dbc1266285f50d7e86e6e0e5dc6d062bc8fe871672139904e5ffe64c6ffb4ff00817ffc0ad4c18787a253ba5f7f7bd8412e5f46e2c264cedf174ed5163943331a658b434c59ec9e11b269e829ab638c80c4ebe51",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="OPK_VLE5",
           d=53324792904038858315192263723283254545483074787890798633039653660252149617980171600267026657529140771159775658738879195140250617228353556116328348849028264703274659951880467413752077036565610540535212787328578538214895356757473570870182465077399397298381931054430588422195894000075748702824202326069035060769,
           # "4befe0eb0c424d83cd2dacb59740cddec599ab3c8833dee354717425993d12ba5441056297153bb3d2667c3e9c76caabc349a07cfab60efa9e5e7b35e971fe7eedac090a1a5a7d8a2cd59de84762f09cacffecb65bf70ed504243721fd0e094c3f216fbb85778ad82829658232a2f472919e992060394e79f2aada9e8a42ce21",
           n=142835122922793749025330256575502295056497497542567510116218571215594275126233968454109622187210826775636498849338698075444746176535521600817256345220859360391018461235792345485152940250710149917233292547942971321737428715910646317103932409489893458880232043953112065831923020102653654729509619797493007421877,
           # "cb676a5de86e2c7d75a17f10fb2e3f81a473e5d2d088833d8c1928ce78caf1000aaa607c83f55b57dc07fac7a9ecad1600df5d033986c02c003884620661a9674042a835b99cf8a024c27a10410eb379ac69e72d6f5a9cf72c185262331c98879cbc225de835d864983d2bd085f1df99341d3cbb0ba3b0a50491c8ee98d691b5",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="OPK_U7_1",
           d=17475986195957943757574904074242422304658173362081677585600748851639490029286438389620507038105559795774320526073163010478330153364523391255038011216263593943354699292782344885194027355230506630618956065857927866862372731619282600435445385756922740190599098191252045828180261382573475677379959211461171990785,
           # "18e2fa361f4e7fc86574d9a93f2113a4d99d272710f303e29e07ebf71444335ce789dbf9816d472b27935ad49202379e44023071706bd0058e2bae45ace0938e75610579240ec87086d27fc0844ba25bba09214ae43037cb902801a58915ce58c6f805fb3ad6cf7996f25e0cf0a94c13e04eb4370ed6b93c39ba2136f8cfd101",
           n=155953663851374694994578770195210461680335336006365319162243501836859648427198435830686161459194454294071574661935850247510664932060289963600412056247313090932844413653073583213727152427594662559135289179264739966243865650543811313418698688479524587435238020502226640392317985331646859875156861501272080679447,
           # "de15dc10818e30c363bd0a87d5f8d89b832329fa25b8388709d94e9b0ee4efdd3e24eed3d931f01ea1b0e2b76265d7dc270ea8012545bb7245c286761210bf46c6dd1fadefc257fabebba29bbbd86e8336460e5d21888a319156e8ba529e4b6a200136ae4aba447fb37a357028142d8b16d79a421d513ecd9b9ec0d908ba8217",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MPK_U7_1",
           d=95953075823669389952001319337265393600483443763257929362373389811990445399471113392232673918575293011410518119816515080434216223438596998627719934436290096017890994233256891140807957589536736612613416733943516284675767341972915592139112823637792088942744737664940855441544686156187220056476173224881489362945,
           # "88a4477997b57337cb144d0656bd2d5f0ef59d6b574b631a79ac8015a4c20d454e1df85682ad25eccc7fb92be373259fffe58741b5a85e50caa68b9fe84f6e295d2176b96c20ff819e8bb889702c474effe1a77710ff3b93e896fa488f1717c75e46a1b0f5898fcacfa35943f1abf80ebb665ba7fde59c4baa61dd2f6c5ec001",
           n=128252503064444053052788188640201543897758269169371735124446582805278549113744867808507765349193770967711109602268211552178520734122518589881401561085640910727196136318699929056930199994610906813248773280352848693744187909316632669494573210514167169175934250602180957156715597174683501443962024957833690922401,
           # "b6a33b825b0cf6abd3c9d39d1c8bdce50a41f9bd5ca2de52c4c447afa9943f5c1365d2e9cb7961ffd877fd38696b4479a8bb7eb8da15bd8d59a1cd7e5ee517d1a20f29bc66974f87796a11f7537529f8f46ac57861484808bfce9ee6cd6527f7fe3bfd57b4a7fd46f8dc047d6c8370de6507620c2b9a3bf864e8ee4c4d2abda1",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_S_AT_META",
           d=16484401020099184131764164918008440448200503456068096412979591201770357485619489291996604568028174221937624376251915698266620594140180930559646607158650864499257579635402502607177766609739090396052223434257707478854232493194493859641563193277211047671917641292658832159350481799569582507522554410999569998571623307675243872469253039129270120370982152717256148538062366710948158354379120113340601535150732604624807871758864314319443195558645714858124235501313151501222396533170751943983307544331232474918348372838567192691312736576160355611492791720603118018330419579685211392769734059175769137347538497909490387519393,
           # "8294e45929b8f95a380c59fe715da5225fd518920a85fdc9a8b2ade6675b7680293c21539fa4466907cb3601b072d8adebb0481ecf069baaee00d0f5cb4396f4ffea11dfd41f3c62fdeb312ee9b4be2026bc40aacd9ff928130fa7af0305228dd5e47c551c2a701653dd6841b9566099de99e2731194ae617ca8d9df99a47c49d9f514620ea1e3742da8dc7dec6756403631a274dc226c6121863e4a571a120b63c38d134853df5b986fac1565e1f3bd8a02d239462967e9c71cedd9ae0c0eec330018ca553cc7cc2fbc73d6ba37be2fe360644ff69ab7c734264675c057417857df4ca206dfdac9a5621f9d8e45dd2e58dc8b4198667de3efd1d5bd7ce007a1",
           n=26501611422893173811932492977591540823101101312408557208817452927521744597940499686730810096280202477201670274657716156440562096851834604344107928206954537144311149185952994317836607145896498645555613080116457133366118849662395292023282889680082924844216029863779759186246842283782453791132237174872533276006421068974969870248543518145340591142044388286901471440798262587076024800472650214207910188909334014080584464045159905922161358898556204056680572757558238828626740600155604567554223165686833699774391390041144728885023575280626925161738865774259335842862245819359068051826548599813333060329490274729150765868507,
           # "d1eee63f19d148c904076c507aa8d4f6c7e931a65476fe5231c06036fea2ecbcb8c811882c4f70e6e3523be73f5c7a83570f3a40bd894399a5ee9f903e8e745ec4e4e034495175b167192535843f06241d6477e3ce1ad5270e590db9cb905404c01aa407433fa2c2ca1f8366c1623fa45bd5ee68e3145a57f9af3e6e68fce41b8c682c0e07f3c48f4b377951b23b467fea0d4ee0e67c0235d0e83ae27e40ad1c060063ceb966835a0ac1eb68066f8b55775ceb7b444ffaeec19548a42247ebe687f881a0c8e5277beec22241e2ddae1c21cec8046eb005302812b7ef42ac153cab317bbeaad73f7ccaced38c433530b7e0ad464150026025a9a3ff5d45e025db",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_B7",
           d=83858470050086040537131800390036022123472028425365591640336152908246256772816161109445098663464812200901376545796558297070688762629307935830876859206117455476786658514740978348043485200865540870266992716492920572081989781654882672475997010735047739295697099675004596194135171966514969313862472321096471558269,
           # "776b1deb8c3e943b3dae67cf2b597ba55c439dc1fa10e4e9ea530df96bd0815cc3ec3ef0267f89a699c5cb64bdb91e5e9ae4c7af03cbcfbfb4755cda55e3a31d510f96a102b5aed90731788a426e371f8ae24f660403377cc0836a06b2a8e159bd177f4cf68e36d447e4b52ca63611cd8416c1efcac52143106c272f7474387d",
           n=160027736414188883868095588683626671575167758405299035591902584315515095656176011141417564892627126261842981529914947474452486079737849174229047454264080614643136224572713892138386900209016912487475988479521571446554828956944809835093619656142938927355236933209697549745402513950924154107548745138826948271249,
           # "e3e3166f47177de4915e915a9d555d980afad96be22cbb8a02516ac8fe69657bd10bc6d072046dcd33e4476e24f128cb7ac613df140cefe71abf080a74d27c114635d3954c55299f6f81c2a0aa14c4c678307f4b3bbf0c64f0006051ea7573b5b0cc290201c76c4d272c981b1bb19bd0a0a0ac046e6e63b0f4cf88d2c98a5c91",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_BACHATA",
           d=7016780565838885449303465084270752926765459531029913255174051788824835669780310688320398025021611395120275753055542580008673960095417619266994849925158027381333513422127110335841602159123517864642255665230516161008116423763398012223661807339491264886885037910176424315360383746853770897693667916856600731281,
           # "09fe029a23ff7e37c749386fcc9a640450546b95e5127489d364d380393c99f5c10da6d7cf0ed955f4a5f3d8d90d97cc7c49069d394206f9b59c11568ffe66163eae377447abb103cd5d4256885cf7984b28cff8a096dc479b9196d66cd534cbdfece7a61de04110bb14a3ba5e0f20ae0bb4d82e18fbff0335904dc09b829e91",
           n=104489604168003416426039807141070741777193347258602005227071536488119349077571511379335134143567676892069418774824152253130757855663118498955019649976160356842918384367763573502009310839158896159489568263873961491372885271855417110360857600083728180912692975612134722895320488917022350390217668706980578834767,
           # "94cc529bb1af0ab8043f09e3ca612d787cc19485d3769546e750f6edb844979ba8f1f9afb8b93b521330b74713831a78a584f7b24780f92dd00e5d56ce8defa3cd39d01752e514a4c2ba7499f334729622049491b1aecee6c9e1c867e996c294b10f5d62ea4504e333424b280162087296c300c01fdf75f47d874df40dbdb94f",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MPK_VLE5",
           d=78665037422872749408592072938682571003507689065948812399191452477835343856768119273783473273124581505326542841000170179608664520148096312908217290632790314556747000375152804693808394158258535304233388995958778249162065365944833234542646900317578588499506970280183991166199514025908738229108469788647428540993,
           # "7005d1bf5be81db7b17c9b16b1d407b308b42e3490e75a93e9d00fd6c812d1d8db2f1041a342964808a037f315a448ecaf0502a5215c58f0de709c5bd87e3a65e0291a1a23547c76cf437ef1d9b434b70dbb417049a31de9ee7becf218a5bb63b05fb84ff49d1e6aaa4b9b4376f47417435ecd85ccda63be9070e7892ecd4a41",
           n=116689766134374762408974416246405460624632385376408123813535461420970438205133705277063613044086048258631213376126121026255925548469835179354605748538991417119927739282862577161935514963672064081693519516966022663560869209863612058080397055878116371553631895303980271393418435431259492653753740544933933167237,
           # "a62bf756a70657b6b560588e85e662e181b6a61ae466ac3d0d2e971f160e88216792cedfb1979b3d6b665068eee8a8699888cd74ec9482c61ae7eae3571e50beccdcb336477c26040d09b46dbd93efc0fece4adde2e00c1cbedafd6ad7c43bd621675a6a46425c5cf6182fc5602be443a372fb4ead4531e64285ce29be913285",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_C7",
           d=34986543445498669534976730416044444533345146299523633632425109168131661519963624503488139058951526980070248403685540999794745465634207833672094888151157963394080712954009921225711749312993398864455008061377005747905589111446279408362191084142339377790599312065612528177004329583982878188768278369869883297893,
           # "31d28ed040a8ace0d56fc94b4a7d29dbb135d62c7905621818d657499fd6ff6fe7417592cececdd3f3d37ec0a361228da34d3e7a2724b7832ced00008fb4ae500357fc3d285c64fbf7efd4bd1ee48ed40190296171acc3c2d0c69e89da5a8fde7e0ba7048aec6bef1bb19646f883fe9d77d8d263545e7c00e8604be38210d065",
           n=136118319844918154070274264249112778948165084774822224836345763107856616267964147170383031730870063739558555632670780676969322266623275677967829188765950717786217174139396323254875457339636928584743045012692909706110003475405715025459834377626982001573627226745676663640380489128907989730621396080369036056881,
           # "c1d6c392828f4620e455c138840ab448cfcd4aca821663335dba9c51dec9b8198301ca6b069adefcd1887f1cec31c15674ab264daeeb82398b419f08b4236904203c48a7db8724f1773d04a6b8c88cb38907a00bc53e86cdb2bbf479a68b8241382bdc5ac6105270efc2da4cb91a36459ccf6a2a87dd56ec4c331dd419ba5931",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_MARTELL",
           d=19234738489527764694131480939773296545866610930907651816074873855680507953058481808753378489980415332596054469347471734802475822536323303458680771247368859976522160234074749800634738940500344521813468947677275804825162772181765352665414296219862853131089114238316159950164709210430270596217749519696186053620124166488667109897188188201290519932193696688755146043178651405304794619107255711486564532982764712488229408487150592833180085941641352413237429437352915331407785306259223933202204166195441205850864525204541304833118376316505108690428833783630915313859658784169395650376065743914525543678339224934686492081049,
           # "985e549fd42c0b4955d3db8c3ee601f65e10a3db08f957fab4016dbad0f60c7e09e8b7a782404cb0fc7c805dfd67fed814765ed58b7a146ed2c1d31b80e3f845a45b6ccda5a0344247be404c23debf027c7b5082373372b49bf78d9058caa66c57d3be829088c3610034faf1ea9f24a21110bbb3865182747ca1779e83c6983c189b3f19f3df49e5f9cdfa57f4f69dfae53e19ee0b1ec30986d59ad11f52bdc022a9499dfa89f8546d266f6026aa307501ca5a619f5413a45ef38f139c3ea8b52f02fbc8983aa878052d9108668ecfc8605057a298355d2f680c34630e224c57dd4c4f2dc0d51766ef7070daddfa3c885a3f94d76c943c6c1054d338e2323b99",
           n=21208436629566625975963102162756528411562638052742265504729281063961328606608461292443725665363008170738376489074754451424170698569316268023344690353625845019706800620151365753965457887943253119655595989500414287506758178277808067526384749341518082803125732357019519072881734707156510045161725668298545465798300735873054546069727119837740909895235909236577123313174168058897103435697938736219186311296498854826788748530073271750780368778514894940503046705835812626853004828735650173130659989580117760371944018916731343856589410465026421587385856213668661458111359003609554782884756019240305917883400455422061102047561,
           # "a800d061e4a42e4c3453a17cc8daa974e23bfaa403b4a60fad6d3516d8ec035c1ebabdcd60009d9b8c639954e616c6cb6cf821e31e58772ffc366e6ffb7314657567b12279a34dd69e46b8a4a628dc2dfabc68fa1d89388d2058a97d2e31520b4fb04bc2f963e110e8541eefd22d90a03eca806b3c6a20c6bd1a7468e61ea1ab283ed1bc462dfae189eb5fb451f802fb868cda9a7409aa52e42b18882e79f4f1c2377829fafd9760468bd1db823bd9080378cf46ef405d91636cafc03acad9fada6b0446dbaf51e9d533887e4a3a8f62114063e0b8920684c28bfbe256aab26e98751166358c201347ba6c3b36d49aab6302fc248eea3c254e15a08429fd2149",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="OPK_U7",
           d=68767263029691866096357734251574865918709617886522092448999129306453757138532239500490410985425564209360710312846442197637235711026712476806656572189449963797074465756381739114141801320743503298494001260730897592951632525520793080422447090178937842108675716984682533832882115904240085795366163760699369735137,
           # "61ed86791440c26491b763730f483c18c32fcd77bdb6f9e9e3e11cdfb9716d22c392c68556219e2b6c1ada57649ce2de559c239a9ff8f33252480421e4a2649df8e3ee0095c9bec361f25a5ec67d0b4d96c73404ff8a115fecf1173a6568845480fd4423b5dba2e5111335655f3bc2f3fec65510648571992e010ba0aaf243e1",
           n=112062066220178427738443862785525076154722436467886674113684659157994377541536847108029939198643240460225100623443252413321518680994545977881936787218837385414340845805059873865237189061988103525795577103109047402830778391550712797149184952796547615741171254704917667174902603252117532292946258789115297304859,
           # "9f94e8f1171fed4b427629c928e807b2220f109ac70a3d5b1b8cbd295bc3fa3226d3903298cd81319b9b08a6f8e77efad0b04139b686ee0d1586175913ad6f65d6cf21bbc7f769885381ba6d840414b26fe7b9b3e393acacb3453e3a0cb79ca21cb38a42685a03462244fdb2a5f1d8b9e20745fb3206e799655c47146310911b",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_S_2020",
           d=23279393021939759060034519445868228839472758837741261854067818272418805756768641231565526622166718926270588271229236382530779348060917909664671058255374759588992027552736902527678083898873132767395972223531632223932201195221430767626223020628322961752642202421940512707498599759864076773621357938010735680766128896407551473272028357720740444671152470225907208127973446696793348614544662952759791643998022904432815631344759183880489339118898187376776319933164432017110794361731557824378051358398386751542611612891508699643284692738433731301345317269185986037017147882214752210117811770160857356247928519614965177516185,
           # "b86887499d157d3b1feb1041b9d2e94065732b41d22feebce317676321d66d1babcc7a53544e35a714c207811e62d134291d616417295e5b0c4aa3d65e40b41a352822263c22cbb4041a1883c76b97a8c925cb428a7b2300622ddaec62209d8dc0c60159f6c7ccfc26768bc469deec22bcd62f49f4c2ca1b2cf0be49d6e5ec563279cdee79c92800c6c965200d316c79285551a54359b37ec4173eadf4c0506d857ddca4831ade7ea8f13097b4e2b630a2d3eb9c57abcc65f84d693c55e361763d8d37bb40cd6e2520684ae05edc62a36cda6747509600f4605b7ed924ee1ad49e66eca1176a20794600173dbb42fced2f1fa0cccc0af3b56d58453bee420099",
           n=28883071077938472407661245331942488233170387261918312031540695280744978851261717473649424827476056534606708256626982408881142064553942998101047804795019037885422182229444518968544140405142715197774219673878167828931972847132386282570137028187467379962570747418248369643545032987433493610859545931305319456105179075834369476614509978113081982688821913518862414849512823156308160849773044813437195751470515913607884799448406025879584269504809064283506073936809767600255470025235784146445306669175491371098563809746880739874020542084604454543601268600081940809771946983468548801720815762263929077422010838327460179081361,
           # "e4cc4967bec817bb348468024c3084b15fd4f7810c8a9d078a4f51cf9974d2e3bae8d5a19a85c0a73befd0675100e642a3f425e3192dfb1de56928c37f45fc142adaa65798ada863b84d4b5f22c3f79b95cc201ac7c292c99475453a62b7b7e06e84833dfca7a0df931084932a80129e543c6d24a13c5f2cba6ef5ffed9efd4dbc20496f5194f0d1aad9d789f32577f8846df9a14778504ccb5dd7507114c148c1937fb99da15f9596d4fa052cdaf1f66d7e5e0c0793628752bf9af3c4ac67e21c21d170ad448160761bddf586a4900fbb7dcc44467f1550d15db774d7cacfe3105b465321a5f95fec22c2011d616a5c0e22f0535dd1f969202be56ec015f891",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_U82",
           d=44496855619679803165653104336186564879908850774839972694462705127997588168286803975590043653124194711059698475265826175607214971536115407519545499937477700706820080193809996580951807439472779704503601460596783840360174836001720463688220528631081175326226958903134350953955505205953328817388154005211146155761,
           # "3f5d99a61561d70c6c335a30d9a11fa8a3ad70fbecf46c9e233d57aa827cccbb137c060a47e693e234ba1b532851053e17446d5582b9fee205c0d12c7613378c8b8c8c0184cdba90d56a308014aac0458c5572699d599a15ba36146b6f2e230034708cf67d31ab837b7bd8e5967fd9a7bf413b7d9314302b18e48962d01cf6f1",
           n=166515755538568792329549905720359670104184682991531279077085725231175580185291627485196407862433640654297919201364497805674073407729252296157623104516786188728481482294489502433378669425783243097269834897334267830444145429440941715192440842438745231766149401710786286171591221262328942704223828091850523658351,
           # "ed2055a7b95db86f7e3101196ae6218015d70d03df6fd5787de150e82927443097a90485757743447e2f4641afcf510acf585f73e79c45b2908d5de8835221a76d93e48ca465ffbe0dd76cdaa98550ab2e7b84a6470d48595742fb54a204442ce67bab989c69adf86457e313eb24c87d80aa7d635449fab0d97b6b08c5f7c86f",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_JADE",
           d=74428655312884956294405835932182701162556415211038095846135618750693107948093115488281652541946729361408467189827346534283521713719634379606956911321529146479374790356451147526039413607475004732846349717355462123491869441042086723599244284707617214198373732627379895046044614727615358364190208672622326894337,
           # "69fd6b9e25ba604e204ec90e8e0769b28417e6b52dda7ac53deb712c549f398a48ea8ad20bf065a093ac85f336f92f1221d3413f3793bc8c7c6057a091828c04f6fb695f43747d0d22de100bccce70ac7a8f9d092afaa7d44fcda99b12454f8c887e383c69e7e21ad15203eaae51d803cf35da09c8d536139c658bebfddccf01",
           n=163116331702800340444973089536097434660595899668465880399551566548427441666471994039443374218885861495406190282895760092875172637508148687008464924333836825736009088064337449510673610977284961323869792047502602680306051577836166419316762176200681293749927743202824150362562479039913808471325774127679021891431,
           # "e8490dcbd3488278442f78ec5634ccdb8befee081ed0d19071480a10c299416ab8d0e9eb19e8975cac260606463c51bb62875ab24690d07905b9c48fe60086da12899bce3dbed91e0157cff76f27a1c09b37e837e7acb71da3c0e30564223ae20216fbcb3de5e93c2d7f98827d61441b988e57497c1ddacb87cec1e73139bf67",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_B82",
           d=95953075823669389952001319337265393600483443763257929362373389811990445399471113392232673918575293011410518119816515080434216223438596998627719934436290096017890994233256891140807957589536736612613416733943516284675767341972915592139112823637792088942744737664940855441544686156187220056476173224881489362945,
           # "88a4477997b57337cb144d0656bd2d5f0ef59d6b574b631a79ac8015a4c20d454e1df85682ad25eccc7fb92be373259fffe58741b5a85e50caa68b9fe84f6e295d2176b96c20ff819e8bb889702c474effe1a77710ff3b93e896fa488f1717c75e46a1b0f5898fcacfa35943f1abf80ebb665ba7fde59c4baa61dd2f6c5ec001",
           n=128252503064444053052788188640201543897758269169371735124446582805278549113744867808507765349193770967711109602268211552178520734122518589881401561085640910727196136318699929056930199994610906813248773280352848693744187909316632669494573210514167169175934250602180957156715597174683501443962024957833690922401,
           # "b6a33b825b0cf6abd3c9d39d1c8bdce50a41f9bd5ca2de52c4c447afa9943f5c1365d2e9cb7961ffd877fd38696b4479a8bb7eb8da15bd8d59a1cd7e5ee517d1a20f29bc66974f87796a11f7537529f8f46ac57861484808bfce9ee6cd6527f7fe3bfd57b4a7fd46f8dc047d6c8370de6507620c2b9a3bf864e8ee4c4d2abda1",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_S_2021",
           d=26173359792425535964862516843126609743904960996310328623866733239562044385499676994537124154816482460754321566951848981110709125938012860861965840457315447314510832126881371285123028245964697101797657980914772760331434963942093574475179009585330900574252203508086772707038013301612811958001272598085969696497201108811084132621189423367956658856799087774491613504231662198290806692275317486525142373666489038032501507650343155154091464830606078629568653135046978280457039550260610350847533953817098434115655727520640249944033237632537711425582243776033515499673863118416797072123552005599108497250161585967021127652753,
           # "cf553c03ac3cf21fdb4097d4a97f35fc6c305a2e30dfbebb7667ba2adfdec99d3277bccd314281c592ade680b42849fde6122659a68cd7e525b764520d612c7c6c141bc4b2594bc88732d4ca0a97e464d7c1ecf4fc2788f1920cb030c1b2b3ea84e8d6191d5e53d56c5fc495051a1d0fdbea947d58a9d773a68152d157d4bf57f2b4fba8182f96ea4c9b798018361054f95b251089c786be542c7881c49b077ad52af25a359bb26257170706217f66533cf4b8379a1fb7a30c955c8ed4c1c6dca905ce6e7e5e92ec7e1bda1db44cd187a9e5137fe44a37cfdbee173a49654994926cb2fdd7857dfc8978d9de73e899e18f5dfe33a64e6414fc5d93738f8c5591",
           n=28217198235173422446606263634610768593293295423839282892290740192814240876665443842588904519398055782702022907276251466936215561549589551937994000428542235082268406071712854579916234613518479255807091809429370988548137082281180894725774087040530206134804518198872887364717063295736122014994726143457150781367947461719582937964872261394320795834560538720821359309137638784531241730656892933630915941724742760351930345999513006644941009360698188346992473796881496936293424874651632520361149771923515481840775048232975111864537323721453884188438160945178265966338094793564615659853208216155428229934145480908583424642679,
           # "df85f4c4ae8c98e78142d403d002276a5bf9edd17870caa848fc45720e8b4be94f6f9a47181417840a5b7d4fc36575129afd6a848a0de3f62fac5b5f687a2219cab8cdf2e7527d6af3c6be84eb99bf519b0b210960fc8f5223c9bc38e8f20d0267642153cf370312b955143e10490c6a207868ac7ac314bbe10f6063a1ba606e28d248a1ee3e7000d12e9c4ebd47ae483b625156b82026fcfdc36118198cac1463aeb56bdfe260efa38ac1d4123c13fe59e0fb0f2f895609c117f7a39fb9f27c356d4748cf7af41e15ea68c6c7c64c4d0a1acb4632965e0260d9b08de9fd81b82050c9929b79ee865f89272483b6fed8a409d6a1af2429d24fd358a4b4da4e77",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_MINIQ2",
           d=115487653959634485504223941734854859273041630205851831295616889623544766183139012458530499949017972453666848257553029156152361023333127576106852328586040526528510523683294722915628768962561342118413197377276548178981667902433348763261681733232630058138433861991703360015015019759723410155180647615881086667265,
           # "a475ba952a9f2f9e58d6ec91b41a03158354ea1e451656d83d15691c07eca3410e7a2401283462f66a0ebf1f91682a80ae61168b2260f4368f93e197a9db65f4139523ef5449a6fa77568a9ffe90e0a34a37f99b7c1ec6ed1683a574d9045993679ef73299991cd43b96fdba6673ae4318f2f635a816f8559d325f9ebe428a01",
           n=130693368862283555679139461069850513057351314387362057390883480025871207026944009177626629630452943443430795590814561262031311030290495613263482198137576656163529221661499406839931833918321737337725007840160354043559069499679018886971759565992627816010982343045357547791474675485776315123941085425086523170401,
           # "ba1d10a245e60471e8d3138611615170f213cae5b895c8af35eb720e2671915f07dd6ccb5384d7580200d18f430c89405dcb0be6a5e91cff0fda970e292d5f0704720473bc61e19590539b1bb08ce2b306755db1f70cf1193933802ca44281fe01699f75e56fb7660fce0342ccc284d497a17ad7d3d15eaa20ad4c67bd92de61",
           e="010001"),
    SlaKey(vendor="Alcatel",
           da_codes=[],
           name="MTK_HIGHWAY",
           d=59006940828126735553173077037943242201832157052520402593455723122542835111429421359691292364683530979213212395468339549462812822624598347903664523341867154475402087181308215256146921907450789029033508744958364412532225869775524539619194608415195645850958496336906376404377251657182833411028517780640086344449,
           # "5407571c851f5b877a2255c6887c5d832369698b481c81db8ac07062dfabc7229d4b00f95956665743f7deeedbf54a17c9a404c97433f46d983bd0c5f49fa4b013b9d86e5f1377f563d8299675c0ea2b81f51c33ad74a265184df9389eefb8e72d2f0585e4a41826b8846b0ee6da5ef8cce471536109fe4c658735247ebbc301",
           n=133977892220514892875149111344085582877684107426241325691772024885050228128386571079894963542969254773582881816858667165089535890948874096679686178847559186931513667983344829088482640787658857227878412195762908642401697642789030340364680779758281059200765007047778473419889696110478973303650416451711486068393,
           # "beca753fd31ef104bbb01b0a7c560c7bc040d30ea18f216b64b7de416b695af2b3350ecc02fa5224b412793f876a7bdbd8cbe7fecd754aa8214a27bfe7ececd8caa16959df83bdaeed524880a820f8dfe601dc70f164ff1921baaa06efd8c584c22269a109d16287356fd30e7eb02a1365ca93fcb8088278f119a2c7298306a9",
           e="010001"),
    SlaKey(vendor="Motorola",
           da_codes=[],
           name="G13",
           d=22050419731562155394106709972952859421194100106456997203564621188227249424701291592884703493390930536680058773297929760551724618568230133040793183340115990462390794010584599304604825181361962475363874162191142504833442946164654238351829079842919675978362478753291817673354538846468019979634676288548417418363463924320170824947490552875292996775107232100185748900473687003803356957304359156634677724577486444094774736019891872575736258571932346764362017119622746749282450857160410786146449479956173648609409863284469607303105156090816916242350888950390600656505661831657766048625847690202923995919597384708979191222433,
           # "AEAC47CD11A5DD6C5EEEC43D8F2C536A2917CEF95AD02F5A7C978E88C35702B590F7A72A2AF28AEB9B5F5B2D8056D03F916595D189C9B6927AC0874980537178AACE8E1831DD654E0B72FF2F44670196A57A43C340355CAF828B331A5715AED4E06D5D18896BCF25B201A0DC9760B0B2EF1CFB4EAB6940D7F8E2EBD86DC1E678AA69F6B0BBF55C688BF72C2123CF42E367F789E2592CE281C7C4752E14F6FD00D54610977DEF753E3890F12F704688537E860D81142805750B805E7CAE3AACDE1CD7A272D227E9F8CCAADCB4D06489664627BAC46CAF5DA0F0740CEEDEBC7ED1C1D1EB1E37C6A8A9E6A0454F742B3248448B20C93D5FF6E5C789907A862C90A1",
           n=27568072452258469631124979998042952077199499020924689540824391049462976832251975336186280290850084215612380996291995854965249453016140752176544503177435743360047796014339620080615918121154500853642163696442596458208114276283726532217928699077936413670239255437895571439500885394524506398422697060713422974958102681053769215158188151660543990224375640194825863469761475571057891851563655933517408680203498008205536450438567523758655066623624002686806613414719125769676194560764819157656002478522800571189379236284227865790549388127212000079839286099840462834583520893472589236519103688311338314257196815867652714275027,
           # "DA61964924F441559A1F8B5264CEB01DACE8E417413BBA4657F4556811D07B85074FD6987F315A7492E003D03C57FC83D3B889F2D4F136D0989E515A08628A7B16A300217162DC35C340B1127046AA86649B763AF97F7C9871964483DE6695CDA2E8CCE82E1F6A0F701AF8BE767BB16927489524F8FC9A2C280F5692E850E4C4E2606436CF2E253147AFAB32E6B92A19FA180C43CF480619B71B3D6A7863C7CC376C0A36BCF8BA3DA89CBF3E6DAA4691DCD769C0AE4535E502A9966AFF3F123C7A0EDA2DF04593B0E1FC60DC688F2BA7617DFE67D31854443ED95D2645323728C594CA49DAA9351A572E3182D0A1B3146C92CEF87380CBD2DEFFEBC4E8F420D3",
           e="010001"),
]
