"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.rule = exports.name = void 0;
const icu_messageformat_parser_1 = require("@formatjs/icu-messageformat-parser");
const context_compat_1 = require("../context-compat");
const util_1 = require("../util");
var LDML;
(function (LDML) {
    LDML["zero"] = "zero";
    LDML["one"] = "one";
    LDML["two"] = "two";
    LDML["few"] = "few";
    LDML["many"] = "many";
    LDML["other"] = "other";
})(LDML || (LDML = {}));
function verifyAst(plConfig, ast) {
    const errors = [];
    for (const el of ast) {
        if ((0, icu_messageformat_parser_1.isPluralElement)(el)) {
            const rules = Object.keys(plConfig);
            for (const rule of rules) {
                if (plConfig[rule] && !el.options[rule]) {
                    errors.push({ messageId: 'missingPlural', data: { rule } });
                }
                if (!plConfig[rule] && el.options[rule]) {
                    errors.push({ messageId: 'forbidden', data: { rule } });
                }
            }
            const { options } = el;
            for (const selector of Object.keys(options)) {
                errors.push(...verifyAst(plConfig, options[selector].value));
            }
        }
    }
    return errors;
}
function checkNode(context, node) {
    const settings = (0, util_1.getSettings)(context);
    const msgs = (0, util_1.extractMessages)(node, settings);
    if (!msgs.length) {
        return;
    }
    const plConfig = context.options[0];
    if (!plConfig) {
        return;
    }
    for (const [{ message: { defaultMessage }, messageNode, },] of msgs) {
        if (!defaultMessage || !messageNode) {
            continue;
        }
        const errors = verifyAst(plConfig, (0, icu_messageformat_parser_1.parse)(defaultMessage, {
            ignoreTag: settings.ignoreTag,
        }));
        for (const error of errors) {
            context.report({
                node: messageNode,
                ...error,
            });
        }
    }
}
exports.name = 'enforce-plural-rules';
exports.rule = {
    meta: {
        type: 'problem',
        docs: {
            description: 'Enforce plural rules to always specify certain categories like `one`/`other`',
            url: 'https://formatjs.github.io/docs/tooling/linter#enforce-plural-rules',
        },
        fixable: 'code',
        schema: [
            {
                type: 'object',
                properties: Object.keys(LDML).reduce((schema, k) => {
                    schema[k] = {
                        type: 'boolean',
                    };
                    return schema;
                }, {}),
                additionalProperties: false,
            },
        ],
        messages: {
            missingPlural: `Missing plural rule "{{rule}}"`,
            forbidden: `Plural rule "{{rule}}" is forbidden`,
        },
    },
    defaultOptions: [],
    create(context) {
        const callExpressionVisitor = (node) => checkNode(context, node);
        const parserServices = (0, context_compat_1.getParserServices)(context);
        //@ts-expect-error defineTemplateBodyVisitor exists in Vue parser
        if (parserServices?.defineTemplateBodyVisitor) {
            //@ts-expect-error
            return parserServices.defineTemplateBodyVisitor({
                CallExpression: callExpressionVisitor,
            }, {
                CallExpression: callExpressionVisitor,
            });
        }
        return {
            JSXOpeningElement: (node) => checkNode(context, node),
            CallExpression: callExpressionVisitor,
        };
    },
};
