import { extractEmojis, filterEmojis, getAllEmojis, hasEmoji, isValidEmojiVersion, } from 'unicode-emoji-utils';
import { getParserServices } from '../context-compat';
import { extractMessages, getSettings } from '../util';
export const name = 'no-emoji';
function checkNode(context, node) {
    const msgs = extractMessages(node, getSettings(context));
    let allowedEmojis = [];
    let versionAbove;
    const [emojiConfig] = context.options;
    if (emojiConfig?.versionAbove &&
        isValidEmojiVersion(emojiConfig.versionAbove) &&
        !versionAbove &&
        allowedEmojis.length === 0) {
        versionAbove = emojiConfig.versionAbove;
        allowedEmojis = getAllEmojis(filterEmojis(versionAbove));
    }
    for (const [{ message: { defaultMessage }, messageNode, },] of msgs) {
        if (!defaultMessage || !messageNode) {
            continue;
        }
        if (hasEmoji(defaultMessage)) {
            if (versionAbove) {
                for (const emoji of extractEmojis(defaultMessage)) {
                    if (!allowedEmojis.includes(emoji)) {
                        context.report({
                            node: messageNode,
                            messageId: 'notAllowedAboveVersion',
                            data: {
                                versionAbove,
                                emoji,
                            },
                        });
                    }
                }
            }
            else {
                context.report({
                    node: messageNode,
                    messageId: 'notAllowed',
                });
            }
        }
    }
}
const versionAboveEnums = [
    '0.6',
    '0.7',
    '1.0',
    '2.0',
    '3.0',
    '4.0',
    '5.0',
    '11.0',
    '12.0',
    '12.1',
    '13.0',
    '13.1',
    '14.0',
    '15.0',
];
export const rule = {
    meta: {
        type: 'problem',
        docs: {
            description: 'Disallow emojis in message',
            url: 'https://formatjs.github.io/docs/tooling/linter#no-emoji',
        },
        fixable: 'code',
        schema: [
            {
                type: 'object',
                properties: { versionAbove: { type: 'string', enum: versionAboveEnums } },
                additionalProperties: false,
            },
        ],
        messages: {
            notAllowed: 'Emojis are not allowed',
            notAllowedAboveVersion: 'Emojis above version {{versionAbove}} are not allowed - Emoji: {{emoji}}',
        },
    },
    defaultOptions: [],
    create(context) {
        const callExpressionVisitor = (node) => checkNode(context, node);
        const parserServices = getParserServices(context);
        //@ts-expect-error defineTemplateBodyVisitor exists in Vue parser
        if (parserServices?.defineTemplateBodyVisitor) {
            //@ts-expect-error
            return parserServices.defineTemplateBodyVisitor({
                CallExpression: callExpressionVisitor,
            }, {
                CallExpression: callExpressionVisitor,
            });
        }
        return {
            JSXOpeningElement: (node) => checkNode(context, node),
            CallExpression: callExpressionVisitor,
        };
    },
};
