/-
Copyright (c) 2024 Amazon.com, Inc. or its affiliates. All Rights Reserved.
Released under Apache 2.0 license as described in the file LICENSE.
Authors: Leonardo de Moura
-/
module
prelude
public import Lean.Meta.Tactic.Revert
public section
namespace Lean.Meta.Grind

/-- Mark used by `markAccessible` -/
private def grindMark := "__grind_mark"

/--
Return `some originalName` if `name` is a name generated by `markAccessible`.
`originalName` is the original name before `markAccessible` was invoked.
-/
def getOriginalName? (name : Name) : Option Name := do
  match name with
  | .str p s => if s == grindMark then some p else none
  | _ => none

/--
Helper tactic for marking accessible names in the local context.
This is a trick used during `grind` preprocessing when `clean := false`.
Recall that during preprocessing, `grind` reverts all hypotheses and reintroduce them while normalizing and
performing eager case splitting.
When `clean := false`, we create a fresh user name unless the name was "marked" by this function.
-/
def _root_.Lean.MVarId.markAccessible (mvarId : MVarId) : MetaM MVarId := mvarId.withContext do
  let mvarDecl ← mvarId.getDecl
  let mut lctx := mvarDecl.lctx
  let n := lctx.numIndices
  for i in *...n do
    let j := n - i - 1
    match lctx.getAt? j with
    | none => pure ()
    | some localDecl =>
      if localDecl.isImplementationDetail then
        continue
      if localDecl.userName.hasMacroScopes then
        continue
      let markedName := Name.str localDecl.userName grindMark
      lctx := lctx.setUserName localDecl.fvarId markedName
  let mvarNew ← Meta.mkFreshExprMVarAt lctx mvarDecl.localInstances mvarDecl.type MetavarKind.syntheticOpaque mvarDecl.userName
  mvarId.assign mvarNew
  return mvarNew.mvarId!

/--
Reverts all free variables in the goal `mvarId`.
**Remark**: Auxiliary local declarations are cleared.
The `grind` tactic also clears them, but this tactic can be used independently by users.
-/
def _root_.Lean.MVarId.revertAll (mvarId : MVarId) : MetaM MVarId := mvarId.withContext do
  mvarId.checkNotAssigned `revertAll
  let mut toRevert := #[]
  for fvarId in (← getLCtx).getFVarIds do
    unless (← fvarId.getDecl).isAuxDecl do
      toRevert := toRevert.push fvarId
  mvarId.setKind .natural
  let (_, mvarId) ← mvarId.revert toRevert
    (preserveOrder := true)
    (clearAuxDeclsInsteadOfRevert := true)
  return mvarId

end Lean.Meta.Grind
