# SPDX-License-Identifier: LGPL-2.1-or-later

# ***************************************************************************
# *   Copyright (c) 2009, 2010 Yorik van Havre <yorik@uncreated.net>        *
# *   Copyright (c) 2009, 2010 Ken Cline <cline@frii.com>                   *
# *                                                                         *
# *   This file is part of the FreeCAD CAx development system.              *
# *                                                                         *
# *   This program is free software; you can redistribute it and/or modify  *
# *   it under the terms of the GNU Lesser General Public License (LGPL)    *
# *   as published by the Free Software Foundation; either version 2 of     *
# *   the License, or (at your option) any later version.                   *
# *   for detail see the LICENCE text file.                                 *
# *                                                                         *
# *   FreeCAD is distributed in the hope that it will be useful,            *
# *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
# *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
# *   GNU Library General Public License for more details.                  *
# *                                                                         *
# *   You should have received a copy of the GNU Library General Public     *
# *   License along with FreeCAD; if not, write to the Free Software        *
# *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  *
# *   USA                                                                   *
# *                                                                         *
# ***************************************************************************
"""Provides the object code for the Text object."""
## @package text
# \ingroup draftobjects
# \brief Provides the object code for the Text object.

## \addtogroup draftobjects
# @{
from PySide.QtCore import QT_TRANSLATE_NOOP

import FreeCAD as App
from draftobjects.draft_annotation import DraftAnnotation
from draftutils import gui_utils
from draftutils.messages import _log


class Text(DraftAnnotation):
    """The Draft Text object."""

    def __init__(self, obj):
        obj.Proxy = self
        self.Type = "Text"
        self.set_properties(obj)

    def set_properties(self, obj):
        """Add properties to the object and set them."""
        properties = obj.PropertiesList

        if "Placement" not in properties:
            _tip = QT_TRANSLATE_NOOP(
                "App::Property", "The placement of the base point " "of the first line"
            )
            obj.addProperty("App::PropertyPlacement", "Placement", "Base", _tip, locked=True)
            obj.Placement = App.Placement()

        if "Text" not in properties:
            _tip = QT_TRANSLATE_NOOP(
                "App::Property",
                "The text displayed by this object.\n"
                "It is a list of strings; each element "
                "in the list will be displayed "
                "in its own line.",
            )
            obj.addProperty("App::PropertyStringList", "Text", "Base", _tip, locked=True)
            obj.Text = []

    def onDocumentRestored(self, obj):
        """Execute code when the document is restored."""
        super().onDocumentRestored(obj)
        gui_utils.restore_view_object(obj, vp_module="view_text", vp_class="ViewProviderText")

        vobj = getattr(obj, "ViewObject", None)
        if vobj is None:
            return

        # See loads:
        if self.stored_type is not None:
            self.update_properties_0v21(obj, vobj)

        if hasattr(vobj, "LineWidth") or hasattr(vobj, "LineColor"):
            self.update_properties_1v1(obj, vobj)

    def update_properties_0v21(self, obj, vobj):
        """Update view properties."""
        # The DisplayMode is updated automatically but the new values are
        # switched: "2D text" becomes "World" and "3D text" becomes "Screen".
        # It should be the other way around:
        vobj.DisplayMode = "World" if vobj.DisplayMode == "Screen" else "Screen"
        _log("v0.21, " + obj.Name + ", renamed 'DisplayMode' options to 'World/Screen'")

    def update_properties_1v1(self, obj, vobj):
        if hasattr(vobj, "LineWidth"):
            vobj.setPropertyStatus("LineWidth", "-LockDynamic")
            vobj.removeProperty("LineWidth")
        if hasattr(vobj, "LineColor"):
            vobj.setPropertyStatus("LineColor", "-LockDynamic")
            vobj.removeProperty("LineColor")
        _log("v1.1, " + obj.Name + ", removed view properties")

    def loads(self, state):
        # Before update_properties_0v21 the self.Type value was stored.
        # We use this to identify older objects that need to be updated.
        self.stored_type = state
        self.Type = "Text"


# Alias for compatibility with v0.18 and earlier
DraftText = Text

## @}
