# SPDX-License-Identifier: LGPL-2.1-or-later

# ***************************************************************************
# *   Copyright (c) 2015 Dan Falck <ddfalck@gmail.com>                      *
# *                                                                         *
# *   This program is free software; you can redistribute it and/or modify  *
# *   it under the terms of the GNU Lesser General Public License (LGPL)    *
# *   as published by the Free Software Foundation; either version 2 of     *
# *   the License, or (at your option) any later version.                   *
# *   for detail see the LICENCE text file.                                 *
# *                                                                         *
# *   This program is distributed in the hope that it will be useful,       *
# *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
# *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
# *   GNU Library General Public License for more details.                  *
# *                                                                         *
# *   You should have received a copy of the GNU Library General Public     *
# *   License along with this program; if not, write to the Free Software   *
# *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  *
# *   USA                                                                   *
# *                                                                         *
# ***************************************************************************

"""Used for CNC machine Stops for Path module. Create an Optional or Mandatory Stop."""

import FreeCAD
import FreeCADGui
import Path
from PySide import QtCore
from PySide.QtCore import QT_TRANSLATE_NOOP

translate = FreeCAD.Qt.translate


class Stop:
    def __init__(self, obj):
        obj.addProperty(
            "App::PropertyEnumeration",
            "Stop",
            "Path",
            QT_TRANSLATE_NOOP("App::Property", "Add an optional or mandatory stop to the program"),
        )
        obj.Stop = ["Optional", "Mandatory"]
        obj.Proxy = self
        mode = 2
        obj.setEditorMode("Placement", mode)

    def dumps(self):
        return None

    def loads(self, state):
        return None

    def onChanged(self, obj, prop):
        pass

    def execute(self, obj):
        if obj.Stop == "Optional":
            word = "M1"
        else:
            word = "M0"

        output = ""
        output = word + "\n"
        path = Path.Path(output)
        obj.Path = path


class _ViewProviderStop:
    def __init__(self, vobj):  # mandatory
        vobj.Proxy = self
        mode = 2
        vobj.setEditorMode("LineWidth", mode)
        vobj.setEditorMode("MarkerColor", mode)
        vobj.setEditorMode("NormalColor", mode)
        vobj.setEditorMode("DisplayMode", mode)
        vobj.setEditorMode("BoundingBox", mode)
        vobj.setEditorMode("Selectable", mode)
        vobj.setEditorMode("ShapeAppearance", mode)
        vobj.setEditorMode("Transparency", mode)
        vobj.setEditorMode("Visibility", mode)

    def dumps(self):  # mandatory
        return None

    def loads(self, state):  # mandatory
        return None

    def getIcon(self):  # optional
        return ":/icons/CAM_Stop.svg"

    def onChanged(self, vobj, prop):  # optional
        mode = 2
        vobj.setEditorMode("LineWidth", mode)
        vobj.setEditorMode("MarkerColor", mode)
        vobj.setEditorMode("NormalColor", mode)
        vobj.setEditorMode("DisplayMode", mode)
        vobj.setEditorMode("BoundingBox", mode)
        vobj.setEditorMode("Selectable", mode)
        vobj.setEditorMode("ShapeAppearance", mode)
        vobj.setEditorMode("Transparency", mode)
        vobj.setEditorMode("Visibility", mode)


class CommandPathStop:
    def GetResources(self):
        return {
            "Pixmap": "CAM_Stop",
            "MenuText": QT_TRANSLATE_NOOP("CAM_Stop", "Stop"),
            "ToolTip": QT_TRANSLATE_NOOP(
                "CAM_Stop", "Adds an optional or mandatory stop to the program"
            ),
        }

    def IsActive(self):
        if FreeCAD.ActiveDocument is not None:
            for o in FreeCAD.ActiveDocument.Objects:
                if o.Name[:3] == "Job":
                    return True
        return False

    def Activated(self):
        FreeCAD.ActiveDocument.openTransaction("Add Optional or Mandatory Stop to the program")
        FreeCADGui.addModule("Path.Op.Gui.Stop")
        snippet = """
import Path
import PathScripts
from PathScripts import PathUtils
prjexists = False
obj = FreeCAD.ActiveDocument.addObject("Path::FeaturePython","Stop")
Path.Op.Gui.Stop.Stop(obj)

Path.Op.Gui.Stop._ViewProviderStop(obj.ViewObject)
PathUtils.addToJob(obj)
"""
        FreeCADGui.doCommand(snippet)
        FreeCAD.ActiveDocument.commitTransaction()
        FreeCAD.ActiveDocument.recompute()


if FreeCAD.GuiUp:
    # register the FreeCAD command
    FreeCADGui.addCommand("CAM_Stop", CommandPathStop())


FreeCAD.Console.PrintLog("Loading PathStop… done\n")
