# SPDX-License-Identifier: LGPL-2.1-or-later

# ***************************************************************************
# *                                                                         *
# *   Copyright (c) 2019 Werner Mayer <wmayer[at]users.sourceforge.net>     *
# *                                                                         *
# *   This program is free software; you can redistribute it and/or modify  *
# *   it under the terms of the GNU Lesser General Public License (LGPL)    *
# *   as published by the Free Software Foundation; either version 2 of     *
# *   the License, or (at your option) any later version.                   *
# *   for detail see the LICENCE text file.                                 *
# *                                                                         *
# *   This program is distributed in the hope that it will be useful,       *
# *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
# *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
# *   GNU Library General Public License for more details.                  *
# *                                                                         *
# *   You should have received a copy of the GNU Library General Public     *
# *   License along with this program; if not, write to the Free Software   *
# *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  *
# *   USA                                                                   *
# *                                                                         *
# ***************************************************************************

__title__ = "BasicShapes.Shapes"
__author__ = "Werner Mayer"
__url__ = "https://www.freecad.org"
__doc__ = "Basic shapes"


import FreeCAD
import Part


def makeTube(outerRadius, innerRadius, height):
    outer_cylinder = Part.makeCylinder(outerRadius, height)
    shape = outer_cylinder
    if innerRadius > 0 and innerRadius < outerRadius:
        inner_cylinder = Part.makeCylinder(innerRadius, height)
        shape = outer_cylinder.cut(inner_cylinder)
    return shape


class TubeFeature:
    def __init__(self, obj):
        obj.Proxy = self
        obj.addProperty("App::PropertyLength","OuterRadius","Tube","Outer radius", locked=True).OuterRadius = 5.0
        obj.addProperty("App::PropertyLength","InnerRadius","Tube","Inner radius", locked=True).InnerRadius = 2.0
        obj.addProperty("App::PropertyLength","Height","Tube", "Height of the tube", locked=True).Height = 10.0
        obj.addExtension("Part::AttachExtensionPython")

    def execute(self, fp):
        if fp.InnerRadius >= fp.OuterRadius:
            raise ValueError("Inner radius must be smaller than outer radius")
        fp.Shape = makeTube(fp.OuterRadius, fp.InnerRadius, fp.Height)


def addTube(doc, name="Tube"):
    """addTube(document, [name]): adds a tube object"""

    obj = doc.addObject("Part::FeaturePython", name)
    TubeFeature(obj)
    if FreeCAD.GuiUp:
        from . import ViewProviderShapes
        ViewProviderShapes.ViewProviderTube(obj.ViewObject)
    return obj
