const tslib = require('tslib');
const core = require('@sentry/core');
const electron = require('electron');

const INTEGRATION_NAME = 'OnUncaughtException';
/** Capture unhandled errors. */
const onUncaughtExceptionIntegration = core.defineIntegration(() => {
    return {
        name: INTEGRATION_NAME,
        setupOnce() {
            // noop
        },
        setup(client) {
            const options = client.getOptions();
            global.process.on('uncaughtException', (error) => {
                const scope = core.getCurrentScope();
                scope.addEventProcessor((event) => tslib.__awaiter(this, void 0, void 0, function* () {
                    return (Object.assign(Object.assign({}, event), { level: 'fatal' }));
                }));
                client.captureException(error, {
                    originalException: error,
                    data: {
                        mechanism: {
                            handled: false,
                            type: 'generic',
                        },
                    },
                }, scope);
                client.flush(options.shutdownTimeout || 2000).then(() => {
                    if (options === null || options === void 0 ? void 0 : options.onFatalError) {
                        options.onFatalError(error);
                    }
                    else if (global.process.listenerCount('uncaughtException') <= 2) {
                        // In addition to this handler there is always one in Electron
                        // The dialog is only shown if there are no other handlers
                        // eslint-disable-next-line no-console
                        console.error('Uncaught Exception:');
                        // eslint-disable-next-line no-console
                        console.error(error);
                        const ref = error.stack;
                        const stack = ref !== undefined ? ref : `${error.name}: ${error.message}`;
                        const message = `Uncaught Exception:\n${stack}`;
                        electron.dialog.showErrorBox('A JavaScript error occurred in the main process', message);
                    }
                }, () => {
                    // ignore
                });
            });
        },
    };
});
/**
 * Capture unhandled errors.
 *
 * @deprecated Use `onUncaughtExceptionIntegration()` instead
 */
// eslint-disable-next-line deprecation/deprecation
const OnUncaughtException = core.convertIntegrationFnToClass(INTEGRATION_NAME, onUncaughtExceptionIntegration);

exports.OnUncaughtException = OnUncaughtException;
exports.onUncaughtExceptionIntegration = onUncaughtExceptionIntegration;
//# sourceMappingURL=onuncaughtexception.js.map
