const core = require('@sentry/core');
const electron = require('electron');
const os = require('os');
const electronNormalize = require('../electron-normalize.js');
const merge = require('../../common/merge.js');

const DEFAULT_OPTIONS = {
    cpu: true,
    screen: true,
    memory: true,
    language: true,
};
const INTEGRATION_NAME = 'AdditionalContext';
/**
 * Adds additional Electron context to events
 */
const additionalContextIntegration = core.defineIntegration((userOptions = {}) => {
    const _lazyDeviceContext = {};
    const options = Object.assign(Object.assign({}, DEFAULT_OPTIONS), userOptions);
    function _setPrimaryDisplayInfo() {
        const display = electron.screen.getPrimaryDisplay();
        const width = Math.floor(display.size.width * display.scaleFactor);
        const height = Math.floor(display.size.height * display.scaleFactor);
        _lazyDeviceContext.screen_density = display.scaleFactor;
        _lazyDeviceContext.screen_resolution = `${width}x${height}`;
    }
    return {
        name: INTEGRATION_NAME,
        setupOnce() {
            // noop
        },
        setup() {
            // Some metrics are only available after app ready so we lazily load them
            electronNormalize.whenAppReady.then(() => {
                const { language, screen } = options;
                if (language) {
                    _lazyDeviceContext.language = electron.app.getLocale();
                }
                if (screen) {
                    _setPrimaryDisplayInfo();
                    electron.screen.on('display-metrics-changed', () => {
                        _setPrimaryDisplayInfo();
                    });
                }
            }, () => {
                // ignore
            });
        },
        processEvent(event) {
            const device = _lazyDeviceContext;
            const { memory, cpu } = options;
            if (memory) {
                const { total, free } = process.getSystemMemoryInfo();
                device.memory_size = total * 1024;
                device.free_memory = free * 1024;
            }
            if (cpu) {
                const cpuInfo = os.cpus();
                if (cpuInfo === null || cpuInfo === void 0 ? void 0 : cpuInfo.length) {
                    const firstCpu = cpuInfo[0];
                    device.processor_count = cpuInfo.length;
                    device.cpu_description = firstCpu.model;
                    device.processor_frequency = firstCpu.speed;
                    if (electron.app.runningUnderARM64Translation) {
                        device.machine_arch = 'arm64';
                    }
                }
            }
            return merge.mergeEvents(event, { contexts: { device } });
        },
    };
});
/**
 * Adds additional Electron context to events
 *
 * @deprecated Use `additionalContextIntegration()z instead
 */
// eslint-disable-next-line deprecation/deprecation
const AdditionalContext = core.convertIntegrationFnToClass(INTEGRATION_NAME, additionalContextIntegration);

exports.AdditionalContext = AdditionalContext;
exports.additionalContextIntegration = additionalContextIntegration;
//# sourceMappingURL=additional-context.js.map
