# SPDX-FileCopyrightText: © 2008-2022 Oprea Dan
# SPDX-FileCopyrightText: © 2008-2022 Bart de Koning
# SPDX-FileCopyrightText: © 2008-2022 Richard Bailey
# SPDX-FileCopyrightText: © 2008-2022 Germar Reitze
# SPDX-FileCopyrightText: © 2025 Christian Buhtz <c.buhtz@posteo.jp>
#
# SPDX-License-Identifier: GPL-2.0-or-later
#
# This file is part of the program "Back In Time" which is released under GNU
# General Public License v2 (GPLv2). See LICENSES directory or go to
# <https://spdx.org/licenses/GPL-2.0-or-later.html>.
"""Separate application managing the systray icon"""
import sys
import os
import subprocess
import signal
import textwrap

# TODO Is this really required? If the client is not configured for X11
#      it may use Wayland or something else...
#      Or is this just required when run as root (where GUIs are not
#      configured normally)?
if not os.getenv('DISPLAY', ''):
    os.putenv('DISPLAY', ':0.0')

import qttools
qttools.register_backintime_path('common')
import logger
# Workaround until the codebase allows a single place to init all translations
import tools
tools.initiate_translation(None)
import snapshots
import progress
import logviewdialog
import encfstools
from PyQt6.QtCore import Qt, QSize, QTimer
from PyQt6.QtWidgets import QSystemTrayIcon, QMenu, QProgressBar, QWidget
from PyQt6.QtGui import QColor, QIcon, QRegion, QPixmap, QPainter


class QtSysTrayIcon:
    """Application instance for the Back In Time systray icon"""

    def __init__(self):

        self.snapshots = snapshots.Snapshots()
        self.config = self.snapshots.config
        self.decode = None

        if len(sys.argv) > 1:
            if not self.config.setCurrentProfile(sys.argv[1]):
                logger.warning(
                    f'Failed to change Profile_ID {sys.argv[1]}', self)

        self.qapp = qttools.create_qapplication(self.config.APP_NAME)
        translator = qttools.initiate_translator(self.config.language())
        self.qapp.installTranslator(translator)
        self.qapp.setQuitOnLastWindowClosed(False)

        import icon
        self.qapp.setWindowIcon(icon.BIT_LOGO)

        self.status_icon = self._create_status_icon()
        self.contextMenu = QMenu()

        self.menuProfileName = self.contextMenu.addAction(
            _('Profile: {profile_name}').format(
                profile_name=self.config.profileName())
        )
        self.contextMenu.addSeparator()

        self.menuStatusMessage = self.contextMenu.addAction(_('Done'))
        self.menuProgress = self.contextMenu.addAction('')
        self.menuProgress.setVisible(False)
        self.contextMenu.addSeparator()

        self.btnPause = self.contextMenu.addAction(
            icon.PAUSE, _('Pause backup process'))
        action = lambda: os.kill(self.snapshots.pid(), signal.SIGSTOP)
        self.btnPause.triggered.connect(action)

        self.btnResume = self.contextMenu.addAction(
            icon.RESUME, _('Resume backup process'))
        action = lambda: os.kill(self.snapshots.pid(), signal.SIGCONT)
        self.btnResume.triggered.connect(action)
        self.btnResume.setVisible(False)

        self.btnStop = self.contextMenu.addAction(
            icon.STOP, _('Stop backup process'))
        self.btnStop.triggered.connect(self.onBtnStop)
        self.contextMenu.addSeparator()

        self.btnDecode = self.contextMenu.addAction(
            icon.VIEW_SNAPSHOT_LOG, _('decode paths'))
        self.btnDecode.setCheckable(True)
        self.btnDecode.setVisible(self.config.snapshotsMode() == 'ssh_encfs')
        self.btnDecode.toggled.connect(self.onBtnDecode)

        self.openLog = self.contextMenu.addAction(
            icon.VIEW_LAST_LOG, _('View Last Log'))
        self.openLog.triggered.connect(self.onOpenLog)
        self.startBIT = self.contextMenu.addAction(
            icon.BIT_LOGO,
            _('Start {appname}').format(appname=self.config.APP_NAME)
        )
        self.startBIT.triggered.connect(self.onStartBIT)
        self.status_icon.setContextMenu(self.contextMenu)

        self.progressBar = self._create_progress_bar()

        self.first_error = self.config.notify()
        self.popup = None
        self.last_message = None

        self.timer = QTimer()
        self.timer.timeout.connect(self.updateInfo)

    def _recolor_pixmap(self, pixmap: QPixmap, color: QColor) -> QPixmap:
        result = QPixmap(pixmap.size())
        result.fill(Qt.GlobalColor.transparent)

        painter = QPainter(result)
        painter.setCompositionMode(
            QPainter.CompositionMode.CompositionMode_Source)
        painter.drawPixmap(0, 0, pixmap)
        painter.setCompositionMode(
            QPainter.CompositionMode.CompositionMode_SourceIn)
        painter.fillRect(result.rect(), color)
        painter.end()

        return result

    def _create_status_icon(self) -> QSystemTrayIcon:
        import icon
        symbolic_logo = QIcon.fromTheme(icon.BIT_LOGO_SYMBOLIC_NAME)

        # Logo color depending on dark/light mode
        dark_mode = qttools.in_dark_mode(self.qapp)
        color = QColor('white' if dark_mode else 'black')

        # Determine systray icon size depending on current graphic environment
        style = self.qapp.style()
        tray_icon_size = style.pixelMetric(style.PixelMetric.PM_SmallIconSize)

        pixmap = symbolic_logo.pixmap(QSize(tray_icon_size, tray_icon_size))
        pixmap = self._recolor_pixmap(pixmap, color)

        return QSystemTrayIcon(QIcon(pixmap))

    def _create_progress_bar(self) -> QProgressBar:
        bar = QProgressBar()

        bar.setMinimum(0)
        bar.setMaximum(100)
        bar.setValue(0)

        bar.setTextVisible(False)

        bar.resize(24, 6)

        import icon
        bar.render(
            icon.BIT_LOGO.pixmap(24),
            sourceRegion=QRegion(0, -14, 24, 6),
            flags=QWidget.RenderFlag.DrawChildren
        )

        return bar

    def prepareExit(self):
        self.timer.stop()

        if not self.status_icon is None:
            self.status_icon.hide()
            self.status_icon = None

        if not self.popup is None:
            self.popup.deleteLater()
            self.popup = None

        self.qapp.processEvents()

    def run(self):
        if not self.snapshots.busy():
            sys.exit()
        self.status_icon.show()
        self.timer.start(500)
        self.qapp.exec()
        self.prepareExit()

    def updateInfo(self):

        # Exit this systray icon "app" when the snapshots is taken
        if not self.snapshots.busy():
            self.prepareExit()
            self.qapp.exit(0)
            return

        paused = tools.processPaused(self.snapshots.pid())
        self.btnPause.setVisible(not paused)
        self.btnResume.setVisible(paused)

        message = self.snapshots.takeSnapshotMessage()
        if message is None and self.last_message is None:
            message = (0, _('Working…'))

        if not message is None:
            if message != self.last_message:
                self.last_message = message

                if self.decode:
                    message = (message[0], self.decode.log(message[1]))

                self.menuStatusMessage.setText(
                    '\n'.join(textwrap.wrap(message[1], width=80)))

                self.status_icon.setToolTip(message[1])

        pg = progress.ProgressFile(self.config)

        if pg.fileReadable():
            pg.load()
            # percent = pg.intValue('percent')
            ## disable progressbar in icon until BiT has it's own icon
            ## fixes bug #902
            # if percent != self.progressBar.value():
            #     self.progressBar.setValue(percent)
            #     self.progressBar.render(
            #         self.pixmap,
            #         sourceRegion=QRegion(0, -14, 24, 6),
            #         flags=QWidget.RenderFlags(QWidget.DrawChildren))
            #     self.status_icon.setIcon(QIcon(self.pixmap))

            self.menuProgress.setText(' | '.join(self.getMenuProgress(pg)))
            self.menuProgress.setVisible(True)

        else:
            # self.status_icon.setIcon(self.icon.BIT_LOGO)
            self.menuProgress.setVisible(False)

    def getMenuProgress(self, pg):
        """See common/app.py::MainWindow.getProgressBarFormat().

        The code is a near duplicate.
        """
        data = (
            ('sent', _('Sent:')),
            ('speed', _('Speed:')),
            ('eta',    _('ETA:'))
        )

        for key, txt in data:
            value = pg.strValue(key, '')

            if not value:
                continue

            yield txt + ' ' + value

    def onStartBIT(self):
        profileID = self.config.currentProfile()
        cmd = ['backintime-qt',]
        if not profileID == '1':
            cmd += ['--profile-id', profileID]
        _proc = subprocess.Popen(cmd)

    def onOpenLog(self):
        dlg = logviewdialog.LogViewDialog(self, systray = True)
        dlg.decode = self.decode
        dlg.cbDecode.setChecked(self.btnDecode.isChecked())
        dlg.exec()

    def onBtnDecode(self, checked):
        if checked:
            self.decode = encfstools.Decode(self.config)
            self.last_message = None
            self.updateInfo()
        else:
            self.decode = None

    def onBtnStop(self):
        os.kill(self.snapshots.pid(), signal.SIGKILL)
        self.btnStop.setEnabled(False)
        self.btnPause.setEnabled(False)
        self.btnResume.setEnabled(False)
        self.snapshots.setTakeSnapshotMessage(0, 'Backup terminated')


if __name__ == '__main__':

    logger.openlog()

    # HACK: Minimal arg parsing to enable debug-level logging
    if '--debug' in sys.argv:
        logger.DEBUG = True

    logger.debug('Sub process tries to show systray icon, '
                 f'called with args {str(sys.argv)}')

    QtSysTrayIcon().run()
